/*
 * s51dude - A Downloader/Uploader for 8051 device programmers
 * Copyright (C) 2008 Lucas Chiesa.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#ifndef _s51dude_h_
#define _s51dude_h_

#include "ihex.h"

#define _(String) gettext(String)
#define gettext_noop(String) (String)
#define N_(String) gettext_noop(String)

/*--------------------
* COMMAND LINE OPTIONS
* ------------------*/

#define     LC_DEV		    'p'
#define			LC_UPLOAD		'u'
#define			LC_ERASE		'e'
#define			LC_READ			'r'
#define			LC_HELP			'h'
#define			LC_VERSION		'V'
#define			LC_WLOCKBITS	'w'
#define			LC_RLOCKBITS	'l'

#define     LC_LONG_DEV  	        "part"
#define			LC_LONG_UPLOAD	        "upload"
#define			LC_LONG_ERASE	        "erase"
#define			LC_LONG_VERSION	        "version"
#define			LC_LONG_VERIFY	        "no-verify"
#define			LC_LONG_DEBUG	        "debug"
#define			LC_LONG_VERBOSE	        "verbose"
#define			LC_LONG_DRY_RUN	        "dry-run"
#define			LC_LONG_READ	        "read"
#define			LC_LONG_HELP	        "help"
#define			LC_LONG_WLOCKBITS       "write-lbits"
#define			LC_LONG_RLOCKBITS       "read-lbits"
#define			LC_LONG_LB1		        "LB1"
#define			LC_LONG_LB2		        "LB2"
#define			LC_LONG_LB3		        "LB3"
#define			LC_LONG_EEPROM	        "eeprom"
#define			LC_LONG_VERIFY_CHUNK	"verify-chunk"

#define			ALLOWED_SHORT_ARGUMENTS "r:u:p:wlhe"

#define			APPNAME			"s51dude"
#define			VERSION_STRING	"0.3.1"

// Maximo largo del nombre del micro
#define MAX_T_MICRO 10

/* Indica que protocolo va a utilizar el usbtiny, protocolo + micro */
#define	PROTOCOL_MSG  16

// Mensaje de configuracion para el usbtiny
#define	CONFIGURE_MSG 15


typedef enum dest {
	NAP,				// NotAPart
	AT89S8253,
	AT89S8252,
	AT89S52,
	AT89S53,
} dest;

typedef enum {
	NAA,				// NotAnAction
	UPLOAD,				// Erase target and then upload to the target.
	READ,				// Only read the target memory.
	ERASE,				// Only erase the target.
	W_LOCKBITS,			// Program the target lock bits.
	R_LOCKBITS,			// Read the target lock bits.
} actions;

/* protocolos validos*/
typedef enum
{
    PROTOCOL_SPI_AVR,
    PROTOCOL_SPI_S8253,
}protocols;

typedef struct micro_t {
	uint16_t     	mem_micro;
	uint16_t    	mem_eeprom;
	uint16_t    	max_chunk;
	uint16_t     	val;
	uint16_t     	ind_r_eeprom;
	uint16_t     	ind_w_eeprom;
	uint16_t     	ind_r_flash;
	uint16_t     	ind_w_flash;
	uint8_t     	enable_programing[4];
	uint8_t     	erase[4];
	uint16_t    	delay;
	uint16_t        protocol;
	char			text_micro[MAX_T_MICRO];
	void			(*write_lockbits) (int, int, int);
	void			(*read_lockbits) ();
} micro_t;

typedef struct global_options {
	micro_t		*target;
	actions		operation;
	char		*path;
	FILE*		file;
} global_options;

extern global_options	options;
extern int verify_flag;
extern int verify_chunk_flag;
extern int verbose_flag;
extern int dry_run_flag;
extern int debug_flag;
extern int	sck_period;
extern int lockbit1_flag;
extern int lockbit2_flag;
extern int lockbit3_flag;
extern int eeprom_flag;
extern unsigned char wlc[];

// Prototipos
void get_params (int argc, char *argv[]);

void load_dev (char *dest);

void load_action (actions modo, char* archivo);

void open_file (void);

uint16_t upload(void);

uint16_t read_hex_file(void);

void read_mem(unsigned int size_mem);

void print_params(void);

void check_open_file(void);

inline void print_error(char *msg);

int usbtiny_open (void);

void usbtiny_close (void);

void usbtiny_configure(int ind);

int usb_control (unsigned int requestid, unsigned int val, unsigned int index );

void usbtiny_powerup(void);

void usbtiny_powerdown(void);

void usbtiny_send_protocol(void);

int usbtiny_trasnparent_spi(unsigned char cmd[4], unsigned char res[4]);

void enable_programing();

int verify(uint16_t size);

int usb_out (unsigned int requestid, unsigned int val, unsigned int index,
		    unsigned char* buffer, int buflen, int bitclk );

int usb_in (unsigned int requestid, unsigned int val, unsigned int index,
		   unsigned char* buffer, int buflen, int bitclk );

void exit_nice (void);

void free_mem (void);

void print_help (void);

void print_version (void);

#endif
