<!--

@license Apache-2.0

Copyright (c) 2018 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# Zip

> Generate array tuples from input arrays.

<section class="intro">

</section>

<!-- /.intro -->

<section class="usage">

## Usage

```javascript
var zip = require( '@stdlib/utils/zip' );
```

#### zip( arr1, arr2,...\[, opts] )

Returns an `array` of `arrays`, where the ith element (tuple) in the returned `array` contains the ith elements of the input `arrays`.

```javascript
var zipped = zip( [ 1, 2 ], [ 'a', 'b' ] );
// returns [ [ 1, 'a' ], [ 2, 'b' ] ]
```

By default, the returned `array` length is the length of the shortest input `array`.

```javascript
var zipped = zip( [ 1, 2, 3 ], [ 'a', 'b' ] );
// returns [ [ 1, 'a' ], [ 2, 'b' ] ]
```

The function accepts an `options` object with optional properties:

-   **trunc**: `boolean` specifying whether the returned `array` should truncate `arrays` longer than the shortest `array`. Default: `true`.
-   **fill**: fill value used for unequal length `arrays`. Default: `null`.
-   **arrays**: `boolean` specifying whether, when provided a single input `array`, the function should interpret the argument as a list of `arrays` to be zipped (i.e., behavior similar to `zip.apply(null, arr)`). Default: `false`.

To turn off truncation,

```javascript
var opts = {
    'trunc': false
};

var zipped = zip( [ 1, 2, 3 ], [ 'a', 'b' ], opts );
// returns [ [ 1, 'a' ], [ 2, 'b' ], [ 3, null ] ]
```

A fill value is included in each tuple for each `array` which does not have an element at the ith index. By default, the fill value is `null`. To specify a different fill value, set the `fill` option.

```javascript
var opts = {
    'trunc': false,
    'fill': ''
};

var zipped = zip( [ 1, 2, 3 ], [ 'a', 'b' ], opts );
// returns [ [ 1, 'a' ], [ 2, 'b' ], [ 3, '' ] ]
```

If the function should interpret a single input `array` as an `array` of `arrays` to be zipped,

<!-- eslint-disable object-curly-newline -->

```javascript
var zipped;
var arr = [ [ 1, 2 ], [ 'a', 'b' ] ];

// Default behavior:
zipped = zip( arr );
// returns [ [ [ 1, 2 ] ], [ [ 'a', 'b' ] ] ]

// Array of arrays:
zipped = zip( arr, { 'arrays': true } );
// returns [ [ 1, 'a' ], [ 2, 'b' ] ]
```

</section>

<!-- /.usage -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var randu = require( '@stdlib/random/base/randu' );
var zip = require( '@stdlib/utils/zip' );

var zipped;
var len;
var y1;
var y2;
var y3;
var x;
var i;

// Simulate some data...
x = new Array( 100 );
len = x.length;
y1 = new Array( len );
y2 = new Array( len );
y3 = new Array( len );

for ( i = 0; i < len; i++ ) {
    x[ i ] = Date.now();
    y1[ i ] = randu() * 100;
    y2[ i ] = randu() * 100;
    y3[ i ] = randu();
}

zipped = zip( x, y1, y2, y3 );

console.log( zipped.join( '\n' ) );
```

</section>

<!-- /.examples -->

<section class="links">

</section>

<!-- /.links -->
