import json

from dojo.models import Finding


class GithubVulnerabilityParser(object):
    def get_scan_types(self):
        return ["Github Vulnerability Scan"]

    def get_label_for_scan_types(self, scan_type):
        return scan_type  # no custom label for now

    def get_description_for_scan_types(self, scan_type):
        return "Import vulnerabilities from Github API."

    def get_findings(self, filename, test):
        data = json.load(filename)
        if "data" not in data:
            raise ValueError("Invalid report file, no 'data' node found")

        vulnerabilityAlerts = self._search_vulnerability_alerts(data["data"])
        if not vulnerabilityAlerts:
            raise ValueError("Invalid report, no 'vulnerabilityAlerts' node found")

        dupes = dict()
        for alert in vulnerabilityAlerts["nodes"]:
            finding = Finding(
                title=alert["securityVulnerability"]["advisory"]["summary"],
                test=test,
                description=alert["securityVulnerability"]["advisory"]["description"],
                severity=self._convert_security(alert["securityVulnerability"].get("severity", "MODERATE")),
                static_finding=True,
                dynamic_finding=False,
                unique_id_from_tool=alert["id"],
            )

            if "vulnerableManifestPath" in alert:
                finding.file_path = alert["vulnerableManifestPath"]

            # if the package is present
            if "package" in alert["securityVulnerability"]:
                finding.component_name = alert["securityVulnerability"]["package"].get("name")

            if "references" in alert["securityVulnerability"]["advisory"]:
                finding.references = ""
                for ref in alert["securityVulnerability"]["advisory"]["references"]:
                    finding.references += ref["url"] + "\r\n"

            if "identifiers" in alert["securityVulnerability"]["advisory"]:
                unsaved_vulnerability_ids = list()
                for identifier in alert["securityVulnerability"]["advisory"]["identifiers"]:
                    if identifier.get("value"):
                        unsaved_vulnerability_ids.append(identifier.get("value"))
                if unsaved_vulnerability_ids:
                    finding.unsaved_vulnerability_ids = unsaved_vulnerability_ids

            if "cvss" in alert["securityVulnerability"]["advisory"]:
                if "score" in alert["securityVulnerability"]["advisory"]["cvss"]:
                    finding.cvssv3_score = alert["securityVulnerability"]["advisory"]["cvss"]["score"]
                if "vectorString" in alert["securityVulnerability"]["advisory"]["cvss"]:
                    finding.cvssv3 = alert["securityVulnerability"]["advisory"]["cvss"]["vectorString"]

            dupe_key = finding.unique_id_from_tool
            if dupe_key in dupes:
                find = dupes[dupe_key]
                find.nb_occurences += 1
            else:
                dupes[dupe_key] = finding

        return list(dupes.values())

    def _search_vulnerability_alerts(self, data):
        if isinstance(data, list):
            for item in data:
                result = self._search_vulnerability_alerts(item)
                if result:
                    return result
        elif isinstance(data, dict):
            for key in data:
                if key == "vulnerabilityAlerts":
                    return data[key]
                result = self._search_vulnerability_alerts(data[key])
                if result:
                    return result
        return None

    def _convert_security(self, val):
        if val.lower() == "moderate":
            return "Medium"
        else:
            return val.title()
