import OperationConfig from "../config/OperationConfig.json" assert {type: "json"};
import Utils, { isWorkerEnvironment } from "../Utils.mjs";
import Recipe from "../Recipe.mjs";
import Dish from "../Dish.mjs";
import {detectFileType, isType} from "./FileType.mjs";
import chiSquared from "chi-squared";

/**
 * A class for detecting encodings, file types and byte frequencies and
 * speculatively executing recipes.
 *
 * @author n1474335 [n1474335@gmail.com]
 * @copyright Crown Copyright 2018
 * @license Apache-2.0
 */
class Magic {

    /**
     * Magic constructor.
     *
     * @param {ArrayBuffer} buf
     * @param {Object[]} [opCriteria]
     * @param {Object} [prevOp]
     */
    constructor(buf, opCriteria=Magic._generateOpCriteria(), prevOp=null) {
        this.inputBuffer = new Uint8Array(buf);
        this.inputStr = Utils.arrayBufferToStr(buf);
        this.opCriteria = opCriteria;
        this.prevOp = prevOp;
    }

    /**
     * Finds operations that claim to be able to decode the input based on various criteria.
     *
     * @returns {Object[]}
     */
    findMatchingInputOps() {
        const matches = [],
            inputEntropy = this.calcEntropy();

        this.opCriteria.forEach(check => {
            // If the input doesn't lie in the required entropy range, move on
            if (check.entropyRange &&
                (inputEntropy < check.entropyRange[0] ||
                inputEntropy > check.entropyRange[1]))
                return;
            // If the input doesn't match the pattern, move on
            if (check.pattern &&
                !check.pattern.test(this.inputStr))
                return;

            matches.push(check);
        });

        return matches;
    }

    /**
     * Attempts to detect the language of the input by comparing its byte frequency
     * to that of several known languages.
     *
     * @param {boolean} [extLang=false] - Extensive language support (false = only check the most
     *                                    common Internet languages)
     * @returns {Object[]}
     */
    detectLanguage(extLang = false) {
        if (!this.inputBuffer.length) return [{
            lang: "Unknown",
            score: Math.MAX_VALUE,
            probability: Math.MIN_VALUE
        }];

        const inputFreq = this._freqDist();
        const langFreqs = extLang ? EXTENSIVE_LANG_FREQS : COMMON_LANG_FREQS;
        const chiSqrs = [];

        for (const lang in langFreqs) {
            const [score, prob] = Magic._chiSqr(inputFreq, langFreqs[lang]);
            chiSqrs.push({
                lang: lang,
                score: score,
                probability: prob
            });
        }

        // Sort results so that the most likely match is at the top
        chiSqrs.sort((a, b) => {
            return a.score - b.score;
        });

        return chiSqrs;
    }

    /**
     * Detects any matching file types for the input.
     *
     * @returns {Object} type
     * @returns {string} type.ext - File extension
     * @returns {string} type.mime - Mime type
     * @returns {string} [type.desc] - Description
     */
    detectFileType() {
        const fileType = detectFileType(this.inputBuffer);

        if (!fileType.length) return null;
        return {
            name: fileType[0].name,
            ext: fileType[0].extension,
            mime: fileType[0].mime,
            desc: fileType[0].description
        };
    }

    /**
     * Detects whether the input buffer is valid UTF8.
     *
     * @returns {boolean}
     */
    isUTF8() {
        const bytes = new Uint8Array(this.inputBuffer);
        let i = 0;
        while (i < bytes.length) {
            if (( // ASCII
                bytes[i] === 0x09 ||
                bytes[i] === 0x0A ||
                bytes[i] === 0x0D ||
                (0x20 <= bytes[i] && bytes[i] <= 0x7E)
            )) {
                i += 1;
                continue;
            }

            if (( // non-overlong 2-byte
                (0xC2 <= bytes[i] && bytes[i] <= 0xDF) &&
                (0x80 <= bytes[i+1] && bytes[i+1] <= 0xBF)
            )) {
                i += 2;
                continue;
            }

            if (( // excluding overlongs
                bytes[i] === 0xE0 &&
                (0xA0 <= bytes[i + 1] && bytes[i + 1] <= 0xBF) &&
                (0x80 <= bytes[i + 2] && bytes[i + 2] <= 0xBF)
            ) ||
            ( // straight 3-byte
                ((0xE1 <= bytes[i] && bytes[i] <= 0xEC) ||
                bytes[i] === 0xEE ||
                bytes[i] === 0xEF) &&
                (0x80 <= bytes[i + 1] && bytes[i+1] <= 0xBF) &&
                (0x80 <= bytes[i+2] && bytes[i+2] <= 0xBF)
            ) ||
            ( // excluding surrogates
                bytes[i] === 0xED &&
                (0x80 <= bytes[i+1] && bytes[i+1] <= 0x9F) &&
                (0x80 <= bytes[i+2] && bytes[i+2] <= 0xBF)
            )) {
                i += 3;
                continue;
            }

            if (( // planes 1-3
                bytes[i] === 0xF0 &&
                (0x90 <= bytes[i + 1] && bytes[i + 1] <= 0xBF) &&
                (0x80 <= bytes[i + 2] && bytes[i + 2] <= 0xBF) &&
                (0x80 <= bytes[i + 3] && bytes[i + 3] <= 0xBF)
            ) ||
            ( // planes 4-15
                (0xF1 <= bytes[i] && bytes[i] <= 0xF3) &&
                (0x80 <= bytes[i + 1] && bytes[i + 1] <= 0xBF) &&
                (0x80 <= bytes[i + 2] && bytes[i + 2] <= 0xBF) &&
                (0x80 <= bytes[i + 3] && bytes[i + 3] <= 0xBF)
            ) ||
            ( // plane 16
                bytes[i] === 0xF4 &&
                (0x80 <= bytes[i + 1] && bytes[i + 1] <= 0x8F) &&
                (0x80 <= bytes[i + 2] && bytes[i + 2] <= 0xBF) &&
                (0x80 <= bytes[i + 3] && bytes[i + 3] <= 0xBF)
            )) {
                i += 4;
                continue;
            }

            return false;
        }

        return true;
    }

    /**
     * Calculates the Shannon entropy of the input data.
     *
     * @returns {number}
     */
    calcEntropy(data=this.inputBuffer, standalone=false) {
        if (!standalone && this.inputEntropy) return this.inputEntropy;

        const prob = this._freqDist(data, standalone);
        let entropy = 0,
            p;

        for (let i = 0; i < prob.length; i++) {
            p = prob[i] / 100;
            if (p === 0) continue;
            entropy += p * Math.log(p) / Math.log(2);
        }

        if (!standalone) this.inputEntropy = -entropy;
        return -entropy;
    }

    /**
     * Generate various simple brute-forced encodings of the data (trucated to 100 bytes).
     *
     * @returns {Object[]} - The encoded data and an operation config to generate it.
     */
    async bruteForce() {
        const sample = new Uint8Array(this.inputBuffer).slice(0, 100);
        const results = [];

        // 1-byte XOR
        for (let i = 1; i < 256; i++) {
            results.push({
                data: sample.map(b => b ^ i).buffer,
                conf: {
                    op: "XOR",
                    args: [{"option": "Hex", "string": i.toString(16)}, "Standard", false]
                }
            });
        }

        // Bit rotate
        for (let i = 1; i < 8; i++) {
            results.push({
                data: sample.map(b => (b >> i) | ((b & (Math.pow(2, i) - 1)) << (8 - i))).buffer,
                conf: {
                    op: "Rotate right",
                    args: [i, false]
                }
            });
        }

        // Character encodings
        const encodings = OperationConfig["Encode text"].args[0].value;

        /**
         * Test character encodings and add them if they change the data.
         */
        const testEnc = async op => {
            for (let i = 0; i < encodings.length; i++) {
                const conf = {
                    op: op,
                    args: [encodings[i]]
                };

                try {
                    const data = await this._runRecipe([conf], sample.buffer);

                    // Only add to the results if it changed the data
                    if (!_buffersEqual(data, sample.buffer)) {
                        results.push({
                            data: data,
                            conf: conf
                        });
                    }
                } catch (err) {
                    continue;
                }
            }
        };

        await testEnc("Encode text");
        await testEnc("Decode text");

        return results;
    }

    /**
     * Checks whether the data passes output criteria for an operation check
     *
     * @param {ArrayBuffer} data
     * @param {Object} criteria
     * @returns {boolean}
     */
    outputCheckPasses(data, criteria) {
        if (criteria.pattern) {
            const dataStr = Utils.arrayBufferToStr(data),
                regex = new RegExp(criteria.pattern, criteria.flags);
            if (!regex.test(dataStr))
                return false;
        }
        if (criteria.entropyRange) {
            const dataEntropy = this.calcEntropy(data, true);
            if (dataEntropy < criteria.entropyRange[0] || dataEntropy > criteria.entropyRange[1])
                return false;
        }
        if (criteria.mime &&
            !isType(criteria.mime, data))
            return false;

        return true;
    }

    /**
     * Speculatively executes matching operations, recording metadata of each result.
     *
     * @param {number} [depth=0] - How many levels to try to execute
     * @param {boolean} [extLang=false] - Extensive language support (false = only check the most
     *     common Internet languages)
     * @param {boolean} [intensive=false] - Run brute-forcing on each branch (significantly affects
     *     performance)
     * @param {Object[]} [recipeConfig=[]] - The recipe configuration up to this point
     * @param {boolean} [useful=false] - Whether the current recipe should be scored highly
     * @param {string} [crib=null] - The regex crib provided by the user, for filtering the operation
     *     output
     * @returns {Object[]} - A sorted list of the recipes most likely to result in correct decoding
     */
    async speculativeExecution(
        depth=0,
        extLang=false,
        intensive=false,
        recipeConfig=[],
        useful=false,
        crib=null) {

        // If we have reached the recursion depth, return
        if (depth < 0) return [];

        // Find any operations that can be run on this data
        const matchingOps = this.findMatchingInputOps();
        let results = [];

        // Record the properties of the current data
        results.push({
            recipe: recipeConfig,
            data: this.inputStr.slice(0, 100),
            languageScores: this.detectLanguage(extLang),
            fileType: this.detectFileType(),
            isUTF8: this.isUTF8(),
            entropy: this.calcEntropy(),
            matchingOps: matchingOps,
            useful: useful,
            matchesCrib: crib && crib.test(this.inputStr)
        });
        const prevOp = recipeConfig[recipeConfig.length - 1];

        // Execute each of the matching operations, then recursively call the speculativeExecution()
        // method on the resulting data, recording the properties of each option.
        await Promise.all(matchingOps.map(async op => {
            const opConfig = {
                    op: op.op,
                    args: op.args
                },
                output = await this._runRecipe([opConfig]);

            // If the recipe returned an empty buffer, do not continue
            if (_buffersEqual(output, new ArrayBuffer())) {
                return;
            }

            // If the recipe is repeating and returning the same data, do not continue
            if (prevOp && op.op === prevOp.op && _buffersEqual(output, this.inputBuffer)) {
                return;
            }

            // If the output criteria for this op doesn't match the output, do not continue
            if (op.output && !this.outputCheckPasses(output, op.output))
                return;

            const magic = new Magic(output, this.opCriteria, OperationConfig[op.op]),
                speculativeResults = await magic.speculativeExecution(
                    depth-1, extLang, intensive, [...recipeConfig, opConfig], op.useful, crib);

            results = results.concat(speculativeResults);
        }));

        if (intensive) {
            // Run brute forcing of various types on the data and create a new branch for each option
            const bfEncodings = await this.bruteForce();

            await Promise.all(bfEncodings.map(async enc => {
                const magic = new Magic(enc.data, this.opCriteria, undefined),
                    bfResults = await magic.speculativeExecution(
                        depth-1, extLang, false, [...recipeConfig, enc.conf], false, crib);

                results = results.concat(bfResults);
            }));
        }

        // Prune branches that result in unhelpful outputs
        const prunedResults = results.filter(r =>
            (r.useful || r.data.length > 0) &&          // The operation resulted in ""
            (                                           // One of the following must be true
                r.languageScores[0].probability > 0 ||    // Some kind of language was found
                r.fileType ||                             // A file was found
                r.isUTF8 ||                               // UTF-8 was found
                r.matchingOps.length ||                   // A matching op was found
                r.matchesCrib                             // The crib matches
            )
        );

        // Return a sorted list of possible recipes along with their properties
        return prunedResults.sort((a, b) => {
            // Each option is sorted based on its most likely language (lower is better)
            let aScore = a.languageScores[0].score,
                bScore = b.languageScores[0].score;

            // If the result is valid UTF8, its score gets boosted (lower being better)
            if (a.isUTF8) aScore -= 100;
            if (b.isUTF8) bScore -= 100;

            // If a recipe results in a file being detected, it receives a relatively good score
            if (a.fileType && aScore > 500) aScore = 500;
            if (b.fileType && bScore > 500) bScore = 500;

            // If the option is marked useful, give it a good score
            if (a.useful && aScore > 100) aScore = 100;
            if (b.useful && bScore > 100) bScore = 100;

            // Shorter recipes are better, so we add the length of the recipe to the score
            aScore += a.recipe.length;
            bScore += b.recipe.length;

            // Lower entropy is "better", so we add the entropy to the score
            aScore += a.entropy;
            bScore += b.entropy;

            // A result with no recipe but matching ops suggests there are better options
            if ((!a.recipe.length && a.matchingOps.length) && b.recipe.length)
                return 1;
            if ((!b.recipe.length && b.matchingOps.length) && a.recipe.length)
                return -1;

            return aScore - bScore;
        });
    }

    /**
     * Runs the given recipe over the input buffer and returns the output.
     *
     * @param {Object[]} recipeConfig
     * @param {ArrayBuffer} [input=this.inputBuffer]
     * @returns {ArrayBuffer}
     */
    async _runRecipe(recipeConfig, input=this.inputBuffer) {
        input = input instanceof ArrayBuffer ? input : input.buffer;
        const dish = new Dish();
        dish.set(input, Dish.ARRAY_BUFFER);

        if (isWorkerEnvironment()) self.loadRequiredModules(recipeConfig);

        const recipe = new Recipe(recipeConfig);
        try {
            await recipe.execute(dish);
            // Return an empty buffer if the recipe did not run to completion
            if (recipe.lastRunOp === recipe.opList[recipe.opList.length - 1]) {
                return await dish.get(Dish.ARRAY_BUFFER);
            } else {
                return new ArrayBuffer();
            }
        } catch (err) {
            // If there are errors, return an empty buffer
            return new ArrayBuffer();
        }
    }

    /**
     * Calculates the number of times each byte appears in the input as a percentage
     *
     * @private
     * @param {ArrayBuffer} [data]
     * @param {boolean} [standalone]
     * @returns {number[]}
     */
    _freqDist(data=this.inputBuffer, standalone=false) {
        if (!standalone && this.freqDist) return this.freqDist;

        const len = data.length,
            counts = new Array(256).fill(0);
        let i = len;

        if (!len) {
            this.freqDist = counts;
            return this.freqDist;
        }

        while (i--) {
            counts[data[i]]++;
        }

        const result = counts.map(c => {
            return c / len * 100;
        });

        if (!standalone) this.freqDist = result;
        return result;
    }

    /**
     * Generates a list of all patterns that operations claim to be able to decode.
     *
     * @private
     * @returns {Object[]}
     */
    static _generateOpCriteria() {
        const opCriteria = [];

        for (const op in OperationConfig) {
            if (!("checks" in OperationConfig[op]))
                continue;

            OperationConfig[op].checks.forEach(check => {
                // Add to the opCriteria list.
                // Compile the regex here and cache the compiled version so we
                // don't have to keep calculating it.
                opCriteria.push({
                    op: op,
                    pattern: check.pattern ? new RegExp(check.pattern, check.flags) : null,
                    args: check.args,
                    useful: check.useful,
                    entropyRange: check.entropyRange,
                    output: check.output
                });
            });
        }

        return opCriteria;
    }

    /**
     * Calculates Pearson's Chi-Squared test for two frequency arrays.
     * https://en.wikipedia.org/wiki/Pearson%27s_chi-squared_test
     *
     * @private
     * @param {number[]} observed
     * @param {number[]} expected
     * @param {number} ddof - Delta degrees of freedom
     * @returns {number[]} - The score and the probability
     */
    static _chiSqr(observed, expected, ddof=0) {
        let tmp,
            score = 0;

        for (let i = 0; i < observed.length; i++) {
            tmp = observed[i] - expected[i];
            score += tmp * tmp / expected[i];
        }

        return [
            score,
            1 - chiSquared.cdf(score, observed.length - 1 - ddof)
        ];
    }

    /**
     * Translates ISO 639(-ish) codes to their full language names as used by Wikipedia
     * Accurate up to 2018-02
     * Taken from http://wikistats.wmflabs.org/display.php?t=wp
     *
     * @param {string} code - ISO 639 code
     * @returns {string} The full name of the language
     */
    static codeToLanguage(code) {
        return {
            "aa":  "Afar",
            "ab":  "Abkhazian",
            "ace": "Acehnese",
            "ady": "Adyghe",
            "af":  "Afrikaans",
            "ak":  "Akan",
            "als": "Alemannic",
            "am":  "Amharic",
            "an":  "Aragonese",
            "ang": "Anglo-Saxon",
            "ar":  "Arabic",
            "arc": "Aramaic",
            "arz": "Egyptian Arabic",
            "as":  "Assamese",
            "ast": "Asturian",
            "atj": "Atikamekw",
            "av":  "Avar",
            "ay":  "Aymara",
            "az":  "Azerbaijani",
            "azb": "South Azerbaijani",
            "ba":  "Bashkir",
            "bar": "Bavarian",
            "bat-smg": "Samogitian",
            "bcl": "Central_Bicolano",
            "be":  "Belarusian",
            "be-tarask": "Belarusian (Taraškievica)",
            "bg":  "Bulgarian",
            "bh":  "Bihari",
            "bi":  "Bislama",
            "bjn": "Banjar",
            "bm":  "Bambara",
            "bn":  "Bengali",
            "bo":  "Tibetan",
            "bpy": "Bishnupriya Manipuri",
            "br":  "Breton",
            "bs":  "Bosnian",
            "bug": "Buginese",
            "bxr": "Buryat (Russia)",
            "ca":  "Catalan",
            "cbk-zam": "Zamboanga Chavacano",
            "cdo": "Min Dong",
            "ce":  "Chechen",
            "ceb": "Cebuano",
            "ch":  "Chamorro",
            "cho": "Choctaw",
            "chr": "Cherokee",
            "chy": "Cheyenne",
            "ckb": "Sorani",
            "co":  "Corsican",
            "cr":  "Cree",
            "crh": "Crimean Tatar",
            "cs":  "Czech",
            "csb": "Kashubian",
            "cu":  "Old Church Slavonic",
            "cv":  "Chuvash",
            "cy":  "Welsh",
            "da":  "Danish",
            "de":  "German",
            "din": "Dinka",
            "diq": "Zazaki",
            "dsb": "Lower Sorbian",
            "dty": "Doteli",
            "dv":  "Divehi",
            "dz":  "Dzongkha",
            "ee":  "Ewe",
            "el":  "Greek",
            "eml": "Emilian-Romagnol",
            "en":  "English",
            "eo":  "Esperanto",
            "es":  "Spanish",
            "et":  "Estonian",
            "eu":  "Basque",
            "ext": "Extremaduran",
            "fa":  "Persian",
            "ff":  "Fula",
            "fi":  "Finnish",
            "fiu-vro": "Võro",
            "fj":  "Fijian",
            "fo":  "Faroese",
            "fr":  "French",
            "frp": "Franco-Provençal/Arpitan",
            "frr": "North Frisian",
            "fur": "Friulian",
            "fy":  "West Frisian",
            "ga":  "Irish",
            "gag": "Gagauz",
            "gan": "Gan",
            "gd":  "Scottish Gaelic",
            "gl":  "Galician",
            "glk": "Gilaki",
            "gn":  "Guarani",
            "gom": "Goan Konkani",
            "got": "Gothic",
            "gu":  "Gujarati",
            "gv":  "Manx",
            "ha":  "Hausa",
            "hak": "Hakka",
            "haw": "Hawaiian",
            "he":  "Hebrew",
            "hi":  "Hindi",
            "hif": "Fiji Hindi",
            "ho":  "Hiri Motu",
            "hr":  "Croatian",
            "hsb": "Upper Sorbian",
            "ht":  "Haitian",
            "hu":  "Hungarian",
            "hy":  "Armenian",
            "hz":  "Herero",
            "ia":  "Interlingua",
            "id":  "Indonesian",
            "ie":  "Interlingue",
            "ig":  "Igbo",
            "ii":  "Sichuan Yi",
            "ik":  "Inupiak",
            "ilo": "Ilokano",
            "io":  "Ido",
            "is":  "Icelandic",
            "it":  "Italian",
            "iu":  "Inuktitut",
            "ja":  "Japanese",
            "jam": "Jamaican",
            "jbo": "Lojban",
            "jv":  "Javanese",
            "ka":  "Georgian",
            "kaa": "Karakalpak",
            "kab": "Kabyle",
            "kbd": "Kabardian Circassian",
            "kbp": "Kabiye",
            "kg":  "Kongo",
            "ki":  "Kikuyu",
            "kj":  "Kuanyama",
            "kk":  "Kazakh",
            "kl":  "Greenlandic",
            "km":  "Khmer",
            "kn":  "Kannada",
            "ko":  "Korean",
            "koi": "Komi-Permyak",
            "kr":  "Kanuri",
            "krc": "Karachay-Balkar",
            "ks":  "Kashmiri",
            "ksh": "Ripuarian",
            "ku":  "Kurdish",
            "kv":  "Komi",
            "kw":  "Cornish",
            "ky":  "Kirghiz",
            "la":  "Latin",
            "lad": "Ladino",
            "lb":  "Luxembourgish",
            "lbe": "Lak",
            "lez": "Lezgian",
            "lg":  "Luganda",
            "li":  "Limburgish",
            "lij": "Ligurian",
            "lmo": "Lombard",
            "ln":  "Lingala",
            "lo":  "Lao",
            "lrc": "Northern Luri",
            "lt":  "Lithuanian",
            "ltg": "Latgalian",
            "lv":  "Latvian",
            "mai": "Maithili",
            "map-bms": "Banyumasan",
            "mdf": "Moksha",
            "mg":  "Malagasy",
            "mh":  "Marshallese",
            "mhr": "Meadow Mari",
            "mi":  "Maori",
            "min": "Minangkabau",
            "mk":  "Macedonian",
            "ml":  "Malayalam",
            "mn":  "Mongolian",
            "mo":  "Moldovan",
            "mr":  "Marathi",
            "mrj": "Hill Mari",
            "ms":  "Malay",
            "mt":  "Maltese",
            "mus": "Muscogee",
            "mwl": "Mirandese",
            "my":  "Burmese",
            "myv": "Erzya",
            "mzn": "Mazandarani",
            "na":  "Nauruan",
            "nah": "Nahuatl",
            "nap": "Neapolitan",
            "nds": "Low Saxon",
            "nds-nl": "Dutch Low Saxon",
            "ne":  "Nepali",
            "new": "Newar / Nepal Bhasa",
            "ng":  "Ndonga",
            "nl":  "Dutch",
            "nn":  "Norwegian (Nynorsk)",
            "no":  "Norwegian (Bokmål)",
            "nov": "Novial",
            "nrm": "Norman",
            "nso": "Northern Sotho",
            "nv":  "Navajo",
            "ny":  "Chichewa",
            "oc":  "Occitan",
            "olo": "Livvi-Karelian",
            "om":  "Oromo",
            "or":  "Oriya",
            "os":  "Ossetian",
            "pa":  "Punjabi",
            "pag": "Pangasinan",
            "pam": "Kapampangan",
            "pap": "Papiamentu",
            "pcd": "Picard",
            "pdc": "Pennsylvania German",
            "pfl": "Palatinate German",
            "pi":  "Pali",
            "pih": "Norfolk",
            "pl":  "Polish",
            "pms": "Piedmontese",
            "pnb": "Western Panjabi",
            "pnt": "Pontic",
            "ps":  "Pashto",
            "pt":  "Portuguese",
            "qu":  "Quechua",
            "rm":  "Romansh",
            "rmy": "Romani",
            "rn":  "Kirundi",
            "ro":  "Romanian",
            "roa-rup": "Aromanian",
            "roa-tara": "Tarantino",
            "ru":  "Russian",
            "rue": "Rusyn",
            "rw":  "Kinyarwanda",
            "sa":  "Sanskrit",
            "sah": "Sakha",
            "sc":  "Sardinian",
            "scn": "Sicilian",
            "sco": "Scots",
            "sd":  "Sindhi",
            "se":  "Northern Sami",
            "sg":  "Sango",
            "sh":  "Serbo-Croatian",
            "si":  "Sinhalese",
            "simple": "Simple English",
            "sk":  "Slovak",
            "sl":  "Slovenian",
            "sm":  "Samoan",
            "sn":  "Shona",
            "so":  "Somali",
            "sq":  "Albanian",
            "sr":  "Serbian",
            "srn": "Sranan",
            "ss":  "Swati",
            "st":  "Sesotho",
            "stq": "Saterland Frisian",
            "su":  "Sundanese",
            "sv":  "Swedish",
            "sw":  "Swahili",
            "szl": "Silesian",
            "ta":  "Tamil",
            "tcy": "Tulu",
            "te":  "Telugu",
            "tet": "Tetum",
            "tg":  "Tajik",
            "th":  "Thai",
            "ti":  "Tigrinya",
            "tk":  "Turkmen",
            "tl":  "Tagalog",
            "tn":  "Tswana",
            "to":  "Tongan",
            "tpi": "Tok Pisin",
            "tr":  "Turkish",
            "ts":  "Tsonga",
            "tt":  "Tatar",
            "tum": "Tumbuka",
            "tw":  "Twi",
            "ty":  "Tahitian",
            "tyv": "Tuvan",
            "udm": "Udmurt",
            "ug":  "Uyghur",
            "uk":  "Ukrainian",
            "ur":  "Urdu",
            "uz":  "Uzbek",
            "ve":  "Venda",
            "vec": "Venetian",
            "vep": "Vepsian",
            "vi":  "Vietnamese",
            "vls": "West Flemish",
            "vo":  "Volapük",
            "wa":  "Walloon",
            "war": "Waray-Waray",
            "wo":  "Wolof",
            "wuu": "Wu",
            "xal": "Kalmyk",
            "xh":  "Xhosa",
            "xmf": "Mingrelian",
            "yi":  "Yiddish",
            "yo":  "Yoruba",
            "za":  "Zhuang",
            "zea": "Zeelandic",
            "zh":  "Chinese",
            "zh-classical": "Classical Chinese",
            "zh-min-nan": "Min Nan",
            "zh-yue": "Cantonese",
            "zu":  "Zulu",
        }[code];
    }

}


/**
 * Byte frequencies of various languages generated from Wikipedia dumps taken in late 2017 and early 2018.
 * The Chi-Squared test cannot accept expected values of 0, so 0.0001 has been used to account for bytes
 * that do not normally appear in the language.
 *
 * The common languages are chosen based on https://w3techs.com/technologies/overview/content_language/all
 * as of early 2018.
 */
const COMMON_LANG_FREQS = {
    "en":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.755, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.843, 0.004, 0.375, 0.002, 0.008, 0.019, 0.008, 0.134, 0.137, 0.137, 0.001, 0.001, 0.972, 0.19, 0.857, 0.017, 0.334, 0.421, 0.246, 0.108, 0.104, 0.112, 0.103, 0.1, 0.127, 0.237, 0.04, 0.027, 0.004, 0.003, 0.004, 0.002, 0.0001, 0.338, 0.218, 0.326, 0.163, 0.121, 0.149, 0.133, 0.192, 0.232, 0.107, 0.082, 0.148, 0.248, 0.134, 0.103, 0.195, 0.012, 0.162, 0.368, 0.366, 0.077, 0.061, 0.127, 0.009, 0.03, 0.015, 0.004, 0.0001, 0.004, 0.0001, 0.003, 0.0001, 6.614, 1.039, 2.327, 2.934, 9.162, 1.606, 1.415, 3.503, 5.718, 0.081, 0.461, 3.153, 1.793, 5.723, 5.565, 1.415, 0.066, 5.036, 4.79, 6.284, 1.992, 0.759, 1.176, 0.139, 1.162, 0.102, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.06, 0.004, 0.003, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.031, 0.006, 0.001, 0.001, 0.001, 0.002, 0.014, 0.001, 0.001, 0.005, 0.005, 0.001, 0.002, 0.017, 0.007, 0.002, 0.003, 0.004, 0.002, 0.001, 0.002, 0.002, 0.012, 0.001, 0.002, 0.001, 0.004, 0.001, 0.001, 0.003, 0.003, 0.002, 0.005, 0.001, 0.001, 0.003, 0.001, 0.003, 0.001, 0.002, 0.001, 0.004, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.02, 0.047, 0.009, 0.009, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.061, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ru":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.512, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.274, 0.002, 0.063, 0.0001, 0.001, 0.009, 0.001, 0.001, 0.118, 0.118, 0.0001, 0.001, 0.595, 0.135, 0.534, 0.009, 0.18, 0.281, 0.15, 0.078, 0.076, 0.077, 0.068, 0.066, 0.083, 0.16, 0.036, 0.016, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.013, 0.009, 0.014, 0.009, 0.007, 0.006, 0.007, 0.006, 0.031, 0.002, 0.003, 0.007, 0.012, 0.007, 0.005, 0.01, 0.001, 0.008, 0.017, 0.011, 0.003, 0.009, 0.005, 0.012, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.003, 0.0001, 0.065, 0.009, 0.022, 0.021, 0.074, 0.01, 0.013, 0.019, 0.054, 0.001, 0.008, 0.036, 0.02, 0.047, 0.055, 0.013, 0.001, 0.052, 0.037, 0.041, 0.026, 0.007, 0.006, 0.003, 0.011, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.469, 2.363, 2.342, 0.986, 0.156, 0.422, 0.252, 0.495, 0.217, 0.136, 0.014, 0.778, 0.56, 0.097, 0.251, 0.811, 0.09, 0.184, 0.165, 0.06, 0.179, 0.021, 0.013, 0.029, 0.05, 0.005, 0.116, 0.045, 0.087, 0.073, 0.067, 0.124, 0.211, 0.16, 0.055, 0.033, 0.036, 0.024, 0.013, 0.02, 0.022, 0.002, 0.0001, 0.1, 0.0001, 0.025, 0.009, 0.011, 3.536, 0.619, 1.963, 0.833, 1.275, 3.452, 0.323, 0.635, 3.408, 0.642, 1.486, 1.967, 1.26, 2.857, 4.587, 1.082, 0.0001, 0.0001, 0.339, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.0001, 0.002, 0.001, 31.356, 12.318, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.131, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "de":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.726, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.303, 0.002, 0.278, 0.0001, 0.0001, 0.007, 0.003, 0.005, 0.149, 0.149, 0.015, 0.001, 0.636, 0.237, 0.922, 0.023, 0.305, 0.472, 0.225, 0.115, 0.11, 0.121, 0.108, 0.11, 0.145, 0.271, 0.049, 0.022, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.413, 0.383, 0.144, 0.412, 0.275, 0.258, 0.273, 0.218, 0.18, 0.167, 0.277, 0.201, 0.328, 0.179, 0.111, 0.254, 0.012, 0.219, 0.602, 0.209, 0.1, 0.185, 0.206, 0.005, 0.01, 0.112, 0.002, 0.0001, 0.002, 0.0001, 0.006, 0.0001, 4.417, 1.306, 1.99, 3.615, 12.382, 1.106, 2.0, 2.958, 6.179, 0.082, 0.866, 2.842, 1.869, 7.338, 2.27, 0.606, 0.016, 6.056, 4.424, 4.731, 3.002, 0.609, 0.918, 0.053, 0.169, 0.824, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.147, 0.002, 0.003, 0.001, 0.006, 0.001, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.03, 0.0001, 0.0001, 0.009, 0.001, 0.002, 0.009, 0.002, 0.001, 0.061, 0.0001, 0.048, 0.122, 0.057, 0.009, 0.001, 0.001, 0.4, 0.001, 0.002, 0.003, 0.003, 0.017, 0.001, 0.003, 0.001, 0.005, 0.0001, 0.001, 0.003, 0.002, 0.003, 0.005, 0.001, 0.001, 0.203, 0.0001, 0.002, 0.001, 0.002, 0.002, 0.438, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.056, 1.237, 0.01, 0.013, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.148, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ja":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.834, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.258, 0.007, 0.036, 0.001, 0.0001, 0.005, 0.002, 0.003, 0.033, 0.033, 0.0001, 0.002, 0.019, 0.052, 0.026, 0.009, 0.281, 0.407, 0.259, 0.126, 0.108, 0.109, 0.095, 0.092, 0.104, 0.184, 0.008, 0.001, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.048, 0.026, 0.039, 0.027, 0.028, 0.022, 0.018, 0.016, 0.03, 0.012, 0.014, 0.02, 0.03, 0.025, 0.025, 0.026, 0.002, 0.026, 0.045, 0.031, 0.013, 0.014, 0.014, 0.006, 0.006, 0.003, 0.001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.077, 0.012, 0.03, 0.026, 0.088, 0.012, 0.017, 0.025, 0.067, 0.002, 0.016, 0.041, 0.039, 0.059, 0.066, 0.016, 0.001, 0.06, 0.043, 0.051, 0.028, 0.009, 0.007, 0.004, 0.015, 0.004, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 2.555, 10.322, 5.875, 4.462, 0.784, 0.468, 0.442, 0.409, 1.173, 0.96, 0.657, 1.448, 1.442, 0.636, 0.341, 0.685, 0.495, 0.342, 0.651, 0.536, 0.435, 0.657, 0.51, 0.978, 0.31, 0.563, 0.439, 0.514, 0.668, 0.438, 0.29, 1.039, 0.423, 0.532, 0.407, 0.691, 0.677, 0.555, 0.911, 0.887, 1.086, 0.531, 0.836, 1.345, 0.438, 0.666, 1.528, 0.959, 0.535, 0.379, 0.302, 0.822, 0.614, 0.308, 0.253, 0.467, 0.807, 0.807, 0.777, 0.809, 1.292, 0.546, 0.524, 0.425, 0.0001, 0.0001, 0.002, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.015, 19.387, 1.167, 4.022, 2.518, 1.734, 1.339, 1.229, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.409, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "es":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.757, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.771, 0.003, 0.315, 0.001, 0.004, 0.019, 0.003, 0.014, 0.132, 0.133, 0.001, 0.001, 0.976, 0.078, 0.703, 0.014, 0.268, 0.331, 0.197, 0.095, 0.086, 0.095, 0.085, 0.084, 0.105, 0.183, 0.053, 0.027, 0.001, 0.002, 0.002, 0.002, 0.0001, 0.242, 0.129, 0.28, 0.129, 0.322, 0.105, 0.099, 0.077, 0.116, 0.074, 0.034, 0.209, 0.196, 0.086, 0.059, 0.187, 0.009, 0.118, 0.247, 0.128, 0.061, 0.072, 0.033, 0.023, 0.018, 0.013, 0.005, 0.0001, 0.005, 0.0001, 0.003, 0.0001, 8.9, 0.939, 3.234, 4.015, 9.642, 0.603, 0.891, 0.531, 5.007, 0.262, 0.107, 4.355, 1.915, 5.487, 6.224, 1.805, 0.423, 4.992, 5.086, 3.402, 2.878, 0.667, 0.044, 0.125, 0.673, 0.299, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.033, 0.009, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.003, 0.0001, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.006, 0.006, 0.001, 0.0001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.008, 0.008, 0.001, 0.001, 0.025, 0.274, 0.002, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.221, 0.003, 0.019, 0.001, 0.373, 0.001, 0.001, 0.005, 0.144, 0.01, 0.631, 0.002, 0.001, 0.002, 0.001, 0.002, 0.001, 0.102, 0.018, 0.006, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.079, 1.766, 0.003, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.008, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.032, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.894, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.162, 0.003, 0.276, 0.0001, 0.0001, 0.012, 0.002, 0.638, 0.153, 0.153, 0.001, 0.002, 0.96, 0.247, 0.715, 0.011, 0.225, 0.339, 0.18, 0.084, 0.081, 0.086, 0.081, 0.084, 0.106, 0.194, 0.063, 0.018, 0.003, 0.002, 0.003, 0.002, 0.0001, 0.208, 0.141, 0.255, 0.128, 0.144, 0.1, 0.095, 0.071, 0.154, 0.072, 0.042, 0.331, 0.173, 0.077, 0.056, 0.167, 0.013, 0.108, 0.214, 0.102, 0.049, 0.062, 0.035, 0.009, 0.014, 0.011, 0.003, 0.0001, 0.003, 0.0001, 0.004, 0.0001, 5.761, 0.627, 2.287, 3.136, 10.738, 0.723, 0.838, 0.669, 5.295, 0.172, 0.12, 4.204, 1.941, 5.522, 4.015, 2.005, 0.584, 5.043, 5.545, 5.13, 4.06, 0.906, 0.051, 0.295, 0.278, 0.085, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.136, 0.003, 0.004, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.034, 0.0001, 0.0001, 0.001, 0.004, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.019, 0.003, 0.0001, 0.0001, 0.001, 0.001, 0.112, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.367, 0.007, 0.034, 0.001, 0.003, 0.001, 0.003, 0.046, 0.303, 1.817, 0.082, 0.045, 0.001, 0.004, 0.029, 0.017, 0.004, 0.002, 0.002, 0.005, 0.038, 0.001, 0.003, 0.0001, 0.002, 0.02, 0.002, 0.054, 0.004, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.113, 2.813, 0.007, 0.026, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.003, 0.001, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.122, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pt":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.934, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.319, 0.004, 0.372, 0.001, 0.002, 0.012, 0.004, 0.016, 0.15, 0.15, 0.001, 0.002, 1.16, 0.21, 0.746, 0.022, 0.296, 0.361, 0.226, 0.106, 0.098, 0.105, 0.096, 0.094, 0.114, 0.207, 0.054, 0.022, 0.006, 0.004, 0.006, 0.002, 0.0001, 0.345, 0.166, 0.295, 0.143, 0.233, 0.136, 0.112, 0.077, 0.129, 0.093, 0.039, 0.119, 0.217, 0.135, 0.164, 0.222, 0.016, 0.14, 0.259, 0.142, 0.064, 0.078, 0.041, 0.021, 0.013, 0.012, 0.007, 0.0001, 0.007, 0.0001, 0.007, 0.0001, 9.026, 0.717, 2.572, 4.173, 8.551, 0.751, 0.906, 0.629, 5.107, 0.172, 0.12, 2.357, 3.189, 4.024, 7.683, 1.87, 0.445, 5.017, 5.188, 3.559, 2.852, 0.875, 0.055, 0.186, 0.122, 0.257, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.034, 0.01, 0.003, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.014, 0.001, 0.001, 0.001, 0.005, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.009, 0.006, 0.0001, 0.0001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.007, 0.007, 0.0001, 0.001, 0.079, 0.267, 0.045, 0.508, 0.002, 0.001, 0.001, 0.424, 0.003, 0.417, 0.113, 0.003, 0.001, 0.255, 0.001, 0.001, 0.005, 0.003, 0.015, 0.161, 0.032, 0.087, 0.003, 0.001, 0.002, 0.001, 0.095, 0.002, 0.005, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.067, 2.471, 0.004, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.007, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.033, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "it":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.828, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.918, 0.002, 0.385, 0.0001, 0.001, 0.007, 0.003, 0.383, 0.13, 0.131, 0.0001, 0.001, 0.948, 0.103, 0.657, 0.014, 0.252, 0.332, 0.195, 0.093, 0.089, 0.095, 0.088, 0.084, 0.098, 0.183, 0.061, 0.035, 0.006, 0.002, 0.006, 0.001, 0.0001, 0.215, 0.131, 0.235, 0.125, 0.08, 0.104, 0.125, 0.057, 0.24, 0.04, 0.038, 0.208, 0.179, 0.133, 0.054, 0.164, 0.025, 0.114, 0.256, 0.12, 0.052, 0.079, 0.038, 0.021, 0.012, 0.012, 0.002, 0.0001, 0.002, 0.0001, 0.005, 0.0001, 8.583, 0.65, 3.106, 3.081, 8.81, 0.801, 1.321, 0.694, 8.492, 0.02, 0.115, 5.238, 1.88, 5.659, 6.812, 1.981, 0.236, 4.962, 3.674, 5.112, 2.35, 1.107, 0.055, 0.027, 0.118, 0.709, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.022, 0.004, 0.002, 0.002, 0.001, 0.001, 0.001, 0.002, 0.013, 0.001, 0.0001, 0.0001, 0.001, 0.004, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.006, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.005, 0.0001, 0.001, 0.005, 0.005, 0.0001, 0.001, 0.153, 0.007, 0.001, 0.001, 0.003, 0.001, 0.001, 0.002, 0.174, 0.033, 0.004, 0.009, 0.036, 0.004, 0.001, 0.001, 0.006, 0.003, 0.097, 0.004, 0.001, 0.001, 0.003, 0.001, 0.002, 0.056, 0.009, 0.007, 0.004, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.043, 0.574, 0.01, 0.009, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.007, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.021, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "zh":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.074, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.273, 0.003, 0.045, 0.0001, 0.001, 0.012, 0.001, 0.004, 0.032, 0.032, 0.001, 0.003, 0.032, 0.068, 0.063, 0.017, 0.386, 0.478, 0.308, 0.149, 0.134, 0.146, 0.127, 0.121, 0.136, 0.231, 0.018, 0.009, 0.007, 0.006, 0.007, 0.0001, 0.0001, 0.045, 0.029, 0.041, 0.028, 0.022, 0.017, 0.02, 0.019, 0.025, 0.01, 0.013, 0.02, 0.033, 0.021, 0.018, 0.028, 0.002, 0.022, 0.045, 0.031, 0.01, 0.013, 0.012, 0.007, 0.005, 0.003, 0.004, 0.0001, 0.004, 0.0001, 0.009, 0.0001, 0.159, 0.026, 0.051, 0.047, 0.17, 0.025, 0.032, 0.057, 0.124, 0.003, 0.021, 0.089, 0.049, 0.12, 0.129, 0.028, 0.002, 0.124, 0.083, 0.1, 0.058, 0.016, 0.016, 0.008, 0.03, 0.012, 0.006, 0.004, 0.006, 0.001, 0.0001, 2.707, 1.09, 1.398, 0.705, 1.23, 1.04, 0.715, 0.952, 1.455, 1.297, 0.845, 1.19, 2.403, 1.193, 0.813, 1.077, 0.889, 0.565, 0.387, 0.47, 0.931, 0.663, 1.035, 0.837, 0.77, 0.772, 1.434, 1.023, 1.668, 0.609, 0.437, 0.793, 0.535, 0.706, 0.48, 0.538, 0.785, 0.909, 0.7, 0.697, 1.017, 0.519, 0.441, 0.567, 0.626, 1.082, 0.814, 1.054, 1.074, 0.811, 0.556, 0.684, 0.903, 0.43, 0.642, 0.78, 2.083, 1.147, 2.006, 1.331, 2.547, 1.015, 0.911, 0.807, 0.0001, 0.0001, 0.069, 0.007, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.126, 1.369, 3.539, 8.968, 5.44, 4.358, 3.141, 2.48, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 1.821, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fa":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.841, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.03, 0.001, 0.048, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.117, 0.117, 0.001, 0.001, 0.009, 0.038, 0.486, 0.012, 0.007, 0.009, 0.007, 0.005, 0.003, 0.004, 0.003, 0.003, 0.003, 0.004, 0.048, 0.001, 0.001, 0.003, 0.001, 0.001, 0.0001, 0.011, 0.006, 0.011, 0.006, 0.005, 0.005, 0.004, 0.005, 0.007, 0.002, 0.002, 0.005, 0.008, 0.005, 0.005, 0.008, 0.001, 0.005, 0.011, 0.008, 0.002, 0.003, 0.004, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.007, 0.0001, 0.058, 0.008, 0.02, 0.02, 0.06, 0.011, 0.012, 0.017, 0.051, 0.001, 0.009, 0.031, 0.018, 0.042, 0.047, 0.015, 0.001, 0.043, 0.03, 0.037, 0.022, 0.005, 0.008, 0.003, 0.009, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.678, 0.557, 0.438, 0.001, 1.227, 2.118, 3.004, 2.445, 2.539, 0.0001, 0.003, 0.021, 5.067, 0.002, 0.007, 0.006, 0.015, 0.005, 0.002, 0.008, 0.07, 0.0001, 0.0001, 0.0001, 0.053, 0.001, 0.0001, 0.018, 0.0001, 0.001, 0.0001, 0.002, 0.002, 0.006, 0.337, 0.015, 0.006, 0.001, 0.059, 6.029, 1.704, 1.216, 2.096, 0.113, 0.433, 0.309, 0.439, 3.398, 0.192, 3.798, 0.977, 1.716, 1.137, 0.259, 0.129, 0.264, 0.12, 0.588, 0.085, 0.033, 0.001, 0.0001, 0.327, 0.0001, 0.0001, 0.0001, 0.068, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 23.012, 12.666, 1.946, 5.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.676, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pl":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.97, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.695, 0.002, 0.242, 0.0001, 0.0001, 0.007, 0.002, 0.011, 0.194, 0.194, 0.0001, 0.001, 0.805, 0.129, 1.016, 0.02, 0.347, 0.542, 0.289, 0.14, 0.138, 0.144, 0.123, 0.13, 0.153, 0.343, 0.068, 0.014, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.17, 0.165, 0.143, 0.124, 0.066, 0.081, 0.113, 0.075, 0.141, 0.107, 0.18, 0.108, 0.192, 0.142, 0.119, 0.322, 0.004, 0.139, 0.268, 0.117, 0.058, 0.041, 0.322, 0.032, 0.008, 0.109, 0.001, 0.0001, 0.001, 0.0001, 0.006, 0.0001, 6.697, 0.859, 2.856, 2.291, 5.604, 0.259, 1.117, 0.918, 6.017, 1.562, 2.537, 1.759, 1.903, 4.231, 5.86, 1.841, 0.006, 3.854, 3.145, 2.863, 1.965, 0.061, 3.408, 0.016, 2.669, 3.631, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.208, 0.018, 1.343, 0.004, 0.168, 0.653, 0.002, 0.145, 0.003, 0.001, 0.001, 0.001, 0.002, 0.004, 0.001, 0.002, 0.002, 0.001, 0.003, 0.126, 0.002, 0.001, 0.002, 0.002, 0.001, 0.65, 0.023, 0.378, 0.002, 0.035, 0.035, 0.002, 0.018, 0.011, 0.001, 0.002, 0.005, 0.001, 0.001, 0.002, 0.003, 0.012, 0.001, 0.002, 0.001, 0.005, 0.001, 0.001, 0.01, 0.004, 0.011, 0.641, 0.003, 0.006, 0.005, 0.001, 0.008, 0.004, 0.056, 0.014, 0.433, 0.007, 0.008, 0.002, 0.0001, 0.0001, 0.025, 0.694, 1.442, 2.413, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.006, 0.003, 0.06, 0.02, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.003, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.205, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.91, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.447, 0.013, 0.297, 0.0001, 0.001, 0.013, 0.003, 0.465, 0.123, 0.123, 0.001, 0.002, 0.653, 0.111, 0.957, 0.015, 0.312, 0.387, 0.238, 0.107, 0.101, 0.108, 0.097, 0.095, 0.109, 0.217, 0.04, 0.028, 0.007, 0.019, 0.007, 0.002, 0.0001, 0.336, 0.309, 0.117, 0.167, 0.132, 0.105, 0.13, 0.135, 0.063, 0.042, 0.261, 0.085, 0.236, 0.083, 0.095, 0.131, 0.004, 0.092, 0.247, 0.219, 0.038, 0.052, 0.037, 0.008, 0.095, 0.019, 0.007, 0.0001, 0.007, 0.0001, 0.005, 0.001, 8.533, 1.3, 0.65, 3.067, 6.656, 0.419, 0.804, 0.718, 6.178, 0.059, 2.986, 5.127, 2.286, 5.537, 2.04, 0.623, 0.006, 5.247, 2.411, 2.743, 2.225, 0.903, 0.049, 0.018, 2.076, 0.792, 0.0001, 0.018, 0.0001, 0.0001, 0.0001, 0.096, 0.004, 0.004, 0.004, 0.002, 0.002, 0.002, 0.041, 0.002, 0.001, 0.001, 0.001, 0.002, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.007, 0.002, 0.001, 0.031, 0.001, 0.003, 0.065, 0.001, 0.001, 0.033, 0.009, 0.047, 1.71, 0.04, 0.005, 0.027, 0.002, 0.003, 0.001, 0.001, 0.647, 0.002, 0.008, 0.002, 0.003, 0.001, 0.004, 0.019, 0.002, 0.132, 3.435, 0.005, 0.004, 0.003, 0.003, 0.525, 0.001, 0.004, 0.002, 0.003, 0.007, 1.206, 0.003, 0.003, 0.002, 0.0001, 0.0001, 0.046, 2.539, 4.197, 1.125, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.007, 0.003, 0.023, 0.009, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.01, 0.007, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.094, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nl":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.158, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.747, 0.002, 0.267, 0.0001, 0.001, 0.008, 0.01, 0.052, 0.196, 0.196, 0.0001, 0.001, 0.504, 0.205, 0.944, 0.013, 0.311, 0.428, 0.229, 0.104, 0.101, 0.109, 0.102, 0.102, 0.137, 0.252, 0.048, 0.012, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.205, 0.192, 0.181, 0.371, 0.131, 0.088, 0.11, 0.236, 0.167, 0.069, 0.091, 0.119, 0.172, 0.137, 0.117, 0.141, 0.005, 0.112, 0.229, 0.137, 0.034, 0.123, 0.084, 0.006, 0.011, 0.064, 0.001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 6.042, 1.063, 1.294, 4.124, 13.689, 0.579, 2.105, 1.822, 5.542, 0.948, 1.42, 3.124, 1.72, 7.129, 4.759, 1.349, 0.015, 5.115, 3.623, 4.903, 1.642, 1.84, 1.06, 0.063, 0.226, 0.656, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.023, 0.003, 0.004, 0.003, 0.002, 0.001, 0.001, 0.002, 0.001, 0.002, 0.0001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.008, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.007, 0.001, 0.001, 0.003, 0.003, 0.001, 0.002, 0.008, 0.009, 0.003, 0.002, 0.005, 0.002, 0.001, 0.003, 0.009, 0.038, 0.001, 0.051, 0.001, 0.005, 0.001, 0.011, 0.004, 0.003, 0.013, 0.008, 0.002, 0.002, 0.008, 0.001, 0.004, 0.001, 0.003, 0.002, 0.01, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.02, 0.166, 0.007, 0.01, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.016, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.022, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ko":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.893, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.919, 0.003, 0.069, 0.0001, 0.0001, 0.007, 0.002, 0.048, 0.269, 0.269, 0.0001, 0.002, 0.501, 0.04, 0.699, 0.01, 0.29, 0.417, 0.259, 0.125, 0.109, 0.112, 0.1, 0.094, 0.109, 0.192, 0.015, 0.002, 0.006, 0.002, 0.006, 0.003, 0.0001, 0.038, 0.026, 0.038, 0.022, 0.02, 0.024, 0.015, 0.013, 0.023, 0.008, 0.015, 0.017, 0.027, 0.016, 0.016, 0.023, 0.002, 0.017, 0.041, 0.027, 0.011, 0.013, 0.01, 0.005, 0.004, 0.002, 0.006, 0.0001, 0.006, 0.0001, 0.012, 0.0001, 0.108, 0.014, 0.037, 0.031, 0.116, 0.024, 0.022, 0.032, 0.084, 0.002, 0.021, 0.064, 0.06, 0.077, 0.092, 0.02, 0.001, 0.086, 0.056, 0.066, 0.046, 0.011, 0.008, 0.004, 0.019, 0.004, 0.0001, 0.002, 0.0001, 0.025, 0.0001, 2.21, 0.565, 0.766, 0.471, 3.043, 0.671, 0.334, 0.049, 1.404, 0.218, 1.17, 1.657, 1.23, 0.278, 0.091, 0.557, 1.645, 0.451, 0.058, 0.386, 1.38, 2.193, 0.506, 1.29, 2.708, 0.68, 0.385, 0.399, 2.758, 3.352, 0.954, 0.141, 1.848, 0.829, 0.071, 0.249, 1.741, 0.637, 0.43, 0.888, 0.537, 0.506, 0.243, 0.027, 1.4, 0.355, 0.026, 0.179, 2.38, 0.404, 0.739, 1.021, 2.205, 0.729, 0.454, 0.308, 1.635, 0.561, 0.035, 0.084, 1.612, 0.309, 0.024, 0.047, 0.0001, 0.0001, 0.034, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.039, 0.089, 0.025, 0.107, 0.071, 0.044, 0.037, 0.043, 3.199, 8.716, 12.558, 3.298, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cs":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.804, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.066, 0.002, 0.232, 0.0001, 0.0001, 0.008, 0.002, 0.009, 0.188, 0.188, 0.007, 0.002, 0.814, 0.094, 1.008, 0.025, 0.299, 0.437, 0.233, 0.115, 0.111, 0.119, 0.106, 0.102, 0.129, 0.233, 0.051, 0.011, 0.002, 0.002, 0.002, 0.002, 0.0001, 0.143, 0.145, 0.103, 0.117, 0.06, 0.072, 0.055, 0.092, 0.08, 0.13, 0.142, 0.093, 0.169, 0.137, 0.088, 0.246, 0.003, 0.104, 0.236, 0.127, 0.039, 0.213, 0.033, 0.007, 0.007, 0.069, 0.002, 0.0001, 0.002, 0.0001, 0.005, 0.0001, 5.018, 1.137, 1.8, 2.299, 5.465, 0.243, 0.288, 1.623, 3.2, 1.177, 2.624, 3.218, 2.048, 4.447, 5.813, 1.952, 0.006, 3.062, 3.218, 3.502, 2.227, 3.008, 0.043, 0.058, 1.313, 1.405, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.104, 0.003, 0.004, 0.003, 0.001, 0.001, 0.001, 0.003, 0.041, 0.001, 0.001, 0.001, 0.049, 0.57, 0.001, 0.012, 0.001, 0.001, 0.002, 0.048, 0.002, 0.001, 0.001, 0.002, 0.011, 0.748, 0.01, 0.981, 0.025, 0.001, 0.025, 0.002, 0.191, 1.9, 0.003, 0.001, 0.005, 0.024, 0.002, 0.002, 0.002, 0.87, 0.001, 0.001, 0.001, 1.984, 0.001, 0.336, 0.006, 0.002, 0.004, 0.031, 0.002, 0.003, 0.006, 0.001, 0.003, 0.001, 0.094, 0.002, 0.007, 0.671, 0.58, 0.001, 0.0001, 0.0001, 0.173, 5.104, 1.615, 2.233, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.021, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.009, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.103, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ar":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.65, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.194, 0.002, 0.102, 0.0001, 0.0001, 0.007, 0.001, 0.002, 0.109, 0.108, 0.002, 0.001, 0.03, 0.046, 0.42, 0.018, 0.182, 0.202, 0.135, 0.063, 0.065, 0.061, 0.055, 0.053, 0.062, 0.113, 0.054, 0.001, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.01, 0.006, 0.009, 0.007, 0.005, 0.004, 0.004, 0.004, 0.005, 0.002, 0.002, 0.005, 0.007, 0.005, 0.004, 0.007, 0.001, 0.005, 0.009, 0.006, 0.002, 0.002, 0.002, 0.001, 0.001, 0.001, 0.007, 0.001, 0.007, 0.0001, 0.004, 0.0001, 0.052, 0.008, 0.019, 0.018, 0.055, 0.008, 0.011, 0.016, 0.045, 0.001, 0.006, 0.028, 0.016, 0.037, 0.04, 0.012, 0.001, 0.038, 0.03, 0.035, 0.02, 0.006, 0.006, 0.002, 0.009, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.055, 1.131, 0.874, 0.939, 4.804, 2.787, 2.235, 1.018, 2.407, 0.349, 3.542, 0.092, 0.4, 0.007, 0.051, 0.053, 0.022, 0.061, 0.01, 0.008, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.008, 0.001, 0.001, 0.0001, 0.002, 0.013, 0.133, 0.049, 0.782, 0.037, 0.335, 0.157, 6.208, 1.599, 1.486, 1.889, 0.276, 0.607, 0.762, 0.341, 1.38, 0.239, 2.041, 0.293, 1.149, 0.411, 0.383, 0.246, 0.406, 0.094, 1.401, 0.223, 0.006, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.027, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 23.298, 20.414, 0.003, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.019, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "vi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.205, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.546, 0.002, 0.241, 0.0001, 0.001, 0.015, 0.013, 0.009, 0.13, 0.13, 0.0001, 0.002, 0.714, 0.089, 0.813, 0.02, 0.259, 0.361, 0.203, 0.104, 0.097, 0.104, 0.089, 0.089, 0.116, 0.194, 0.047, 0.017, 0.002, 0.002, 0.002, 0.002, 0.0001, 0.148, 0.175, 0.293, 0.111, 0.056, 0.04, 0.092, 0.206, 0.057, 0.03, 0.119, 0.232, 0.178, 0.247, 0.036, 0.156, 0.056, 0.062, 0.184, 0.397, 0.022, 0.114, 0.033, 0.033, 0.019, 0.009, 0.005, 0.0001, 0.005, 0.0001, 0.003, 0.0001, 2.683, 0.66, 3.149, 0.627, 1.148, 0.076, 2.542, 4.362, 3.528, 0.019, 0.59, 1.486, 1.611, 5.924, 2.001, 0.761, 0.201, 1.559, 1.014, 3.555, 1.77, 0.861, 0.05, 0.173, 0.826, 0.047, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.021, 0.214, 0.011, 0.478, 0.002, 0.039, 0.001, 0.324, 0.002, 0.072, 0.001, 0.198, 0.002, 0.32, 0.002, 0.048, 0.141, 1.485, 0.001, 0.116, 0.015, 0.106, 0.001, 0.025, 0.002, 0.579, 0.004, 0.289, 0.004, 0.257, 0.005, 0.174, 1.516, 1.221, 0.326, 0.818, 0.013, 0.337, 0.005, 0.51, 0.014, 0.324, 0.408, 0.115, 0.147, 0.492, 0.002, 0.218, 0.82, 0.26, 0.102, 0.383, 0.379, 0.016, 0.006, 0.094, 0.005, 0.132, 2.233, 4.628, 0.009, 0.062, 0.003, 0.385, 0.0001, 0.0001, 0.047, 4.542, 1.653, 0.065, 0.997, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.011, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 6.74, 0.019, 0.004, 0.002, 0.009, 0.006, 0.004, 0.003, 0.003, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "el":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.389, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.245, 0.003, 0.167, 0.001, 0.0001, 0.005, 0.002, 0.015, 0.1, 0.101, 0.0001, 0.001, 0.487, 0.058, 0.449, 0.01, 0.151, 0.215, 0.114, 0.058, 0.055, 0.058, 0.052, 0.051, 0.065, 0.119, 0.032, 0.001, 0.003, 0.003, 0.003, 0.0001, 0.0001, 0.021, 0.016, 0.024, 0.014, 0.012, 0.012, 0.011, 0.013, 0.012, 0.005, 0.006, 0.013, 0.018, 0.01, 0.009, 0.015, 0.001, 0.013, 0.025, 0.017, 0.005, 0.006, 0.008, 0.002, 0.002, 0.001, 0.005, 0.0001, 0.005, 0.0001, 0.002, 0.0001, 0.125, 0.018, 0.039, 0.039, 0.142, 0.017, 0.026, 0.036, 0.105, 0.002, 0.017, 0.072, 0.036, 0.093, 0.102, 0.022, 0.002, 0.099, 0.07, 0.077, 0.046, 0.014, 0.01, 0.005, 0.02, 0.005, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 1.502, 1.948, 1.522, 1.805, 3.613, 1.458, 0.354, 0.481, 0.073, 0.584, 0.024, 0.002, 0.912, 0.435, 0.305, 0.001, 0.006, 0.156, 0.057, 0.068, 0.049, 0.097, 0.01, 0.064, 0.017, 0.048, 0.112, 0.037, 0.115, 0.048, 0.003, 0.099, 0.122, 0.029, 0.001, 0.129, 0.119, 0.011, 0.03, 0.034, 0.002, 0.008, 0.0001, 0.022, 0.85, 0.749, 0.601, 1.063, 0.004, 3.95, 0.27, 0.716, 0.649, 2.656, 0.14, 1.63, 0.422, 2.831, 1.733, 1.214, 1.337, 2.636, 0.149, 3.615, 0.0001, 0.0001, 0.06, 0.007, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 28.675, 14.922, 0.013, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.013, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.282, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.667, 0.001, 0.345, 0.0001, 0.0001, 0.007, 0.002, 0.013, 0.083, 0.083, 0.0001, 0.0001, 0.902, 0.146, 1.182, 0.007, 0.152, 0.25, 0.108, 0.06, 0.06, 0.065, 0.065, 0.066, 0.089, 0.153, 0.044, 0.004, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.178, 0.164, 0.421, 0.354, 0.095, 0.078, 0.149, 0.127, 0.181, 0.06, 0.161, 0.209, 0.174, 0.099, 0.072, 0.149, 0.019, 0.12, 0.249, 0.206, 0.034, 0.058, 0.04, 0.006, 0.012, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 6.63, 0.945, 0.963, 3.448, 8.696, 0.922, 2.03, 1.373, 4.448, 0.429, 1.949, 3.417, 3.024, 6.448, 3.193, 1.076, 0.019, 6.923, 3.891, 5.562, 1.877, 1.653, 0.074, 0.114, 0.424, 0.075, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.022, 0.039, 0.002, 0.003, 0.007, 0.074, 0.004, 0.007, 0.005, 0.002, 0.002, 0.0001, 0.003, 0.008, 0.002, 0.004, 0.001, 0.002, 0.0001, 0.011, 0.001, 0.001, 0.012, 0.001, 0.005, 0.002, 0.001, 0.001, 0.001, 0.004, 0.001, 0.003, 0.21, 0.017, 0.005, 0.004, 1.574, 0.853, 0.002, 0.007, 0.008, 0.038, 0.004, 0.047, 0.001, 0.014, 0.002, 0.009, 0.187, 0.01, 0.004, 0.012, 0.004, 0.002, 0.808, 0.001, 0.008, 0.002, 0.004, 0.002, 0.006, 0.002, 0.003, 0.001, 0.0001, 0.0001, 0.393, 3.436, 0.069, 0.044, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.014, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.021, 0.021, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.019, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hu":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.827, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.714, 0.004, 0.265, 0.0001, 0.0001, 0.007, 0.001, 0.007, 0.159, 0.159, 0.001, 0.002, 1.016, 0.461, 0.937, 0.013, 0.261, 0.429, 0.206, 0.109, 0.106, 0.113, 0.103, 0.105, 0.137, 0.238, 0.073, 0.019, 0.004, 0.004, 0.004, 0.002, 0.0001, 0.469, 0.135, 0.097, 0.073, 0.142, 0.093, 0.075, 0.087, 0.095, 0.062, 0.133, 0.086, 0.175, 0.085, 0.042, 0.096, 0.003, 0.071, 0.186, 0.107, 0.027, 0.069, 0.028, 0.009, 0.008, 0.025, 0.002, 0.0001, 0.002, 0.0001, 0.004, 0.0001, 6.316, 1.591, 0.619, 1.364, 7.125, 0.648, 2.159, 0.946, 3.15, 0.796, 3.265, 4.526, 2.054, 3.978, 3.047, 0.846, 0.006, 3.327, 4.35, 5.787, 0.902, 1.395, 0.037, 0.035, 1.463, 2.94, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.129, 0.02, 0.003, 0.003, 0.001, 0.001, 0.001, 0.003, 0.002, 0.014, 0.001, 0.001, 0.001, 0.006, 0.0001, 0.001, 0.004, 0.667, 0.001, 0.068, 0.001, 0.0001, 0.005, 0.001, 0.001, 0.009, 0.007, 0.002, 0.003, 0.026, 0.026, 0.002, 0.024, 2.603, 0.002, 0.001, 0.003, 0.001, 0.002, 0.002, 0.003, 2.374, 0.001, 0.002, 0.001, 0.448, 0.001, 0.001, 0.005, 0.169, 0.003, 0.702, 0.002, 0.002, 0.76, 0.001, 0.004, 0.002, 0.223, 0.002, 0.382, 0.004, 0.004, 0.001, 0.0001, 0.0001, 0.028, 7.544, 0.01, 0.845, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.021, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.128, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ro":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.044, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.178, 0.003, 0.287, 0.001, 0.001, 0.038, 0.002, 0.011, 0.2, 0.201, 0.001, 0.002, 1.114, 0.333, 0.783, 0.015, 0.314, 0.397, 0.224, 0.108, 0.105, 0.107, 0.098, 0.099, 0.123, 0.221, 0.062, 0.021, 0.007, 0.006, 0.007, 0.002, 0.0001, 0.27, 0.164, 0.289, 0.16, 0.109, 0.099, 0.098, 0.077, 0.163, 0.044, 0.047, 0.132, 0.205, 0.095, 0.07, 0.207, 0.004, 0.158, 0.242, 0.12, 0.072, 0.085, 0.033, 0.021, 0.01, 0.019, 0.006, 0.0001, 0.006, 0.0001, 0.007, 0.0001, 7.568, 0.638, 3.253, 2.492, 8.352, 0.862, 0.693, 0.377, 7.77, 0.16, 0.142, 3.906, 1.919, 5.009, 3.799, 1.948, 0.008, 5.326, 2.857, 4.711, 4.259, 0.743, 0.045, 0.139, 0.103, 0.506, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.128, 0.004, 0.004, 1.675, 0.002, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.003, 0.104, 0.001, 0.001, 0.002, 0.001, 0.018, 0.003, 0.001, 0.001, 0.001, 0.016, 0.733, 0.007, 0.695, 0.006, 0.05, 0.046, 0.002, 0.038, 0.012, 0.339, 0.002, 0.003, 0.001, 0.001, 0.002, 0.004, 0.016, 0.001, 0.003, 0.001, 0.004, 0.716, 0.001, 0.007, 0.003, 0.004, 0.005, 0.003, 0.002, 0.005, 0.001, 0.003, 0.001, 0.002, 0.003, 0.007, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.048, 1.213, 1.681, 0.01, 0.0001, 0.003, 1.446, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.003, 0.016, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.127, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "id":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.029, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.265, 0.003, 0.293, 0.001, 0.002, 0.008, 0.004, 0.02, 0.156, 0.156, 0.001, 0.002, 0.897, 0.232, 0.837, 0.025, 0.281, 0.301, 0.205, 0.089, 0.081, 0.088, 0.077, 0.074, 0.084, 0.156, 0.047, 0.017, 0.004, 0.004, 0.004, 0.002, 0.0001, 0.336, 0.259, 0.156, 0.221, 0.076, 0.084, 0.101, 0.111, 0.249, 0.128, 0.292, 0.143, 0.276, 0.131, 0.06, 0.365, 0.008, 0.137, 0.448, 0.233, 0.076, 0.043, 0.063, 0.011, 0.049, 0.014, 0.01, 0.0001, 0.01, 0.0001, 0.002, 0.0001, 14.771, 1.913, 0.506, 3.424, 6.588, 0.273, 2.854, 1.797, 6.389, 0.58, 3.078, 2.893, 3.104, 7.626, 2.047, 2.047, 0.011, 4.279, 3.371, 3.841, 3.795, 0.171, 0.34, 0.026, 1.249, 0.063, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.031, 0.005, 0.004, 0.003, 0.003, 0.002, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.004, 0.002, 0.001, 0.001, 0.001, 0.001, 0.012, 0.003, 0.001, 0.001, 0.001, 0.002, 0.005, 0.001, 0.001, 0.006, 0.006, 0.001, 0.002, 0.051, 0.005, 0.002, 0.002, 0.003, 0.001, 0.002, 0.003, 0.002, 0.009, 0.001, 0.002, 0.001, 0.003, 0.001, 0.001, 0.004, 0.003, 0.004, 0.003, 0.002, 0.002, 0.002, 0.001, 0.003, 0.002, 0.002, 0.002, 0.004, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.055, 0.03, 0.005, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.003, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.006, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.03, 0.003, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sk":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.159, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.105, 0.002, 0.192, 0.0001, 0.0001, 0.007, 0.002, 0.005, 0.209, 0.21, 0.013, 0.002, 0.819, 0.162, 1.046, 0.023, 0.302, 0.407, 0.233, 0.125, 0.121, 0.119, 0.111, 0.11, 0.127, 0.222, 0.055, 0.011, 0.002, 0.003, 0.002, 0.001, 0.0001, 0.172, 0.157, 0.128, 0.107, 0.068, 0.073, 0.08, 0.101, 0.088, 0.103, 0.136, 0.098, 0.191, 0.186, 0.106, 0.263, 0.004, 0.11, 0.26, 0.138, 0.041, 0.2, 0.032, 0.006, 0.008, 0.071, 0.001, 0.0001, 0.001, 0.0001, 0.004, 0.0001, 6.363, 1.243, 1.749, 2.177, 5.774, 0.29, 0.367, 1.611, 4.04, 1.457, 2.743, 2.816, 2.062, 4.279, 6.818, 1.868, 0.006, 3.912, 3.184, 3.285, 2.066, 3.292, 0.044, 0.067, 1.073, 1.331, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.113, 0.006, 0.004, 0.002, 0.002, 0.001, 0.001, 0.002, 0.077, 0.003, 0.0001, 0.001, 0.033, 0.618, 0.006, 0.066, 0.001, 0.001, 0.001, 0.046, 0.001, 0.006, 0.001, 0.001, 0.001, 0.013, 0.009, 0.007, 0.027, 0.001, 0.026, 0.001, 0.106, 1.828, 0.001, 0.001, 0.067, 0.259, 0.001, 0.002, 0.006, 0.586, 0.001, 0.001, 0.001, 0.717, 0.001, 0.002, 0.005, 0.002, 0.004, 0.16, 0.12, 0.002, 0.005, 0.038, 0.002, 0.001, 0.54, 0.002, 0.006, 0.806, 0.828, 0.001, 0.0001, 0.0001, 0.114, 4.297, 1.036, 1.463, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.003, 0.014, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.112, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "da":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.925, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.716, 0.002, 0.323, 0.001, 0.001, 0.007, 0.004, 0.044, 0.149, 0.15, 0.001, 0.001, 0.888, 0.199, 1.047, 0.017, 0.356, 0.494, 0.245, 0.119, 0.115, 0.124, 0.118, 0.127, 0.168, 0.257, 0.046, 0.018, 0.001, 0.002, 0.001, 0.002, 0.0001, 0.185, 0.17, 0.132, 0.265, 0.124, 0.155, 0.096, 0.211, 0.151, 0.076, 0.153, 0.12, 0.178, 0.102, 0.069, 0.125, 0.005, 0.111, 0.307, 0.131, 0.057, 0.087, 0.054, 0.005, 0.012, 0.01, 0.002, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 4.818, 1.29, 0.375, 4.241, 11.595, 1.856, 2.915, 1.153, 4.647, 0.373, 2.179, 3.858, 2.304, 5.903, 3.8, 1.073, 0.008, 6.456, 4.455, 5.128, 1.418, 1.705, 0.066, 0.033, 0.579, 0.056, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.052, 0.003, 0.002, 0.001, 0.001, 0.008, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.033, 0.003, 0.0001, 0.001, 0.001, 0.013, 0.005, 0.0001, 0.001, 0.002, 0.008, 0.001, 0.002, 0.01, 0.006, 0.001, 0.001, 0.01, 0.595, 0.559, 0.002, 0.002, 0.02, 0.001, 0.004, 0.001, 0.004, 0.001, 0.001, 0.005, 0.002, 0.003, 0.005, 0.001, 0.001, 0.011, 0.001, 0.585, 0.001, 0.002, 0.003, 0.011, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.02, 1.836, 0.004, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.052, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.851, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.647, 0.002, 0.239, 0.0001, 0.0001, 0.006, 0.003, 0.009, 0.115, 0.115, 0.0001, 0.004, 0.594, 0.296, 1.014, 0.011, 0.404, 0.475, 0.268, 0.112, 0.107, 0.117, 0.106, 0.107, 0.133, 0.295, 0.069, 0.007, 0.003, 0.004, 0.003, 0.001, 0.0001, 0.183, 0.111, 0.1, 0.068, 0.113, 0.064, 0.065, 0.195, 0.087, 0.098, 0.225, 0.146, 0.211, 0.097, 0.06, 0.172, 0.005, 0.116, 0.314, 0.181, 0.037, 0.143, 0.044, 0.006, 0.048, 0.009, 0.001, 0.0001, 0.001, 0.0001, 0.004, 0.0001, 9.681, 0.162, 0.176, 0.832, 6.272, 0.12, 0.289, 1.322, 8.475, 1.576, 3.754, 4.597, 2.281, 6.958, 4.47, 1.345, 0.007, 2.326, 6.029, 6.589, 4.108, 1.653, 0.05, 0.021, 1.301, 0.041, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.101, 0.002, 0.002, 0.001, 0.004, 0.002, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.061, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.008, 0.0001, 0.001, 0.001, 0.032, 0.0001, 0.001, 0.032, 0.02, 0.001, 0.001, 2.624, 0.003, 0.001, 0.001, 0.002, 0.014, 0.0001, 0.002, 0.001, 0.01, 0.001, 0.001, 0.003, 0.002, 0.002, 0.005, 0.001, 0.001, 0.349, 0.001, 0.002, 0.001, 0.002, 0.001, 0.005, 0.002, 0.004, 0.001, 0.0001, 0.0001, 0.039, 3.028, 0.006, 0.023, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.007, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.101, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "th":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.353, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.736, 0.001, 0.084, 0.0001, 0.0001, 0.003, 0.001, 0.003, 0.081, 0.081, 0.0001, 0.001, 0.043, 0.029, 0.16, 0.005, 0.088, 0.106, 0.121, 0.047, 0.051, 0.082, 0.032, 0.03, 0.033, 0.045, 0.008, 0.004, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.013, 0.009, 0.013, 0.008, 0.008, 0.006, 0.006, 0.006, 0.008, 0.003, 0.003, 0.006, 0.01, 0.006, 0.005, 0.009, 0.001, 0.007, 0.015, 0.012, 0.003, 0.003, 0.006, 0.001, 0.002, 0.001, 0.003, 0.0001, 0.003, 0.0001, 0.001, 0.0001, 0.08, 0.011, 0.029, 0.025, 0.092, 0.012, 0.017, 0.027, 0.069, 0.001, 0.009, 0.042, 0.023, 0.063, 0.066, 0.017, 0.001, 0.062, 0.045, 0.056, 0.028, 0.008, 0.007, 0.003, 0.015, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 1.311, 1.859, 0.629, 0.364, 0.845, 0.001, 0.034, 1.547, 1.721, 0.971, 0.381, 0.156, 0.367, 0.089, 0.014, 0.016, 0.045, 0.009, 0.014, 0.115, 0.776, 0.653, 0.138, 0.742, 0.12, 1.918, 0.573, 0.602, 0.112, 0.028, 0.443, 0.069, 0.115, 1.089, 0.883, 1.745, 0.026, 0.859, 0.001, 0.829, 0.228, 0.108, 0.682, 0.53, 0.008, 1.369, 0.031, 0.006, 0.627, 1.083, 2.149, 0.218, 0.714, 0.916, 0.178, 0.322, 26.536, 5.927, 0.003, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.007, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 31.884, 0.001, 0.018, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bg":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.55, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.448, 0.001, 0.106, 0.0001, 0.0001, 0.005, 0.001, 0.003, 0.12, 0.12, 0.002, 0.001, 0.557, 0.131, 0.613, 0.011, 0.182, 0.272, 0.137, 0.074, 0.072, 0.075, 0.066, 0.065, 0.083, 0.144, 0.028, 0.009, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.013, 0.009, 0.015, 0.008, 0.007, 0.006, 0.006, 0.006, 0.041, 0.002, 0.003, 0.007, 0.011, 0.006, 0.005, 0.01, 0.001, 0.006, 0.015, 0.011, 0.003, 0.01, 0.005, 0.007, 0.001, 0.001, 0.003, 0.0001, 0.003, 0.0001, 0.002, 0.0001, 0.088, 0.012, 0.031, 0.028, 0.092, 0.009, 0.016, 0.024, 0.077, 0.002, 0.014, 0.045, 0.037, 0.056, 0.066, 0.019, 0.001, 0.063, 0.052, 0.05, 0.037, 0.008, 0.006, 0.003, 0.013, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.651, 2.091, 3.127, 0.625, 0.166, 0.165, 0.297, 0.452, 0.133, 0.189, 0.677, 0.001, 0.018, 0.001, 0.079, 0.727, 0.091, 0.092, 0.108, 0.095, 0.081, 0.039, 0.009, 0.034, 0.052, 0.011, 0.114, 0.044, 0.167, 0.089, 0.136, 0.155, 0.116, 0.171, 0.083, 0.024, 0.037, 0.04, 0.014, 0.018, 0.016, 0.009, 0.001, 0.0001, 0.001, 0.002, 0.012, 0.008, 5.212, 0.516, 1.875, 0.701, 1.296, 3.589, 0.274, 0.882, 3.979, 0.288, 1.391, 1.465, 0.909, 3.169, 3.698, 1.109, 0.0001, 0.0001, 0.048, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.0001, 0.015, 0.006, 31.942, 11.185, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.201, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "he":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.485, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.289, 0.001, 0.262, 0.0001, 0.0001, 0.005, 0.001, 0.096, 0.104, 0.103, 0.0001, 0.001, 0.64, 0.203, 0.573, 0.005, 0.181, 0.234, 0.129, 0.06, 0.061, 0.062, 0.055, 0.054, 0.065, 0.138, 0.049, 0.013, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.016, 0.011, 0.014, 0.009, 0.007, 0.007, 0.006, 0.007, 0.009, 0.003, 0.003, 0.008, 0.012, 0.007, 0.005, 0.01, 0.001, 0.008, 0.016, 0.012, 0.003, 0.004, 0.005, 0.002, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.007, 0.0001, 0.073, 0.008, 0.021, 0.022, 0.081, 0.015, 0.013, 0.021, 0.056, 0.001, 0.007, 0.043, 0.024, 0.051, 0.061, 0.011, 0.001, 0.058, 0.038, 0.043, 0.032, 0.007, 0.005, 0.003, 0.012, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.003, 0.002, 0.003, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 2.008, 2.447, 0.696, 1.135, 3.773, 4.868, 0.394, 0.995, 0.678, 4.903, 0.173, 0.854, 2.776, 1.153, 2.22, 0.562, 1.585, 0.919, 1.159, 0.101, 0.969, 0.062, 0.568, 1.054, 2.634, 1.902, 2.428, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.009, 0.002, 0.002, 0.002, 0.006, 0.004, 0.005, 0.005, 0.008, 0.005, 0.001, 0.002, 0.01, 0.002, 0.005, 0.001, 0.0001, 0.0001, 0.008, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.015, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.044, 42.985, 0.006, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.013, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "uk":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.595, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.309, 0.001, 0.06, 0.0001, 0.001, 0.01, 0.001, 0.059, 0.134, 0.135, 0.002, 0.002, 0.619, 0.137, 0.568, 0.01, 0.199, 0.281, 0.159, 0.081, 0.077, 0.082, 0.071, 0.067, 0.079, 0.158, 0.041, 0.017, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.014, 0.009, 0.015, 0.009, 0.007, 0.006, 0.007, 0.006, 0.029, 0.002, 0.003, 0.007, 0.011, 0.006, 0.005, 0.01, 0.001, 0.008, 0.016, 0.01, 0.003, 0.01, 0.004, 0.011, 0.001, 0.001, 0.003, 0.0001, 0.003, 0.0001, 0.004, 0.0001, 0.067, 0.008, 0.022, 0.02, 0.069, 0.01, 0.012, 0.018, 0.056, 0.001, 0.008, 0.037, 0.02, 0.046, 0.054, 0.014, 0.001, 0.051, 0.037, 0.039, 0.027, 0.007, 0.006, 0.003, 0.012, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.481, 1.842, 2.043, 1.429, 0.162, 0.46, 0.448, 0.496, 0.265, 0.125, 0.001, 0.003, 0.806, 0.001, 0.316, 0.84, 0.08, 0.077, 0.114, 0.065, 0.394, 0.018, 2.734, 0.422, 0.001, 0.01, 0.11, 0.047, 0.088, 0.083, 0.052, 0.13, 0.228, 0.124, 0.058, 0.089, 0.032, 0.023, 0.02, 0.023, 0.023, 0.004, 0.0001, 0.09, 0.0001, 0.001, 0.008, 0.014, 3.574, 0.601, 2.221, 0.664, 1.335, 1.986, 0.299, 0.851, 2.427, 0.557, 1.658, 1.688, 1.249, 3.061, 4.029, 1.082, 0.0001, 0.0001, 0.335, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.018, 0.0001, 0.002, 0.001, 28.71, 14.784, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.144, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lt":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.086, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.626, 0.002, 0.167, 0.001, 0.0001, 0.009, 0.001, 0.01, 0.234, 0.234, 0.001, 0.002, 1.069, 0.088, 1.436, 0.009, 0.347, 0.549, 0.256, 0.135, 0.132, 0.151, 0.128, 0.13, 0.15, 0.368, 0.06, 0.018, 0.001, 0.002, 0.002, 0.001, 0.0001, 0.213, 0.143, 0.054, 0.128, 0.066, 0.049, 0.096, 0.041, 0.157, 0.121, 0.23, 0.188, 0.16, 0.109, 0.037, 0.238, 0.002, 0.129, 0.21, 0.163, 0.036, 0.209, 0.013, 0.047, 0.01, 0.016, 0.002, 0.0001, 0.002, 0.0001, 0.003, 0.0001, 8.107, 0.954, 0.391, 1.797, 4.13, 0.204, 1.223, 0.172, 9.411, 1.587, 2.883, 2.415, 2.501, 3.736, 4.946, 1.811, 0.003, 4.047, 5.62, 3.782, 3.399, 1.76, 0.016, 0.008, 1.047, 0.248, 0.0001, 0.015, 0.0001, 0.002, 0.0001, 0.475, 0.005, 0.003, 0.002, 0.002, 0.411, 0.001, 0.001, 0.006, 0.001, 0.001, 0.001, 0.019, 0.313, 0.0001, 0.001, 0.001, 0.001, 0.006, 0.247, 0.001, 0.0001, 0.001, 1.225, 0.001, 0.136, 0.001, 0.001, 0.108, 0.003, 0.111, 0.001, 0.364, 0.781, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.003, 0.002, 0.299, 0.001, 0.004, 0.013, 0.355, 0.007, 0.002, 0.007, 0.931, 0.001, 0.004, 0.001, 0.001, 0.004, 0.002, 0.003, 0.003, 0.003, 0.037, 0.575, 0.001, 0.0001, 0.0001, 0.29, 0.016, 2.467, 2.697, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.033, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.477, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.115, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.127, 0.002, 0.244, 0.0001, 0.0001, 0.007, 0.004, 0.029, 0.125, 0.125, 0.001, 0.001, 0.736, 0.236, 1.026, 0.016, 0.357, 0.45, 0.2, 0.113, 0.108, 0.13, 0.122, 0.121, 0.148, 0.271, 0.033, 0.009, 0.004, 0.002, 0.004, 0.001, 0.0001, 0.218, 0.193, 0.121, 0.247, 0.133, 0.148, 0.105, 0.221, 0.171, 0.071, 0.137, 0.127, 0.194, 0.145, 0.08, 0.133, 0.007, 0.124, 0.352, 0.152, 0.062, 0.099, 0.053, 0.006, 0.016, 0.016, 0.005, 0.0001, 0.005, 0.0001, 0.002, 0.001, 6.479, 0.879, 0.246, 3.008, 9.683, 1.285, 2.701, 0.948, 5.112, 0.784, 2.645, 3.726, 2.383, 5.836, 3.991, 1.273, 0.009, 6.373, 4.403, 5.512, 1.465, 1.904, 0.067, 0.025, 0.761, 0.055, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.031, 0.01, 0.005, 0.003, 0.003, 0.012, 0.002, 0.003, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.02, 0.003, 0.002, 0.002, 0.001, 0.013, 0.005, 0.002, 0.001, 0.002, 0.001, 0.001, 0.003, 0.042, 0.013, 0.002, 0.002, 0.016, 0.934, 0.093, 0.004, 0.01, 0.021, 0.004, 0.076, 0.002, 0.01, 0.001, 0.002, 0.012, 0.007, 0.039, 0.01, 0.004, 0.006, 0.015, 0.002, 0.552, 0.004, 0.006, 0.078, 0.011, 0.006, 0.007, 0.003, 0.0001, 0.0001, 0.197, 1.726, 0.009, 0.008, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.017, 0.007, 0.044, 0.016, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.01, 0.009, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.002, 0.027, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.893, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.172, 0.002, 0.34, 0.0001, 0.001, 0.011, 0.002, 0.016, 0.182, 0.182, 0.001, 0.002, 0.943, 0.135, 1.23, 0.019, 0.3, 0.38, 0.204, 0.106, 0.1, 0.109, 0.096, 0.094, 0.112, 0.22, 0.065, 0.02, 0.009, 0.004, 0.009, 0.002, 0.0001, 0.156, 0.17, 0.109, 0.14, 0.063, 0.069, 0.111, 0.12, 0.137, 0.079, 0.163, 0.086, 0.175, 0.178, 0.118, 0.22, 0.004, 0.116, 0.267, 0.137, 0.108, 0.095, 0.03, 0.008, 0.009, 0.078, 0.011, 0.0001, 0.011, 0.0001, 0.002, 0.0001, 8.648, 1.028, 0.78, 2.344, 6.653, 0.218, 1.346, 0.572, 7.393, 3.932, 2.783, 2.724, 2.195, 4.91, 6.755, 1.994, 0.007, 4.039, 3.61, 3.329, 3.254, 2.478, 0.043, 0.016, 0.083, 1.288, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.039, 0.005, 0.004, 0.003, 0.002, 0.001, 0.003, 0.353, 0.002, 0.001, 0.001, 0.001, 0.016, 0.678, 0.001, 0.001, 0.004, 0.158, 0.001, 0.011, 0.002, 0.001, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.009, 0.005, 0.008, 0.001, 0.033, 0.524, 0.003, 0.002, 0.003, 0.001, 0.001, 0.002, 0.002, 0.01, 0.001, 0.005, 0.001, 0.004, 0.001, 0.001, 0.008, 0.004, 0.005, 0.005, 0.002, 0.004, 0.004, 0.001, 0.004, 0.002, 0.004, 0.006, 0.006, 0.016, 0.36, 0.002, 0.0001, 0.0001, 0.021, 0.044, 1.208, 0.914, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.011, 0.005, 0.028, 0.01, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.038, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "no":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.028, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.853, 0.002, 0.247, 0.0001, 0.001, 0.006, 0.004, 0.016, 0.159, 0.158, 0.001, 0.001, 0.698, 0.213, 1.037, 0.017, 0.377, 0.496, 0.255, 0.116, 0.113, 0.123, 0.117, 0.116, 0.152, 0.295, 0.042, 0.013, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.196, 0.176, 0.125, 0.246, 0.126, 0.148, 0.099, 0.211, 0.167, 0.071, 0.132, 0.135, 0.185, 0.133, 0.091, 0.127, 0.006, 0.11, 0.321, 0.146, 0.058, 0.092, 0.051, 0.007, 0.014, 0.011, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 4.956, 1.168, 0.243, 2.996, 11.38, 1.384, 2.632, 1.02, 4.719, 0.546, 2.591, 3.946, 2.341, 6.218, 3.979, 1.354, 0.009, 6.417, 4.712, 5.821, 1.424, 1.732, 0.061, 0.029, 0.639, 0.049, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.041, 0.006, 0.003, 0.002, 0.002, 0.009, 0.002, 0.002, 0.001, 0.002, 0.001, 0.001, 0.002, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.034, 0.002, 0.001, 0.002, 0.001, 0.014, 0.003, 0.001, 0.001, 0.001, 0.002, 0.001, 0.002, 0.028, 0.009, 0.001, 0.002, 0.012, 0.765, 0.126, 0.003, 0.003, 0.021, 0.001, 0.062, 0.001, 0.006, 0.001, 0.001, 0.007, 0.003, 0.006, 0.006, 0.002, 0.003, 0.012, 0.001, 0.598, 0.002, 0.004, 0.062, 0.009, 0.004, 0.004, 0.002, 0.0001, 0.0001, 0.152, 1.588, 0.007, 0.007, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.004, 0.022, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.039, 0.001, 0.001, 0.004, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.872, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.68, 0.001, 0.1, 0.0001, 0.0001, 0.009, 0.0001, 0.005, 0.176, 0.176, 0.0001, 0.003, 0.5, 0.178, 0.762, 0.011, 0.275, 0.318, 0.214, 0.099, 0.096, 0.093, 0.078, 0.075, 0.084, 0.129, 0.031, 0.008, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.017, 0.01, 0.025, 0.013, 0.007, 0.006, 0.019, 0.007, 0.026, 0.003, 0.008, 0.007, 0.014, 0.016, 0.013, 0.016, 0.001, 0.009, 0.02, 0.011, 0.006, 0.008, 0.003, 0.004, 0.001, 0.003, 0.002, 0.0001, 0.002, 0.0001, 0.018, 0.0001, 0.453, 0.047, 0.05, 0.128, 0.37, 0.027, 0.066, 0.039, 0.393, 0.16, 0.152, 0.148, 0.154, 0.268, 0.352, 0.1, 0.001, 0.219, 0.193, 0.185, 0.165, 0.107, 0.003, 0.002, 0.007, 0.07, 0.053, 0.001, 0.053, 0.0001, 0.0001, 2.152, 2.07, 1.61, 1.756, 0.112, 0.204, 0.344, 0.339, 0.366, 0.003, 0.007, 0.001, 0.001, 0.031, 0.0001, 0.007, 0.082, 0.095, 0.143, 0.054, 0.071, 0.047, 0.006, 0.035, 1.459, 0.284, 0.347, 0.2, 0.143, 0.119, 0.086, 0.186, 0.072, 0.175, 0.071, 0.052, 0.034, 0.041, 0.014, 0.02, 0.016, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 4.933, 0.477, 1.401, 0.663, 1.33, 3.708, 0.225, 0.704, 3.913, 0.001, 1.472, 1.2, 1.198, 2.623, 3.682, 1.022, 0.0001, 0.0001, 0.018, 0.003, 0.054, 0.041, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 30.181, 10.982, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.062, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ca":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.816, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.948, 0.002, 0.294, 0.001, 0.011, 0.035, 0.001, 0.634, 0.154, 0.154, 0.001, 0.002, 1.001, 0.144, 0.747, 0.01, 0.301, 0.411, 0.25, 0.137, 0.131, 0.135, 0.12, 0.123, 0.144, 0.212, 0.051, 0.029, 0.002, 0.003, 0.003, 0.001, 0.0001, 0.252, 0.125, 0.23, 0.119, 0.296, 0.09, 0.091, 0.066, 0.12, 0.061, 0.034, 0.213, 0.174, 0.072, 0.049, 0.171, 0.012, 0.097, 0.192, 0.11, 0.053, 0.092, 0.024, 0.034, 0.01, 0.009, 0.002, 0.0001, 0.002, 0.0001, 0.004, 0.0001, 9.132, 1.004, 2.746, 3.236, 9.343, 0.681, 0.95, 0.465, 5.412, 0.169, 0.095, 4.932, 2.114, 4.848, 3.551, 1.884, 0.571, 5.202, 5.696, 4.416, 2.672, 1.094, 0.036, 0.312, 0.252, 0.123, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.044, 0.004, 0.004, 0.002, 0.002, 0.001, 0.001, 0.001, 0.002, 0.015, 0.001, 0.001, 0.001, 0.005, 0.001, 0.001, 0.001, 0.001, 0.002, 0.006, 0.003, 0.001, 0.001, 0.001, 0.001, 0.021, 0.001, 0.001, 0.003, 0.003, 0.001, 0.001, 0.327, 0.012, 0.002, 0.002, 0.002, 0.001, 0.001, 0.088, 0.218, 0.355, 0.001, 0.01, 0.003, 0.236, 0.001, 0.038, 0.005, 0.007, 0.161, 0.374, 0.002, 0.003, 0.003, 0.047, 0.003, 0.002, 0.063, 0.01, 0.034, 0.003, 0.002, 0.002, 0.0001, 0.0001, 0.099, 1.903, 0.005, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.004, 0.012, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.039, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sl":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.06, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.437, 0.024, 0.237, 0.001, 0.001, 0.007, 0.002, 0.011, 0.174, 0.174, 0.021, 0.002, 1.072, 0.17, 1.037, 0.022, 0.277, 0.429, 0.215, 0.122, 0.124, 0.121, 0.109, 0.108, 0.134, 0.239, 0.061, 0.025, 0.005, 0.006, 0.005, 0.002, 0.0001, 0.162, 0.141, 0.1, 0.122, 0.063, 0.075, 0.091, 0.086, 0.111, 0.082, 0.154, 0.138, 0.185, 0.145, 0.099, 0.224, 0.004, 0.106, 0.263, 0.133, 0.042, 0.163, 0.031, 0.007, 0.007, 0.087, 0.013, 0.0001, 0.014, 0.0001, 0.006, 0.0001, 7.7, 1.204, 0.709, 2.364, 7.782, 0.229, 1.139, 0.879, 6.985, 3.327, 2.701, 3.64, 2.037, 5.283, 6.653, 2.232, 0.006, 4.152, 3.513, 3.409, 1.654, 3.049, 0.039, 0.016, 0.079, 1.473, 0.0001, 0.01, 0.0001, 0.0001, 0.0001, 0.054, 0.004, 0.003, 0.002, 0.002, 0.001, 0.001, 0.011, 0.002, 0.002, 0.0001, 0.001, 0.021, 0.847, 0.001, 0.0001, 0.001, 0.002, 0.002, 0.027, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.056, 0.644, 0.007, 0.001, 0.003, 0.001, 0.001, 0.002, 0.003, 0.013, 0.001, 0.027, 0.001, 0.005, 0.001, 0.001, 0.007, 0.003, 0.004, 0.005, 0.002, 0.003, 0.006, 0.001, 0.004, 0.002, 0.004, 0.028, 0.008, 0.018, 0.391, 0.002, 0.0001, 0.0001, 0.071, 0.059, 0.881, 1.071, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.01, 0.005, 0.024, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.054, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.879, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.099, 0.004, 0.432, 0.0001, 0.0001, 0.013, 0.002, 0.007, 0.207, 0.208, 0.0001, 0.003, 0.965, 0.082, 1.276, 0.01, 0.332, 0.476, 0.254, 0.122, 0.117, 0.123, 0.105, 0.106, 0.127, 0.271, 0.045, 0.023, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.208, 0.134, 0.062, 0.128, 0.074, 0.067, 0.074, 0.058, 0.112, 0.068, 0.189, 0.194, 0.144, 0.089, 0.055, 0.234, 0.002, 0.136, 0.249, 0.163, 0.042, 0.182, 0.012, 0.007, 0.003, 0.051, 0.001, 0.0001, 0.001, 0.0001, 0.003, 0.0001, 8.58, 1.078, 0.806, 2.221, 4.451, 0.231, 1.228, 0.175, 6.667, 1.704, 2.603, 2.424, 2.389, 3.209, 2.883, 1.908, 0.003, 4.056, 5.825, 4.121, 3.633, 1.801, 0.012, 0.009, 0.029, 1.289, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.124, 2.988, 0.003, 0.002, 0.001, 0.006, 0.331, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.015, 0.083, 0.0001, 0.001, 0.001, 0.001, 0.007, 1.174, 0.07, 0.0001, 0.001, 0.001, 0.002, 0.003, 0.001, 0.001, 0.005, 0.012, 0.009, 0.001, 0.06, 0.627, 0.004, 0.097, 0.002, 0.001, 0.001, 0.001, 0.001, 0.002, 0.006, 1.565, 0.0001, 0.002, 0.0001, 0.0001, 0.01, 0.002, 0.005, 0.002, 0.002, 0.005, 0.01, 0.106, 0.006, 0.002, 0.003, 0.01, 0.298, 0.012, 0.176, 0.002, 0.0001, 0.0001, 0.03, 0.013, 6.068, 1.452, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.051, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.11, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "et":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.183, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.759, 0.003, 0.281, 0.0001, 0.0001, 0.013, 0.001, 0.037, 0.198, 0.199, 0.001, 0.003, 0.786, 0.203, 1.175, 0.017, 0.35, 0.548, 0.272, 0.142, 0.137, 0.143, 0.127, 0.129, 0.154, 0.323, 0.059, 0.022, 0.017, 0.003, 0.017, 0.003, 0.0001, 0.235, 0.096, 0.074, 0.061, 0.173, 0.056, 0.064, 0.105, 0.122, 0.088, 0.255, 0.166, 0.186, 0.114, 0.065, 0.208, 0.003, 0.138, 0.296, 0.251, 0.046, 0.167, 0.033, 0.011, 0.008, 0.01, 0.008, 0.0001, 0.008, 0.0001, 0.004, 0.0001, 9.665, 0.664, 0.152, 2.822, 7.678, 0.189, 1.393, 1.095, 7.816, 1.25, 3.234, 4.738, 2.585, 4.03, 3.549, 1.167, 0.005, 3.003, 6.68, 5.333, 4.153, 1.613, 0.043, 0.017, 0.074, 0.045, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.13, 0.015, 0.01, 0.006, 0.004, 0.003, 0.003, 0.004, 0.002, 0.002, 0.001, 0.002, 0.003, 0.005, 0.001, 0.003, 0.002, 0.002, 0.003, 0.102, 0.002, 0.008, 0.003, 0.003, 0.002, 0.004, 0.002, 0.001, 0.044, 0.005, 0.006, 0.003, 0.016, 0.035, 0.003, 0.002, 0.833, 0.002, 0.001, 0.002, 0.002, 0.01, 0.001, 0.006, 0.001, 0.005, 0.001, 0.001, 0.017, 0.004, 0.012, 0.007, 0.005, 0.763, 0.179, 0.003, 0.015, 0.005, 0.008, 0.007, 0.518, 0.012, 0.028, 0.003, 0.0001, 0.0001, 0.02, 2.358, 0.019, 0.061, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.004, 0.104, 0.037, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.123, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.374, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.123, 0.002, 0.071, 0.0001, 0.001, 0.004, 0.0001, 0.023, 0.08, 0.08, 0.0001, 0.001, 0.255, 0.072, 0.052, 0.006, 0.068, 0.07, 0.044, 0.02, 0.019, 0.023, 0.019, 0.019, 0.021, 0.04, 0.021, 0.006, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.008, 0.004, 0.007, 0.004, 0.005, 0.003, 0.004, 0.003, 0.006, 0.001, 0.002, 0.003, 0.005, 0.004, 0.003, 0.005, 0.0001, 0.003, 0.008, 0.005, 0.002, 0.002, 0.002, 0.001, 0.001, 0.001, 0.007, 0.0001, 0.008, 0.0001, 0.001, 0.0001, 0.049, 0.007, 0.017, 0.016, 0.052, 0.008, 0.01, 0.017, 0.038, 0.001, 0.004, 0.024, 0.015, 0.034, 0.035, 0.012, 0.001, 0.033, 0.03, 0.034, 0.015, 0.005, 0.005, 0.002, 0.008, 0.001, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 1.039, 0.443, 1.278, 0.061, 0.0001, 0.273, 0.146, 1.879, 0.535, 0.214, 0.013, 0.729, 0.054, 1.826, 0.0001, 0.253, 0.014, 0.012, 0.0001, 0.042, 0.14, 2.07, 0.133, 0.43, 0.035, 0.004, 0.215, 0.046, 0.503, 0.014, 0.016, 0.269, 0.037, 0.213, 0.023, 0.155, 24.777, 7.162, 0.554, 0.224, 1.23, 0.009, 0.8, 0.117, 0.393, 0.245, 0.995, 0.828, 2.018, 0.001, 0.771, 0.001, 0.001, 0.707, 0.299, 0.18, 1.226, 0.94, 0.0001, 0.0001, 0.133, 0.001, 2.558, 1.303, 0.0001, 0.0001, 0.008, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.261, 0.0001, 0.024, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
};
const EXTENSIVE_LANG_FREQS = Object.assign({}, COMMON_LANG_FREQS, {
    "aa":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.161, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.548, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.29, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.29, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.452, 0.645, 0.645, 2.581, 9.032, 0.0001, 5.161, 3.871, 5.806, 0.0001, 1.935, 2.581, 1.29, 5.161, 2.581, 1.29, 0.0001, 4.516, 0.645, 3.226, 0.645, 0.0001, 1.29, 0.0001, 0.645, 1.29, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.581, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.29, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 1.29, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.871, 0.645, 2.581, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ab":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.925, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.477, 0.003, 0.06, 0.0001, 0.0001, 0.005, 0.0001, 0.013, 0.269, 0.273, 0.001, 0.001, 0.518, 0.306, 0.76, 0.006, 0.291, 0.709, 0.42, 0.294, 0.242, 0.237, 0.242, 0.222, 0.25, 0.32, 0.04, 0.028, 0.008, 0.0001, 0.008, 0.002, 0.0001, 0.004, 0.004, 0.004, 0.006, 0.001, 0.002, 0.001, 0.001, 0.033, 0.012, 0.001, 0.001, 0.002, 0.001, 0.011, 0.003, 0.0001, 0.002, 0.009, 0.002, 0.002, 0.007, 0.006, 0.01, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.679, 0.013, 0.02, 0.028, 0.073, 0.002, 0.006, 0.01, 0.057, 0.001, 0.005, 0.035, 0.039, 0.025, 0.031, 0.027, 0.011, 0.045, 0.036, 0.027, 0.037, 0.009, 0.002, 0.01, 0.027, 0.004, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 3.029, 1.109, 1.569, 1.131, 0.085, 0.805, 0.262, 0.083, 0.992, 0.002, 0.003, 2.495, 0.791, 0.003, 0.006, 0.03, 0.654, 0.059, 0.04, 0.137, 0.332, 0.075, 0.041, 0.012, 0.142, 2.586, 0.087, 1.002, 0.086, 0.047, 0.045, 0.323, 0.073, 0.328, 0.016, 0.067, 0.011, 0.052, 0.054, 0.455, 0.024, 0.199, 0.0001, 0.026, 0.015, 0.539, 0.001, 0.001, 7.771, 0.517, 0.209, 1.034, 0.683, 1.368, 0.238, 0.686, 3.093, 0.042, 0.729, 1.305, 0.754, 1.868, 1.136, 0.676, 0.0001, 0.0001, 0.065, 0.019, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.001, 0.155, 0.0001, 0.005, 0.002, 22.83, 11.878, 3.39, 2.86, 0.019, 0.007, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.386, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ace": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.36, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.198, 0.0001, 0.137, 0.0001, 0.001, 0.007, 0.0001, 0.256, 0.125, 0.125, 0.0001, 0.0001, 1.042, 0.179, 1.302, 0.01, 0.401, 0.568, 0.284, 0.118, 0.113, 0.112, 0.099, 0.093, 0.097, 0.13, 0.041, 0.007, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.777, 0.587, 0.153, 0.133, 0.028, 0.036, 0.256, 0.095, 0.205, 0.159, 0.483, 0.331, 0.444, 0.183, 0.028, 0.481, 0.019, 0.179, 0.473, 0.324, 0.101, 0.026, 0.042, 0.006, 0.021, 0.009, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 11.224, 1.773, 0.851, 1.583, 5.304, 0.086, 3.693, 3.458, 3.728, 0.528, 2.425, 2.037, 2.4, 8.165, 2.618, 1.607, 0.015, 2.485, 1.74, 2.806, 6.018, 0.112, 0.344, 0.01, 1.486, 0.04, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.039, 0.008, 0.005, 0.009, 0.016, 0.007, 0.006, 0.006, 0.01, 0.004, 0.008, 0.003, 0.002, 0.004, 0.012, 0.004, 0.007, 0.003, 0.004, 0.014, 0.002, 0.001, 0.001, 0.002, 0.004, 0.016, 0.001, 0.001, 0.002, 0.002, 0.001, 0.007, 0.007, 0.006, 0.003, 0.008, 0.005, 0.002, 0.001, 0.019, 1.193, 0.401, 0.007, 0.574, 0.003, 0.006, 0.002, 0.006, 0.025, 0.011, 0.006, 0.008, 0.873, 0.004, 0.151, 0.002, 0.005, 0.005, 0.008, 0.007, 0.004, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.025, 3.205, 0.014, 0.012, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.004, 0.001, 0.012, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.061, 0.078, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.011, 0.039, 0.001, 0.001, 0.005, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ady": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.142, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.359, 0.001, 0.089, 0.0001, 0.0001, 0.003, 0.0001, 0.006, 0.111, 0.11, 0.001, 0.001, 0.645, 0.309, 0.862, 0.007, 0.118, 0.279, 0.082, 0.059, 0.052, 0.055, 0.051, 0.045, 0.057, 0.071, 0.053, 0.011, 0.003, 0.003, 0.003, 0.001, 0.0001, 0.008, 0.007, 0.003, 0.003, 0.002, 0.003, 0.0001, 0.002, 1.228, 0.004, 0.001, 0.003, 0.004, 0.002, 0.004, 0.005, 0.0001, 0.001, 0.006, 0.003, 0.002, 0.004, 0.002, 0.008, 0.0001, 0.0001, 0.005, 0.0001, 0.005, 0.0001, 0.001, 0.0001, 0.05, 0.009, 0.016, 0.02, 0.067, 0.009, 0.011, 0.022, 0.046, 0.001, 0.006, 0.031, 0.02, 0.036, 0.037, 0.013, 0.0001, 0.038, 0.031, 0.043, 0.016, 0.004, 0.008, 0.003, 0.011, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.778, 0.778, 1.192, 2.098, 0.406, 1.886, 0.203, 0.188, 0.585, 0.672, 2.887, 2.927, 0.717, 6.004, 0.019, 0.21, 0.317, 0.122, 0.019, 0.226, 0.145, 0.045, 0.02, 0.041, 0.09, 0.005, 0.262, 0.059, 0.092, 0.079, 0.053, 0.07, 0.076, 0.092, 0.086, 0.055, 0.029, 0.124, 0.039, 0.031, 0.045, 0.011, 0.0001, 0.031, 0.0001, 0.018, 0.005, 0.018, 2.762, 0.645, 0.171, 1.681, 0.855, 1.134, 0.39, 0.89, 1.618, 0.147, 1.755, 1.169, 1.845, 1.192, 0.989, 0.792, 0.0001, 0.0001, 0.094, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.0001, 0.003, 0.004, 0.0001, 0.0001, 20.033, 23.044, 0.001, 0.227, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.229, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "af":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.732, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.494, 0.002, 0.361, 0.0001, 0.001, 0.008, 0.001, 0.297, 0.136, 0.136, 0.002, 0.001, 0.651, 0.269, 0.893, 0.01, 0.25, 0.371, 0.17, 0.095, 0.09, 0.104, 0.09, 0.086, 0.122, 0.213, 0.049, 0.019, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.241, 0.154, 0.103, 0.382, 0.093, 0.072, 0.119, 0.168, 0.14, 0.087, 0.137, 0.088, 0.157, 0.131, 0.103, 0.129, 0.004, 0.104, 0.29, 0.11, 0.03, 0.115, 0.083, 0.006, 0.008, 0.015, 0.002, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 6.202, 1.128, 0.17, 4.12, 13.284, 0.609, 2.484, 1.201, 6.602, 0.17, 2.299, 2.976, 1.671, 6.221, 4.571, 1.147, 0.006, 5.197, 4.908, 4.263, 1.7, 1.691, 1.336, 0.018, 0.832, 0.043, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.182, 0.005, 0.004, 0.003, 0.002, 0.001, 0.001, 0.001, 0.003, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.002, 0.001, 0.001, 0.024, 0.002, 0.001, 0.001, 0.001, 0.003, 0.118, 0.001, 0.001, 0.017, 0.016, 0.001, 0.001, 0.076, 0.018, 0.001, 0.005, 0.004, 0.002, 0.002, 0.003, 0.003, 0.032, 0.053, 0.119, 0.001, 0.004, 0.001, 0.014, 0.007, 0.003, 0.004, 0.007, 0.002, 0.003, 0.005, 0.001, 0.005, 0.002, 0.003, 0.003, 0.007, 0.003, 0.003, 0.002, 0.0001, 0.0001, 0.084, 0.264, 0.004, 0.005, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.003, 0.001, 0.0001, 0.009, 0.004, 0.022, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.003, 0.181, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ak":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.856, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 17.14, 0.001, 0.181, 0.0001, 0.002, 0.004, 0.001, 0.124, 0.134, 0.137, 0.001, 0.0001, 0.719, 0.119, 1.218, 0.014, 0.179, 0.257, 0.137, 0.052, 0.061, 0.075, 0.065, 0.054, 0.059, 0.197, 0.031, 0.029, 0.002, 0.015, 0.002, 0.018, 0.0001, 0.566, 0.167, 0.173, 0.118, 0.172, 0.085, 0.258, 0.093, 0.098, 0.056, 0.193, 0.111, 0.204, 0.399, 0.102, 0.121, 0.012, 0.083, 0.271, 0.145, 0.084, 0.04, 0.206, 0.011, 0.078, 0.02, 0.025, 0.0001, 0.025, 0.0001, 0.0001, 0.0001, 10.911, 1.752, 0.404, 1.704, 5.791, 1.18, 0.501, 1.542, 4.479, 0.04, 1.975, 0.667, 3.211, 7.434, 5.302, 0.888, 0.03, 2.693, 2.695, 1.838, 2.674, 0.126, 2.695, 0.023, 2.4, 0.077, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.046, 0.01, 0.002, 0.005, 0.002, 0.006, 0.095, 0.003, 0.01, 0.003, 0.006, 0.011, 0.002, 0.017, 0.002, 0.004, 0.052, 0.011, 0.001, 0.002, 1.774, 0.002, 0.002, 0.001, 0.0001, 0.02, 0.0001, 1.749, 0.01, 0.01, 0.0001, 0.0001, 0.151, 0.004, 0.001, 0.002, 0.006, 0.022, 0.001, 0.003, 0.005, 0.01, 0.002, 0.003, 0.002, 0.005, 0.001, 0.003, 0.01, 0.006, 0.005, 0.012, 0.015, 0.552, 0.007, 0.003, 0.008, 0.006, 0.006, 0.392, 0.013, 0.005, 0.007, 0.004, 0.0001, 0.0001, 0.146, 0.054, 0.004, 0.004, 0.139, 0.0001, 0.0001, 3.532, 0.002, 0.008, 0.003, 0.34, 0.547, 0.0001, 0.045, 0.018, 0.0001, 0.0001, 0.018, 0.055, 0.008, 0.002, 0.016, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.048, 0.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "als": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.981, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.805, 0.003, 0.368, 0.0001, 0.0001, 0.09, 0.001, 0.06, 0.177, 0.177, 0.009, 0.001, 0.909, 0.215, 1.001, 0.022, 0.318, 0.46, 0.232, 0.128, 0.122, 0.132, 0.116, 0.119, 0.142, 0.206, 0.063, 0.024, 0.004, 0.003, 0.004, 0.001, 0.0001, 0.315, 0.452, 0.163, 0.512, 0.205, 0.236, 0.319, 0.219, 0.188, 0.156, 0.222, 0.212, 0.32, 0.172, 0.112, 0.199, 0.01, 0.225, 0.653, 0.132, 0.131, 0.173, 0.23, 0.004, 0.019, 0.129, 0.009, 0.0001, 0.009, 0.0001, 0.003, 0.001, 3.964, 1.276, 2.626, 3.453, 8.363, 1.057, 2.308, 3.744, 6.377, 0.069, 0.66, 2.78, 2.213, 4.452, 3.12, 0.516, 0.012, 5.572, 4.629, 4.341, 2.669, 0.935, 0.979, 0.046, 0.315, 0.925, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.124, 0.003, 0.002, 0.002, 0.034, 0.001, 0.001, 0.001, 0.005, 0.003, 0.0001, 0.0001, 0.004, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.023, 0.002, 0.001, 0.01, 0.001, 0.003, 0.02, 0.003, 0.002, 0.048, 0.001, 0.034, 0.042, 0.156, 0.005, 0.005, 0.003, 1.018, 0.003, 0.001, 0.003, 0.354, 0.039, 0.002, 0.022, 0.079, 0.004, 0.001, 0.002, 0.004, 0.003, 0.015, 0.003, 0.029, 0.017, 0.333, 0.001, 0.002, 0.045, 0.004, 0.015, 0.5, 0.004, 0.001, 0.002, 0.0001, 0.0001, 0.108, 2.635, 0.006, 0.005, 0.0001, 0.005, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.011, 0.005, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.005, 0.12, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "am":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.067, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.441, 0.005, 0.08, 0.001, 0.0001, 0.003, 0.0001, 0.013, 0.12, 0.121, 0.002, 0.001, 0.021, 0.111, 0.25, 0.041, 0.102, 0.167, 0.089, 0.049, 0.044, 0.048, 0.044, 0.043, 0.057, 0.081, 0.018, 0.001, 0.048, 0.019, 0.048, 0.008, 0.0001, 0.009, 0.005, 0.007, 0.005, 0.005, 0.004, 0.003, 0.003, 0.004, 0.004, 0.002, 0.003, 0.006, 0.003, 0.002, 0.004, 0.001, 0.003, 0.007, 0.005, 0.002, 0.002, 0.002, 0.001, 0.001, 0.001, 0.017, 0.0001, 0.02, 0.0001, 0.007, 0.0001, 0.059, 0.06, 0.021, 0.018, 0.066, 0.009, 0.014, 0.02, 0.05, 0.001, 0.005, 0.029, 0.021, 0.042, 0.045, 0.014, 0.001, 0.09, 0.032, 0.04, 0.026, 0.005, 0.007, 0.003, 0.012, 0.002, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.402, 0.178, 0.052, 0.194, 0.053, 0.478, 0.259, 0.003, 10.51, 5.557, 5.996, 6.414, 2.305, 3.741, 0.258, 0.015, 0.706, 0.091, 0.071, 0.613, 0.064, 1.598, 0.107, 0.008, 0.907, 0.126, 0.312, 0.688, 0.12, 0.989, 0.129, 0.009, 2.006, 0.213, 0.679, 0.599, 0.206, 1.204, 0.134, 0.012, 1.72, 0.213, 0.231, 1.059, 0.087, 1.793, 0.284, 0.013, 1.151, 0.255, 0.312, 0.726, 0.115, 2.127, 0.177, 0.025, 0.19, 0.059, 0.032, 0.208, 0.015, 0.466, 0.016, 0.003, 0.0001, 0.0001, 0.096, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.005, 0.009, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.017, 0.046, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 29.467, 0.047, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.001, 0.0001, 0.017, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "an":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.253, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.49, 0.005, 0.725, 0.0001, 0.0001, 0.005, 0.001, 0.998, 0.246, 0.246, 0.002, 0.002, 1.083, 0.164, 0.685, 0.057, 0.291, 0.382, 0.213, 0.125, 0.12, 0.124, 0.115, 0.119, 0.131, 0.221, 0.057, 0.029, 0.007, 0.01, 0.006, 0.001, 0.0001, 0.411, 0.169, 0.298, 0.091, 0.216, 0.095, 0.1, 0.059, 0.154, 0.037, 0.024, 0.177, 0.199, 0.072, 0.146, 0.19, 0.011, 0.122, 0.227, 0.128, 0.065, 0.101, 0.021, 0.037, 0.032, 0.028, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 9.483, 1.074, 3.3, 3.436, 7.765, 0.618, 0.822, 0.72, 5.365, 0.027, 0.17, 3.124, 1.916, 5.869, 6.23, 1.654, 0.435, 4.741, 4.813, 3.981, 2.96, 0.573, 0.028, 0.256, 1.248, 0.309, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.014, 0.007, 0.003, 0.003, 0.002, 0.001, 0.001, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.007, 0.002, 0.001, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.002, 0.028, 0.174, 0.002, 0.002, 0.003, 0.001, 0.001, 0.008, 0.012, 0.227, 0.002, 0.014, 0.002, 0.209, 0.001, 0.002, 0.004, 0.013, 0.086, 0.54, 0.002, 0.002, 0.003, 0.002, 0.004, 0.002, 0.027, 0.014, 0.019, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.127, 1.249, 0.007, 0.007, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.009, 0.005, 0.014, 0.005, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.002, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.013, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ang": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.542, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.629, 0.001, 0.406, 0.001, 0.001, 0.005, 0.001, 0.041, 0.166, 0.166, 0.001, 0.001, 0.772, 0.085, 0.973, 0.007, 0.229, 0.292, 0.152, 0.081, 0.082, 0.095, 0.083, 0.089, 0.101, 0.139, 0.058, 0.032, 0.011, 0.001, 0.011, 0.001, 0.0001, 0.204, 0.193, 0.317, 0.089, 0.179, 0.148, 0.229, 0.279, 0.189, 0.034, 0.031, 0.128, 0.195, 0.168, 0.087, 0.103, 0.007, 0.125, 0.419, 0.122, 0.043, 0.034, 0.145, 0.006, 0.012, 0.007, 0.02, 0.0001, 0.02, 0.0001, 0.0001, 0.0001, 5.666, 0.997, 2.318, 3.22, 8.139, 1.491, 2.061, 1.574, 3.89, 0.022, 0.109, 2.731, 2.332, 6.4, 3.389, 0.62, 0.014, 4.435, 4.532, 3.015, 1.701, 0.127, 1.341, 0.09, 0.658, 0.04, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 0.032, 0.62, 0.006, 0.006, 0.004, 0.003, 0.052, 0.002, 0.001, 0.001, 0.002, 0.033, 0.008, 0.478, 0.002, 0.002, 0.01, 0.003, 0.05, 1.069, 0.004, 0.001, 0.004, 0.002, 0.003, 0.003, 0.011, 0.012, 0.009, 0.068, 0.141, 0.003, 0.009, 0.037, 0.013, 0.751, 0.006, 0.002, 1.085, 0.003, 0.002, 0.01, 0.039, 0.996, 0.002, 0.008, 0.002, 0.002, 0.371, 0.007, 0.005, 0.069, 0.002, 0.003, 0.002, 0.008, 0.006, 0.003, 0.005, 0.004, 0.005, 0.004, 2.003, 0.078, 0.0001, 0.0001, 0.009, 3.7, 2.566, 0.742, 0.075, 0.766, 0.127, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.012, 0.006, 0.017, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.006, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.024, 0.022, 0.003, 0.001, 0.003, 0.002, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "arc": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.038, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.39, 0.001, 0.055, 0.0001, 0.0001, 0.007, 0.0001, 0.005, 0.294, 0.294, 0.0001, 0.0001, 0.039, 0.041, 0.295, 0.017, 0.207, 0.161, 0.078, 0.046, 0.044, 0.053, 0.042, 0.044, 0.043, 0.091, 0.189, 0.006, 0.003, 0.004, 0.003, 0.0001, 0.0001, 0.01, 0.01, 0.013, 0.007, 0.004, 0.004, 0.006, 0.005, 0.007, 0.003, 0.005, 0.008, 0.011, 0.008, 0.004, 0.008, 0.001, 0.007, 0.013, 0.004, 0.003, 0.005, 0.004, 0.001, 0.001, 0.002, 0.005, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.107, 0.013, 0.023, 0.039, 0.088, 0.011, 0.022, 0.025, 0.081, 0.003, 0.021, 0.05, 0.023, 0.07, 0.066, 0.018, 0.002, 0.062, 0.042, 0.051, 0.032, 0.013, 0.011, 0.006, 0.012, 0.006, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.359, 0.027, 0.139, 0.022, 0.095, 0.021, 0.095, 0.051, 0.776, 0.005, 0.029, 0.002, 0.032, 0.003, 0.011, 0.005, 6.959, 0.008, 1.918, 0.561, 0.013, 2.47, 0.003, 1.261, 3.75, 0.282, 0.787, 0.504, 0.018, 4.683, 0.009, 0.786, 1.796, 2.249, 2.761, 0.874, 0.009, 1.007, 0.747, 0.053, 0.199, 0.858, 2.538, 1.15, 2.879, 0.016, 0.009, 0.021, 0.023, 0.056, 0.023, 0.019, 0.01, 0.046, 0.007, 0.011, 0.024, 0.035, 0.015, 0.012, 0.048, 0.023, 0.008, 0.047, 0.0001, 0.0001, 0.004, 0.019, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.832, 0.001, 0.126, 0.053, 0.042, 0.017, 0.001, 0.0001, 0.0001, 0.009, 0.024, 0.108, 0.212, 0.141, 0.001, 0.004, 41.501, 0.031, 0.0001, 0.0001, 0.002, 0.019, 0.018, 0.0001, 0.001, 0.004, 0.004, 0.0001, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "arz": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.02, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.739, 0.003, 0.126, 0.0001, 0.0001, 0.004, 0.001, 0.003, 0.118, 0.124, 0.002, 0.001, 0.064, 0.045, 0.405, 0.01, 0.141, 0.269, 0.129, 0.067, 0.063, 0.072, 0.064, 0.065, 0.08, 0.165, 0.039, 0.002, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.012, 0.009, 0.011, 0.008, 0.005, 0.005, 0.005, 0.006, 0.006, 0.005, 0.004, 0.009, 0.011, 0.005, 0.003, 0.007, 0.0001, 0.006, 0.013, 0.009, 0.001, 0.004, 0.004, 0.001, 0.001, 0.001, 0.006, 0.001, 0.006, 0.0001, 0.002, 0.0001, 0.091, 0.01, 0.025, 0.026, 0.093, 0.01, 0.015, 0.024, 0.072, 0.002, 0.01, 0.045, 0.023, 0.064, 0.06, 0.013, 0.001, 0.06, 0.046, 0.047, 0.027, 0.009, 0.007, 0.004, 0.017, 0.005, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.111, 1.136, 0.763, 1.043, 4.458, 2.752, 2.413, 1.721, 2.708, 1.077, 3.156, 0.021, 0.238, 0.002, 0.017, 0.028, 0.008, 0.018, 0.006, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.003, 0.003, 0.004, 0.0001, 0.003, 0.019, 0.06, 0.018, 0.274, 0.041, 0.116, 0.08, 6.51, 1.771, 0.79, 1.749, 0.151, 0.593, 0.743, 0.294, 1.313, 0.079, 2.202, 0.292, 1.274, 0.493, 0.453, 0.187, 0.361, 0.078, 1.267, 0.19, 0.005, 0.002, 0.002, 0.011, 0.002, 0.0001, 0.0001, 0.025, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.004, 0.01, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.004, 21.565, 21.383, 0.022, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.029, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "as":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.296, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.811, 0.001, 0.086, 0.0001, 0.0001, 0.005, 0.0001, 0.083, 0.075, 0.077, 0.0001, 0.001, 0.203, 0.086, 0.044, 0.006, 0.008, 0.009, 0.006, 0.004, 0.003, 0.003, 0.002, 0.002, 0.003, 0.004, 0.022, 0.007, 0.002, 0.003, 0.002, 0.001, 0.0001, 0.015, 0.009, 0.013, 0.007, 0.006, 0.005, 0.005, 0.006, 0.011, 0.003, 0.003, 0.005, 0.01, 0.007, 0.004, 0.011, 0.001, 0.008, 0.013, 0.013, 0.003, 0.002, 0.004, 0.0001, 0.001, 0.001, 0.01, 0.0001, 0.01, 0.0001, 0.001, 0.0001, 0.213, 0.031, 0.074, 0.083, 0.255, 0.044, 0.045, 0.095, 0.18, 0.004, 0.017, 0.099, 0.058, 0.166, 0.164, 0.046, 0.002, 0.151, 0.14, 0.179, 0.063, 0.023, 0.027, 0.005, 0.036, 0.003, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.537, 0.769, 0.261, 0.102, 0.001, 0.242, 0.382, 1.586, 0.215, 0.133, 0.002, 0.429, 0.033, 1.928, 0.026, 0.213, 0.004, 0.0001, 0.0001, 0.14, 0.003, 1.299, 0.21, 0.401, 0.056, 0.073, 0.394, 0.328, 0.382, 0.006, 0.051, 0.353, 0.081, 0.128, 0.02, 0.231, 1.75, 0.525, 21.552, 9.182, 1.32, 0.031, 0.846, 0.112, 0.982, 0.29, 0.858, 1.027, 2.855, 0.297, 0.931, 0.0001, 0.0001, 0.0001, 0.293, 0.318, 0.674, 0.559, 0.001, 0.0001, 0.584, 0.0001, 2.717, 1.766, 0.0001, 0.0001, 0.009, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.161, 0.0001, 0.072, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ast": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.724, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.007, 0.002, 0.424, 0.002, 0.001, 0.01, 0.003, 0.548, 0.156, 0.156, 0.002, 0.003, 1.046, 0.096, 0.743, 0.015, 0.245, 0.288, 0.158, 0.086, 0.078, 0.093, 0.076, 0.077, 0.093, 0.166, 0.056, 0.032, 0.002, 0.005, 0.002, 0.002, 0.0001, 0.218, 0.121, 0.236, 0.117, 0.257, 0.089, 0.088, 0.078, 0.115, 0.051, 0.038, 0.23, 0.167, 0.117, 0.051, 0.161, 0.007, 0.094, 0.198, 0.134, 0.043, 0.06, 0.041, 0.061, 0.037, 0.011, 0.014, 0.0001, 0.014, 0.0001, 0.001, 0.0001, 8.074, 0.835, 3.151, 3.345, 9.578, 0.701, 0.803, 0.452, 5.046, 0.025, 0.11, 4.637, 2.087, 5.542, 5.253, 1.877, 0.488, 4.828, 5.384, 3.477, 3.909, 0.672, 0.055, 0.4, 0.967, 0.259, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.04, 0.01, 0.002, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.003, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.01, 0.01, 0.001, 0.0001, 0.001, 0.002, 0.009, 0.001, 0.001, 0.005, 0.006, 0.0001, 0.001, 0.026, 0.531, 0.001, 0.001, 0.002, 0.001, 0.002, 0.002, 0.002, 0.291, 0.001, 0.019, 0.001, 0.46, 0.001, 0.001, 0.005, 0.157, 0.004, 0.608, 0.002, 0.002, 0.003, 0.002, 0.004, 0.002, 0.119, 0.021, 0.027, 0.002, 0.001, 0.003, 0.0001, 0.0001, 0.073, 2.207, 0.003, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.012, 0.005, 0.007, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.039, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "atj": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.34, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.835, 0.0001, 0.034, 0.0001, 0.0001, 0.001, 0.0001, 0.005, 0.045, 0.047, 0.0001, 0.0001, 0.548, 0.045, 1.11, 0.006, 0.039, 0.075, 0.033, 0.013, 0.017, 0.015, 0.02, 0.018, 0.017, 0.061, 0.024, 0.003, 0.015, 0.0001, 0.015, 0.002, 0.0001, 0.175, 0.012, 0.062, 0.025, 0.193, 0.022, 0.01, 0.006, 0.035, 0.021, 0.212, 0.019, 0.332, 0.208, 0.141, 0.099, 0.007, 0.017, 0.034, 0.12, 0.001, 0.003, 0.089, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 11.805, 0.044, 6.264, 0.083, 5.028, 0.008, 0.026, 0.952, 15.443, 0.004, 9.886, 0.134, 2.846, 5.167, 5.337, 2.131, 0.022, 2.079, 2.27, 7.277, 0.131, 0.025, 4.581, 0.005, 0.015, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.009, 0.046, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.015, 0.069, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "av":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.031, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.23, 0.001, 0.083, 0.0001, 0.0001, 0.007, 0.001, 0.001, 0.166, 0.166, 0.001, 0.001, 0.458, 0.25, 0.562, 0.01, 0.133, 0.234, 0.149, 0.084, 0.058, 0.065, 0.053, 0.053, 0.06, 0.094, 0.055, 0.017, 0.001, 0.003, 0.001, 0.003, 0.0001, 0.011, 0.006, 0.01, 0.003, 0.003, 0.003, 0.003, 0.002, 0.777, 0.001, 0.002, 0.002, 0.006, 0.003, 0.003, 0.002, 0.0001, 0.002, 0.007, 0.008, 0.003, 0.006, 0.001, 0.011, 0.001, 0.0001, 0.007, 0.0001, 0.007, 0.0001, 0.009, 0.0001, 0.075, 0.008, 0.02, 0.025, 0.067, 0.007, 0.015, 0.018, 0.067, 0.001, 0.008, 0.038, 0.014, 0.043, 0.038, 0.019, 0.001, 0.041, 0.043, 0.036, 0.031, 0.01, 0.006, 0.003, 0.01, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.671, 1.227, 0.995, 2.675, 0.059, 0.905, 0.851, 0.335, 0.128, 0.084, 1.771, 0.03, 0.884, 0.039, 0.044, 0.818, 0.134, 0.075, 0.027, 0.273, 0.227, 0.015, 0.029, 0.016, 0.039, 0.006, 0.125, 0.043, 0.127, 0.032, 0.014, 0.032, 0.185, 0.089, 0.062, 0.016, 0.021, 0.082, 0.047, 0.033, 0.042, 0.006, 0.002, 0.039, 0.002, 0.019, 0.005, 0.013, 7.089, 1.927, 0.825, 1.964, 1.317, 1.929, 0.263, 0.636, 2.852, 0.187, 1.471, 3.734, 0.878, 1.983, 1.647, 0.208, 0.0001, 0.0001, 0.195, 0.006, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.022, 0.0001, 0.001, 0.0001, 30.778, 12.343, 0.0001, 0.534, 0.0001, 0.002, 0.0001, 0.001, 0.025, 0.022, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.177, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ay":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.588, 0.005, 0.247, 0.0001, 0.0001, 0.0001, 0.027, 1.72, 0.603, 0.602, 0.046, 0.001, 1.21, 0.158, 1.031, 0.021, 0.387, 0.817, 0.515, 0.316, 0.306, 0.36, 0.273, 0.279, 0.341, 0.428, 0.504, 0.129, 0.064, 0.005, 0.064, 0.147, 0.0001, 0.442, 0.126, 0.339, 0.185, 0.072, 0.071, 0.077, 0.1, 0.109, 0.302, 0.254, 0.268, 0.282, 0.145, 0.064, 0.43, 0.127, 0.121, 0.288, 0.2, 0.25, 0.05, 0.191, 0.012, 0.11, 0.013, 0.007, 0.0001, 0.008, 0.0001, 0.002, 0.004, 14.491, 0.243, 1.49, 0.745, 1.57, 0.085, 0.27, 2.104, 6.268, 1.613, 3.058, 2.342, 2.397, 3.14, 1.316, 1.65, 1.821, 3.874, 4.07, 2.906, 5.224, 0.153, 1.248, 0.859, 2.145, 0.119, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.211, 0.009, 0.003, 0.004, 0.002, 0.001, 0.002, 0.002, 0.003, 0.002, 0.001, 0.002, 0.002, 0.003, 0.002, 0.002, 0.004, 0.008, 0.001, 0.016, 0.006, 0.002, 0.001, 0.001, 0.005, 0.126, 0.002, 0.002, 0.008, 0.019, 0.001, 0.001, 0.061, 0.068, 0.001, 0.003, 0.22, 0.002, 0.002, 0.004, 0.004, 0.062, 0.002, 0.003, 0.001, 0.11, 0.003, 0.049, 0.044, 0.259, 0.029, 0.076, 0.026, 0.004, 0.004, 0.007, 0.009, 0.003, 0.038, 0.01, 0.012, 0.003, 0.005, 0.006, 0.0001, 0.0001, 0.133, 0.88, 0.003, 0.004, 0.0001, 0.001, 0.0001, 0.002, 0.001, 0.003, 0.002, 0.0001, 0.006, 0.002, 0.031, 0.01, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.004, 0.004, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.01, 0.003, 0.207, 0.001, 0.004, 0.008, 0.005, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "az":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.803, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.785, 0.003, 0.222, 0.0001, 0.001, 0.009, 0.001, 0.007, 0.139, 0.141, 0.001, 0.002, 0.64, 0.404, 0.91, 0.014, 0.244, 0.339, 0.188, 0.096, 0.09, 0.102, 0.087, 0.087, 0.102, 0.202, 0.038, 0.019, 0.004, 0.002, 0.004, 0.004, 0.0001, 0.276, 0.242, 0.068, 0.094, 0.057, 0.061, 0.057, 0.095, 0.062, 0.008, 0.127, 0.055, 0.202, 0.081, 0.086, 0.077, 0.107, 0.098, 0.172, 0.115, 0.037, 0.055, 0.005, 0.062, 0.066, 0.023, 0.006, 0.0001, 0.006, 0.0001, 0.004, 0.001, 7.007, 1.378, 0.673, 3.497, 1.722, 0.535, 0.389, 0.748, 6.853, 0.041, 1.544, 4.525, 2.336, 5.203, 1.602, 0.396, 1.07, 4.974, 2.444, 2.338, 1.812, 1.06, 0.008, 0.478, 1.947, 0.87, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.147, 0.01, 0.009, 0.005, 0.005, 0.009, 0.003, 0.033, 0.002, 0.001, 0.001, 0.003, 0.002, 0.001, 0.002, 0.082, 0.004, 0.001, 0.002, 0.028, 0.04, 0.001, 0.012, 0.001, 0.002, 6.259, 0.001, 0.001, 0.046, 0.034, 0.075, 1.454, 0.026, 0.003, 0.003, 0.001, 0.001, 0.001, 0.001, 0.485, 0.001, 0.001, 0.001, 0.011, 0.002, 0.016, 0.001, 0.001, 0.187, 2.533, 0.009, 0.004, 0.005, 0.028, 0.457, 0.003, 0.014, 0.003, 0.01, 0.017, 1.158, 0.011, 0.03, 0.004, 0.0001, 0.0001, 0.067, 2.145, 2.985, 1.196, 0.079, 0.0001, 0.0001, 6.24, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.207, 0.052, 0.0001, 0.018, 0.0001, 0.0001, 0.0001, 0.001, 0.008, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.14, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "azb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.225, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.112, 0.002, 0.032, 0.0001, 0.0001, 0.003, 0.0001, 0.002, 0.275, 0.275, 0.002, 0.001, 0.028, 0.165, 0.744, 0.053, 0.037, 0.078, 0.041, 0.038, 0.027, 0.033, 0.024, 0.023, 0.03, 0.03, 0.059, 0.003, 0.004, 0.001, 0.003, 0.0001, 0.0001, 0.005, 0.004, 0.007, 0.004, 0.002, 0.002, 0.002, 0.003, 0.008, 0.002, 0.002, 0.004, 0.004, 0.003, 0.001, 0.007, 0.001, 0.004, 0.011, 0.002, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.005, 0.0001, 0.005, 0.0001, 0.022, 0.0001, 0.096, 0.009, 0.017, 0.038, 0.09, 0.012, 0.02, 0.043, 0.1, 0.0001, 0.026, 0.053, 0.017, 0.052, 0.064, 0.04, 0.001, 0.055, 0.055, 0.106, 0.015, 0.003, 0.052, 0.004, 0.018, 0.009, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.77, 0.455, 0.528, 0.028, 2.648, 1.417, 3.922, 1.536, 3.205, 0.004, 0.23, 0.004, 7.975, 0.001, 0.011, 0.01, 0.002, 0.06, 0.27, 0.013, 0.004, 0.001, 0.0001, 0.0001, 0.033, 0.002, 0.0001, 0.023, 0.001, 0.001, 0.0001, 0.002, 0.02, 0.007, 0.378, 0.004, 0.281, 0.002, 0.413, 5.027, 1.244, 0.85, 1.199, 0.132, 0.444, 0.158, 0.386, 2.668, 0.253, 3.47, 0.613, 1.73, 0.767, 0.17, 0.092, 0.269, 0.09, 0.326, 0.153, 0.08, 0.001, 0.001, 0.271, 0.002, 0.0001, 0.0001, 0.181, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 18.661, 14.13, 1.511, 8.604, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.0001, 0.763, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ba":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.692, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.933, 0.002, 0.044, 0.0001, 0.0001, 0.005, 0.0001, 0.001, 0.147, 0.147, 0.0001, 0.004, 0.482, 0.143, 0.604, 0.015, 0.158, 0.244, 0.135, 0.077, 0.08, 0.076, 0.061, 0.06, 0.081, 0.125, 0.052, 0.011, 0.008, 0.003, 0.008, 0.001, 0.0001, 0.003, 0.003, 0.006, 0.002, 0.002, 0.001, 0.002, 0.002, 0.025, 0.001, 0.002, 0.002, 0.003, 0.002, 0.001, 0.002, 0.0001, 0.001, 0.004, 0.005, 0.004, 0.007, 0.001, 0.012, 0.0001, 0.001, 0.006, 0.0001, 0.006, 0.0001, 0.002, 0.0001, 0.021, 0.003, 0.012, 0.011, 0.026, 0.004, 0.004, 0.006, 0.021, 0.001, 0.003, 0.02, 0.007, 0.023, 0.02, 0.005, 0.0001, 0.016, 0.01, 0.014, 0.014, 0.002, 0.003, 0.001, 0.009, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.739, 1.424, 2.096, 1.348, 0.183, 0.244, 0.115, 0.088, 0.621, 0.006, 0.016, 3.259, 0.202, 0.093, 0.068, 0.404, 0.112, 0.175, 0.076, 1.0, 0.273, 0.018, 0.005, 0.012, 0.081, 3.093, 0.13, 0.026, 0.084, 0.041, 0.082, 0.063, 0.299, 0.879, 0.098, 0.434, 0.038, 0.036, 0.005, 0.017, 0.043, 0.504, 0.0001, 0.196, 0.001, 0.016, 0.036, 0.445, 4.844, 0.952, 0.303, 0.533, 0.952, 2.488, 0.102, 0.15, 1.49, 1.18, 1.231, 3.558, 1.237, 2.847, 1.277, 0.365, 0.0001, 0.0001, 0.244, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.004, 0.0001, 0.002, 0.001, 24.156, 12.667, 4.154, 3.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.235, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bar": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.604, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.871, 0.004, 0.418, 0.0001, 0.0001, 0.008, 0.002, 0.216, 0.21, 0.21, 0.009, 0.001, 0.803, 0.202, 1.146, 0.023, 0.266, 0.394, 0.199, 0.121, 0.109, 0.119, 0.109, 0.117, 0.138, 0.187, 0.117, 0.02, 0.004, 0.005, 0.004, 0.003, 0.0001, 0.352, 0.447, 0.201, 0.532, 0.247, 0.245, 0.332, 0.228, 0.204, 0.156, 0.293, 0.235, 0.338, 0.204, 0.224, 0.214, 0.034, 0.205, 0.697, 0.181, 0.119, 0.18, 0.276, 0.005, 0.01, 0.114, 0.021, 0.0001, 0.021, 0.0001, 0.003, 0.003, 8.177, 1.169, 1.993, 4.065, 6.625, 1.095, 2.102, 3.003, 6.12, 0.162, 0.941, 2.0, 2.327, 6.606, 4.578, 0.55, 0.014, 3.249, 4.677, 4.042, 3.018, 0.854, 1.171, 0.071, 0.239, 0.864, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.102, 0.003, 0.003, 0.002, 0.004, 0.004, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.014, 0.001, 0.001, 0.016, 0.001, 0.002, 0.009, 0.001, 0.001, 0.039, 0.001, 0.036, 0.116, 0.061, 0.007, 0.003, 0.001, 0.274, 0.073, 0.002, 0.002, 0.004, 0.027, 0.002, 0.002, 0.002, 0.004, 0.001, 0.001, 0.004, 0.002, 0.01, 0.016, 0.006, 0.001, 0.154, 0.002, 0.005, 0.001, 0.002, 0.002, 0.176, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.07, 0.891, 0.007, 0.006, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.007, 0.004, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.103, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bcl": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.379, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.071, 0.002, 0.217, 0.001, 0.003, 0.005, 0.002, 0.116, 0.161, 0.16, 0.0001, 0.001, 0.914, 0.25, 0.911, 0.022, 0.337, 0.439, 0.274, 0.132, 0.116, 0.128, 0.121, 0.133, 0.144, 0.229, 0.055, 0.02, 0.017, 0.001, 0.017, 0.022, 0.0001, 0.585, 0.233, 0.246, 0.128, 0.11, 0.148, 0.111, 0.118, 0.238, 0.077, 0.175, 0.149, 0.27, 0.198, 0.07, 0.296, 0.013, 0.12, 0.508, 0.14, 0.057, 0.048, 0.04, 0.004, 0.02, 0.015, 0.025, 0.0001, 0.025, 0.0001, 0.0001, 0.0001, 15.454, 1.486, 0.494, 1.897, 2.968, 0.126, 4.169, 0.861, 6.432, 0.033, 2.688, 2.392, 2.068, 10.392, 5.039, 1.872, 0.022, 3.21, 4.66, 2.796, 1.875, 0.174, 0.643, 0.021, 1.752, 0.121, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.039, 0.006, 0.003, 0.003, 0.005, 0.004, 0.002, 0.003, 0.009, 0.002, 0.004, 0.002, 0.003, 0.004, 0.003, 0.002, 0.007, 0.003, 0.002, 0.009, 0.004, 0.002, 0.001, 0.002, 0.002, 0.008, 0.004, 0.003, 0.013, 0.011, 0.003, 0.001, 0.027, 0.035, 0.013, 0.004, 0.005, 0.003, 0.003, 0.006, 0.004, 0.006, 0.004, 0.003, 0.007, 0.019, 0.005, 0.003, 0.005, 0.018, 0.01, 0.022, 0.014, 0.003, 0.004, 0.003, 0.01, 0.004, 0.006, 0.004, 0.005, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.019, 0.136, 0.005, 0.006, 0.0001, 0.0001, 0.0001, 0.011, 0.004, 0.01, 0.002, 0.0001, 0.006, 0.003, 0.016, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.017, 0.012, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.007, 0.034, 0.001, 0.008, 0.01, 0.006, 0.004, 0.002, 0.003, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "be":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.607, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.35, 0.001, 0.055, 0.0001, 0.0001, 0.006, 0.0001, 0.05, 0.155, 0.156, 0.001, 0.002, 0.628, 0.121, 0.612, 0.009, 0.188, 0.295, 0.148, 0.088, 0.085, 0.087, 0.076, 0.074, 0.089, 0.156, 0.032, 0.017, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.009, 0.006, 0.026, 0.004, 0.005, 0.003, 0.019, 0.003, 0.047, 0.001, 0.002, 0.004, 0.009, 0.01, 0.004, 0.01, 0.0001, 0.005, 0.013, 0.005, 0.003, 0.013, 0.004, 0.018, 0.001, 0.002, 0.002, 0.0001, 0.002, 0.0001, 0.003, 0.0001, 0.046, 0.006, 0.014, 0.013, 0.042, 0.007, 0.007, 0.01, 0.04, 0.001, 0.006, 0.023, 0.014, 0.029, 0.035, 0.009, 0.001, 0.032, 0.024, 0.024, 0.019, 0.004, 0.003, 0.002, 0.006, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.314, 1.922, 1.481, 1.13, 0.14, 0.481, 1.007, 0.569, 0.351, 0.001, 0.001, 1.93, 0.479, 0.541, 0.221, 1.357, 0.128, 0.261, 0.085, 0.08, 0.203, 0.012, 2.438, 0.059, 0.001, 0.01, 0.103, 0.048, 0.097, 0.076, 0.995, 0.141, 0.181, 0.137, 0.046, 0.12, 0.029, 0.02, 0.016, 0.019, 0.023, 0.001, 0.0001, 0.081, 0.0001, 0.017, 0.007, 0.023, 7.12, 0.583, 1.325, 0.884, 1.382, 1.613, 0.241, 1.022, 0.011, 0.528, 1.726, 1.757, 1.251, 2.924, 1.397, 1.062, 0.0001, 0.0001, 0.283, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.021, 0.0001, 0.002, 0.001, 26.294, 17.28, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.156, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bh":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.941, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.272, 0.0001, 0.067, 0.0001, 0.001, 0.014, 0.0001, 0.006, 0.074, 0.074, 0.0001, 0.001, 0.205, 0.047, 0.036, 0.005, 0.139, 0.215, 0.134, 0.072, 0.07, 0.074, 0.065, 0.069, 0.075, 0.087, 0.017, 0.007, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.006, 0.004, 0.005, 0.002, 0.003, 0.002, 0.004, 0.002, 0.007, 0.001, 0.002, 0.003, 0.003, 0.003, 0.002, 0.004, 0.0001, 0.002, 0.006, 0.006, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.009, 0.0001, 0.1, 0.014, 0.029, 0.038, 0.115, 0.019, 0.024, 0.049, 0.081, 0.001, 0.007, 0.043, 0.023, 0.079, 0.071, 0.019, 0.001, 0.072, 0.065, 0.081, 0.029, 0.011, 0.014, 0.002, 0.014, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.902, 0.534, 1.035, 0.031, 0.0001, 0.22, 0.29, 2.243, 0.258, 0.137, 0.021, 0.553, 0.066, 1.318, 0.0001, 0.336, 0.009, 0.009, 0.0001, 0.03, 0.023, 1.891, 0.248, 0.639, 0.037, 0.011, 0.202, 0.05, 0.683, 0.024, 0.014, 0.375, 0.074, 0.252, 0.031, 0.13, 24.792, 6.19, 0.487, 0.175, 1.097, 0.001, 0.677, 0.098, 0.808, 0.311, 0.975, 0.521, 2.028, 0.0001, 1.424, 0.0001, 0.0001, 0.605, 0.237, 0.107, 1.177, 0.742, 0.0001, 0.0001, 0.117, 0.003, 3.031, 1.138, 0.0001, 0.0001, 0.016, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 29.692, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.859, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.276, 0.003, 0.256, 0.0001, 0.0001, 0.003, 0.003, 0.016, 0.486, 0.484, 0.001, 0.0001, 0.638, 0.156, 1.372, 0.022, 0.455, 0.969, 0.456, 0.237, 0.231, 0.247, 0.248, 0.25, 0.297, 0.612, 0.044, 0.019, 0.005, 0.0001, 0.004, 0.004, 0.0001, 0.449, 0.264, 0.227, 0.165, 0.234, 0.192, 0.164, 0.234, 0.179, 0.456, 0.316, 0.231, 0.458, 0.197, 0.135, 0.315, 0.005, 0.168, 0.606, 0.235, 0.049, 0.123, 0.109, 0.008, 0.231, 0.017, 0.005, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 8.019, 2.445, 0.575, 1.178, 6.318, 0.449, 2.782, 1.275, 5.992, 0.203, 1.688, 4.658, 3.419, 6.494, 6.015, 1.447, 0.023, 2.565, 2.973, 3.583, 1.992, 0.459, 0.92, 0.044, 0.557, 0.136, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.108, 0.019, 0.014, 0.005, 0.005, 0.004, 0.006, 0.01, 0.005, 0.008, 0.002, 0.002, 0.012, 0.031, 0.002, 0.001, 0.002, 0.004, 0.003, 0.089, 0.007, 0.003, 0.003, 0.004, 0.004, 0.002, 0.001, 0.001, 0.007, 0.004, 0.002, 0.004, 0.052, 0.019, 0.003, 0.005, 0.023, 0.009, 0.014, 0.014, 0.008, 0.023, 0.003, 0.01, 0.005, 0.015, 0.003, 0.004, 0.019, 0.013, 0.011, 0.022, 0.006, 0.01, 0.007, 0.004, 0.018, 0.01, 0.009, 0.009, 0.011, 0.009, 0.011, 0.009, 0.0001, 0.0001, 0.048, 0.113, 0.02, 0.046, 0.0001, 0.002, 0.0001, 0.005, 0.001, 0.002, 0.0001, 0.001, 0.032, 0.011, 0.078, 0.027, 0.001, 0.0001, 0.001, 0.018, 0.002, 0.0001, 0.017, 0.009, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.037, 0.005, 0.097, 0.0001, 0.0001, 0.007, 0.003, 0.001, 0.003, 0.001, 0.002, 0.001, 0.006, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bjn": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.274, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.352, 0.002, 0.406, 0.0001, 0.001, 0.013, 0.001, 0.109, 0.199, 0.198, 0.002, 0.001, 0.988, 0.406, 0.819, 0.036, 0.185, 0.196, 0.136, 0.076, 0.062, 0.071, 0.054, 0.058, 0.057, 0.091, 0.102, 0.025, 0.002, 0.003, 0.002, 0.005, 0.0001, 0.244, 0.391, 0.098, 0.173, 0.034, 0.031, 0.106, 0.136, 0.207, 0.121, 0.411, 0.116, 0.312, 0.12, 0.035, 0.341, 0.003, 0.133, 0.409, 0.258, 0.061, 0.026, 0.09, 0.002, 0.038, 0.007, 0.012, 0.0001, 0.012, 0.0001, 0.0001, 0.0001, 19.717, 2.113, 0.418, 2.814, 2.089, 0.126, 3.097, 2.135, 6.446, 0.654, 2.733, 2.879, 2.871, 8.542, 1.048, 1.844, 0.007, 3.384, 2.985, 3.613, 4.514, 0.083, 0.972, 0.009, 1.107, 0.035, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.03, 0.008, 0.005, 0.007, 0.006, 0.006, 0.006, 0.004, 0.006, 0.004, 0.008, 0.003, 0.003, 0.007, 0.001, 0.001, 0.002, 0.002, 0.002, 0.008, 0.003, 0.002, 0.002, 0.004, 0.002, 0.014, 0.001, 0.002, 0.005, 0.005, 0.002, 0.002, 0.012, 0.002, 0.002, 0.004, 0.012, 0.005, 0.004, 0.011, 0.007, 0.182, 0.006, 0.005, 0.004, 0.004, 0.003, 0.005, 0.009, 0.008, 0.005, 0.005, 0.003, 0.002, 0.001, 0.003, 0.006, 0.004, 0.004, 0.003, 0.003, 0.003, 0.002, 0.002, 0.0001, 0.0001, 0.019, 0.193, 0.007, 0.009, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.005, 0.002, 0.005, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.035, 0.03, 0.004, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.019, 0.008, 0.026, 0.006, 0.003, 0.008, 0.005, 0.003, 0.002, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bm":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.129, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.167, 0.007, 0.144, 0.0001, 0.001, 0.013, 0.002, 0.256, 0.237, 0.237, 0.007, 0.003, 0.973, 0.158, 0.97, 0.007, 0.243, 0.224, 0.128, 0.052, 0.064, 0.06, 0.072, 0.055, 0.07, 0.12, 0.287, 0.015, 0.0001, 0.01, 0.0001, 0.005, 0.0001, 0.444, 0.348, 0.111, 0.212, 0.105, 0.277, 0.105, 0.044, 0.094, 0.171, 0.429, 0.132, 0.368, 0.21, 0.091, 0.065, 0.003, 0.072, 0.446, 0.184, 0.079, 0.027, 0.078, 0.004, 0.046, 0.018, 0.018, 0.0001, 0.014, 0.0001, 0.017, 0.0001, 12.037, 2.27, 0.406, 1.816, 3.589, 1.305, 1.615, 0.299, 5.301, 0.672, 3.384, 3.18, 2.268, 7.22, 3.282, 0.194, 0.029, 2.428, 2.045, 1.645, 2.796, 0.059, 0.96, 0.016, 1.69, 0.107, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.237, 0.003, 0.001, 0.017, 0.017, 0.007, 0.015, 0.003, 0.008, 0.011, 0.026, 0.017, 0.001, 0.0001, 0.018, 0.005, 0.013, 0.002, 0.004, 0.018, 1.999, 0.0001, 0.0001, 0.0001, 0.002, 0.172, 0.0001, 1.879, 0.012, 0.017, 0.004, 0.0001, 0.054, 0.002, 0.001, 0.001, 0.002, 0.003, 0.005, 0.027, 0.322, 0.21, 0.005, 0.017, 0.007, 0.002, 0.001, 0.011, 0.002, 0.012, 0.238, 0.014, 0.415, 0.435, 0.001, 0.007, 0.005, 0.009, 0.01, 0.017, 0.003, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.064, 1.039, 0.002, 0.033, 0.027, 0.0001, 0.0001, 4.089, 0.016, 0.002, 0.003, 0.0001, 0.433, 0.0001, 0.024, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.065, 0.05, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.015, 0.0001, 0.003, 0.233, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.319, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.406, 0.001, 0.076, 0.0001, 0.0001, 0.012, 0.0001, 0.015, 0.057, 0.058, 0.0001, 0.001, 0.196, 0.086, 0.029, 0.005, 0.005, 0.006, 0.004, 0.002, 0.002, 0.002, 0.002, 0.001, 0.002, 0.002, 0.016, 0.009, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.005, 0.003, 0.004, 0.002, 0.002, 0.002, 0.002, 0.002, 0.003, 0.002, 0.001, 0.002, 0.003, 0.002, 0.002, 0.003, 0.0001, 0.002, 0.004, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.004, 0.001, 0.001, 0.0001, 0.043, 0.007, 0.016, 0.016, 0.05, 0.009, 0.009, 0.017, 0.038, 0.001, 0.004, 0.022, 0.013, 0.034, 0.034, 0.01, 0.001, 0.031, 0.027, 0.033, 0.016, 0.005, 0.005, 0.002, 0.008, 0.001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.359, 0.551, 0.299, 0.082, 0.002, 0.229, 0.186, 2.436, 0.034, 0.152, 0.002, 0.333, 0.036, 2.245, 0.026, 0.384, 0.008, 0.001, 0.001, 0.181, 0.002, 1.31, 0.16, 0.34, 0.043, 0.053, 0.26, 0.209, 0.4, 0.015, 0.042, 0.46, 0.067, 0.212, 0.008, 0.16, 1.542, 0.621, 24.834, 6.808, 1.602, 0.04, 0.792, 0.149, 1.148, 0.261, 0.867, 1.261, 2.631, 0.001, 0.874, 0.001, 0.001, 0.001, 0.381, 0.232, 0.963, 0.451, 0.001, 0.001, 0.701, 0.0001, 2.837, 1.811, 0.0001, 0.0001, 0.013, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.991, 0.0001, 0.03, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.169, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.69, 0.0001, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.01, 0.01, 0.0001, 0.0001, 0.002, 0.003, 0.005, 0.001, 0.003, 0.004, 0.003, 0.002, 0.001, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.012, 0.002, 0.004, 0.004, 0.015, 0.003, 0.003, 0.006, 0.011, 0.0001, 0.001, 0.005, 0.003, 0.01, 0.01, 0.003, 0.0001, 0.008, 0.008, 0.01, 0.004, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.3, 0.21, 1.61, 0.004, 1.096, 0.171, 0.232, 0.056, 0.006, 0.125, 0.009, 7.85, 0.044, 0.821, 0.01, 0.147, 0.305, 1.571, 0.233, 1.086, 0.826, 0.17, 1.379, 0.052, 0.974, 0.101, 0.175, 0.065, 0.005, 0.008, 0.253, 0.318, 0.893, 0.39, 1.207, 0.915, 0.217, 0.014, 2.41, 0.028, 0.071, 0.06, 0.002, 0.023, 0.001, 0.018, 0.001, 0.001, 0.003, 0.913, 2.028, 0.112, 1.086, 0.005, 0.001, 0.055, 0.005, 0.003, 0.951, 0.005, 10.217, 21.49, 2.602, 0.016, 0.0001, 0.0001, 0.014, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 32.905, 0.0001, 0.024, 0.009, 0.002, 0.006, 0.004, 0.005, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bpy": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.902, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.282, 0.0001, 0.009, 0.0001, 0.0001, 0.224, 0.0001, 0.002, 0.281, 0.281, 0.0001, 0.0001, 0.306, 0.253, 0.183, 0.08, 0.005, 0.009, 0.002, 0.004, 0.002, 0.003, 0.003, 0.003, 0.003, 0.003, 0.197, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.016, 0.008, 0.017, 0.005, 0.005, 0.002, 0.004, 0.002, 0.003, 0.003, 0.005, 0.003, 0.007, 0.007, 0.001, 0.007, 0.0001, 0.004, 0.019, 0.004, 0.016, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.014, 0.0001, 0.118, 0.01, 0.016, 0.026, 0.05, 0.006, 0.015, 0.031, 0.057, 0.004, 0.009, 0.031, 0.017, 0.064, 0.06, 0.015, 0.001, 0.059, 0.03, 0.047, 0.04, 0.005, 0.005, 0.001, 0.018, 0.002, 0.0001, 0.016, 0.0001, 0.0001, 0.0001, 0.094, 0.582, 0.295, 0.004, 0.001, 0.199, 0.278, 1.651, 0.006, 0.325, 0.001, 0.49, 0.119, 1.057, 0.003, 0.285, 0.0001, 0.0001, 0.0001, 0.034, 0.032, 0.592, 0.143, 0.798, 0.084, 0.129, 0.075, 0.036, 0.484, 0.004, 0.03, 0.329, 0.051, 0.128, 0.007, 0.019, 1.405, 0.659, 24.309, 6.387, 2.166, 0.231, 0.814, 0.355, 0.961, 0.379, 1.131, 0.99, 2.941, 0.034, 0.919, 0.004, 0.001, 0.001, 0.243, 0.193, 0.791, 1.05, 0.0001, 0.0001, 0.626, 0.0001, 4.392, 1.335, 0.0001, 0.0001, 0.04, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.31, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "br":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.678, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.255, 0.004, 0.515, 0.0001, 0.0001, 0.007, 0.002, 0.663, 0.246, 0.246, 0.001, 0.002, 0.881, 0.746, 0.901, 0.014, 0.258, 0.444, 0.187, 0.109, 0.115, 0.122, 0.109, 0.12, 0.152, 0.228, 0.115, 0.024, 0.015, 0.004, 0.016, 0.003, 0.0001, 0.347, 0.279, 0.201, 0.205, 0.261, 0.098, 0.212, 0.134, 0.164, 0.075, 0.201, 0.168, 0.253, 0.109, 0.059, 0.199, 0.006, 0.146, 0.289, 0.136, 0.097, 0.091, 0.051, 0.019, 0.032, 0.015, 0.024, 0.0001, 0.024, 0.0001, 0.001, 0.0001, 9.146, 1.127, 0.833, 2.777, 10.42, 0.294, 1.799, 2.456, 3.655, 0.167, 1.352, 2.97, 1.505, 5.492, 4.696, 0.867, 0.019, 5.665, 2.33, 3.448, 2.744, 1.784, 0.434, 0.03, 0.247, 2.302, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 0.1, 0.012, 0.008, 0.007, 0.005, 0.004, 0.003, 0.003, 0.004, 0.005, 0.002, 0.002, 0.004, 0.005, 0.003, 0.002, 0.003, 0.002, 0.002, 0.011, 0.005, 0.002, 0.002, 0.002, 0.002, 0.074, 0.002, 0.003, 0.005, 0.005, 0.001, 0.004, 0.021, 0.015, 0.009, 0.005, 0.007, 0.003, 0.004, 0.009, 0.013, 0.045, 0.076, 0.018, 0.003, 0.013, 0.003, 0.005, 0.011, 0.591, 0.009, 0.012, 0.018, 0.007, 0.006, 0.004, 0.009, 0.467, 0.008, 0.021, 0.017, 0.008, 0.005, 0.006, 0.0001, 0.0001, 0.048, 1.28, 0.01, 0.011, 0.0001, 0.001, 0.0001, 0.004, 0.002, 0.002, 0.002, 0.0001, 0.032, 0.015, 0.039, 0.015, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.006, 0.009, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.009, 0.096, 0.003, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bs":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.108, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.139, 0.002, 0.313, 0.001, 0.001, 0.017, 0.002, 0.011, 0.204, 0.204, 0.001, 0.006, 0.915, 0.157, 1.176, 0.034, 0.332, 0.467, 0.264, 0.159, 0.151, 0.151, 0.132, 0.126, 0.142, 0.226, 0.068, 0.015, 0.006, 0.007, 0.006, 0.001, 0.0001, 0.156, 0.174, 0.174, 0.143, 0.072, 0.074, 0.155, 0.136, 0.152, 0.073, 0.147, 0.082, 0.163, 0.218, 0.118, 0.225, 0.003, 0.11, 0.283, 0.122, 0.105, 0.088, 0.031, 0.007, 0.007, 0.073, 0.025, 0.0001, 0.025, 0.0001, 0.008, 0.0001, 8.723, 0.95, 0.762, 2.331, 6.777, 0.26, 1.369, 0.582, 7.412, 3.867, 2.673, 2.682, 2.205, 4.994, 6.632, 1.941, 0.005, 3.955, 3.612, 3.234, 3.103, 2.415, 0.036, 0.017, 0.061, 1.207, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.038, 0.004, 0.003, 0.002, 0.002, 0.001, 0.003, 0.388, 0.002, 0.001, 0.001, 0.001, 0.016, 0.618, 0.001, 0.0001, 0.003, 0.172, 0.002, 0.018, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.003, 0.001, 0.001, 0.006, 0.003, 0.004, 0.002, 0.035, 0.482, 0.001, 0.001, 0.003, 0.001, 0.001, 0.002, 0.001, 0.008, 0.001, 0.002, 0.001, 0.003, 0.001, 0.001, 0.007, 0.004, 0.003, 0.004, 0.002, 0.003, 0.004, 0.002, 0.004, 0.002, 0.002, 0.003, 0.006, 0.012, 0.366, 0.002, 0.0001, 0.0001, 0.02, 0.032, 1.199, 0.874, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.014, 0.006, 0.021, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bug": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.068, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.164, 0.0001, 0.016, 0.0001, 0.0001, 0.003, 0.001, 0.137, 0.016, 0.016, 0.0001, 0.001, 0.196, 1.935, 1.044, 0.004, 0.035, 0.02, 0.023, 0.01, 0.009, 0.007, 0.007, 0.006, 0.007, 0.013, 0.007, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.516, 0.311, 0.434, 0.185, 0.139, 0.134, 0.304, 0.324, 0.039, 0.055, 0.029, 0.369, 0.412, 0.063, 0.111, 1.316, 0.017, 0.157, 0.558, 0.13, 0.016, 0.233, 0.012, 0.002, 0.073, 0.002, 0.007, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 9.887, 0.241, 1.633, 1.832, 7.179, 0.088, 0.757, 0.513, 7.161, 0.111, 1.126, 1.683, 2.724, 6.291, 2.861, 1.308, 0.04, 7.537, 3.873, 3.7, 4.723, 0.375, 1.036, 0.149, 1.531, 0.172, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.047, 0.009, 0.005, 0.004, 0.009, 0.007, 0.006, 0.004, 0.009, 0.039, 0.01, 0.038, 0.003, 0.005, 0.002, 0.001, 0.004, 0.012, 0.007, 0.011, 0.011, 0.02, 0.001, 0.02, 0.012, 0.019, 0.011, 0.012, 0.002, 0.001, 0.006, 0.003, 0.004, 0.003, 0.047, 0.002, 0.016, 0.005, 0.004, 0.01, 0.405, 2.36, 0.01, 0.013, 0.003, 0.001, 0.008, 0.004, 0.008, 0.004, 0.008, 0.005, 0.176, 0.005, 0.002, 0.003, 0.012, 0.005, 0.008, 0.007, 0.003, 0.003, 0.004, 0.003, 0.0001, 0.0001, 0.007, 2.887, 0.002, 0.04, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.003, 0.023, 0.014, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.008, 0.007, 0.001, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.048, 0.15, 0.04, 0.0001, 0.0001, 0.002, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "bxr": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.49, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.129, 0.001, 0.08, 0.0001, 0.0001, 0.012, 0.0001, 0.001, 0.147, 0.147, 0.0001, 0.002, 0.553, 0.131, 0.523, 0.004, 0.151, 0.243, 0.109, 0.074, 0.068, 0.074, 0.065, 0.062, 0.079, 0.12, 0.022, 0.018, 0.003, 0.001, 0.002, 0.001, 0.0001, 0.004, 0.002, 0.007, 0.001, 0.002, 0.002, 0.002, 0.004, 0.037, 0.001, 0.001, 0.002, 0.003, 0.003, 0.003, 0.003, 0.0001, 0.002, 0.004, 0.003, 0.001, 0.011, 0.001, 0.019, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.037, 0.005, 0.011, 0.009, 0.029, 0.005, 0.007, 0.031, 0.027, 0.001, 0.005, 0.019, 0.012, 0.022, 0.025, 0.008, 0.001, 0.023, 0.018, 0.017, 0.016, 0.003, 0.002, 0.001, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.392, 0.859, 1.489, 1.628, 0.046, 1.574, 0.057, 0.037, 0.549, 0.002, 0.003, 0.546, 0.265, 4.264, 0.148, 0.174, 0.118, 0.207, 0.029, 0.069, 0.123, 0.028, 0.013, 0.033, 0.034, 0.005, 0.055, 0.03, 0.09, 0.073, 0.049, 0.037, 0.094, 0.079, 0.088, 0.076, 0.026, 0.12, 0.011, 0.016, 0.032, 0.306, 0.001, 0.058, 0.001, 0.071, 0.033, 1.461, 5.842, 1.346, 0.152, 2.003, 2.072, 0.704, 0.52, 0.475, 1.576, 1.562, 0.254, 3.078, 0.893, 3.534, 3.045, 0.105, 0.0001, 0.0001, 0.188, 0.005, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.006, 0.002, 27.741, 14.028, 2.178, 0.307, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.003, 0.075, 0.002, 0.001, 0.004, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cdo": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.899, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.597, 0.001, 0.273, 0.0001, 0.0001, 0.004, 0.0001, 0.004, 0.549, 0.551, 0.0001, 0.001, 0.624, 3.929, 0.732, 0.03, 0.251, 0.611, 0.29, 0.189, 0.163, 0.163, 0.16, 0.156, 0.166, 0.215, 0.133, 0.012, 0.001, 0.0001, 0.001, 0.002, 0.0001, 0.053, 0.117, 0.299, 0.251, 0.017, 0.027, 0.504, 0.23, 0.082, 0.03, 0.071, 0.135, 0.356, 0.159, 0.039, 0.068, 0.004, 0.027, 0.229, 0.101, 0.044, 0.025, 0.062, 0.001, 0.013, 0.003, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.822, 0.392, 1.504, 1.05, 0.748, 0.033, 6.691, 1.959, 3.832, 0.006, 1.877, 0.724, 0.396, 5.597, 0.623, 0.123, 0.005, 0.411, 2.143, 0.557, 2.118, 0.037, 0.065, 0.039, 0.184, 0.014, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.562, 0.653, 0.229, 0.604, 0.418, 0.298, 0.318, 0.129, 0.175, 0.171, 0.118, 0.212, 0.31, 0.409, 0.113, 0.98, 0.125, 0.066, 0.036, 0.255, 0.106, 0.397, 0.142, 0.124, 0.138, 0.172, 0.096, 0.139, 0.338, 0.116, 0.144, 0.186, 0.41, 1.078, 0.77, 0.114, 1.515, 0.081, 0.097, 0.077, 0.628, 0.714, 1.044, 0.603, 1.183, 1.024, 0.119, 0.129, 0.135, 0.183, 0.537, 1.615, 1.19, 0.067, 0.211, 0.1, 0.216, 1.217, 0.179, 0.199, 0.306, 0.119, 0.135, 0.091, 0.0001, 0.0001, 0.041, 7.531, 2.472, 1.618, 0.001, 0.001, 0.0001, 0.002, 0.002, 0.001, 2.018, 0.0001, 0.014, 0.006, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.757, 0.108, 0.212, 0.359, 1.361, 0.793, 0.503, 0.549, 0.397, 0.002, 0.003, 0.004, 0.001, 0.0001, 0.218, 0.03, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ce":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.477, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.593, 0.0001, 0.003, 0.0001, 0.0001, 0.014, 0.0001, 0.0001, 0.462, 0.462, 0.0001, 0.166, 0.461, 0.186, 0.813, 0.002, 0.175, 0.094, 0.109, 0.14, 0.045, 0.029, 0.022, 0.02, 0.031, 0.028, 0.033, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.145, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.004, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.145, 0.144, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.704, 1.438, 1.762, 1.875, 0.015, 2.329, 0.449, 0.169, 0.835, 0.009, 0.342, 0.05, 1.751, 0.164, 0.611, 0.21, 0.068, 0.113, 0.056, 0.04, 0.434, 0.02, 0.006, 0.019, 0.028, 0.002, 0.404, 0.034, 0.196, 0.056, 0.049, 0.075, 0.184, 0.229, 0.057, 0.026, 0.146, 0.02, 0.017, 0.02, 0.129, 0.002, 0.0001, 0.004, 0.0001, 0.008, 0.009, 0.018, 7.603, 0.877, 1.017, 0.93, 0.629, 1.84, 0.05, 0.386, 1.788, 1.009, 1.778, 2.253, 0.873, 3.199, 2.291, 0.075, 0.0001, 0.0001, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 28.632, 13.675, 0.0001, 0.638, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.405, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ceb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.228, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.341, 0.0001, 0.15, 0.0001, 0.0001, 0.002, 0.0001, 0.016, 0.068, 0.068, 0.0001, 0.0001, 1.15, 0.441, 1.259, 0.001, 0.028, 0.059, 0.035, 0.022, 0.021, 0.022, 0.021, 0.021, 0.026, 0.036, 0.037, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.098, 0.168, 0.578, 0.161, 0.203, 0.063, 0.093, 0.198, 0.052, 0.044, 0.126, 0.151, 0.236, 0.118, 0.082, 0.261, 0.02, 0.131, 0.295, 0.118, 0.081, 0.041, 0.087, 0.005, 0.015, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 15.378, 2.318, 0.367, 1.953, 2.974, 0.093, 5.126, 1.479, 4.851, 0.069, 2.449, 3.4, 2.839, 8.407, 4.701, 1.442, 0.019, 2.43, 4.783, 3.214, 2.941, 0.169, 0.623, 0.03, 1.539, 0.068, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.059, 0.004, 0.005, 0.004, 0.004, 0.003, 0.008, 0.005, 0.003, 0.002, 0.001, 0.004, 0.009, 0.002, 0.004, 0.002, 0.003, 0.0001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.01, 0.005, 0.001, 0.002, 0.001, 0.001, 0.002, 0.006, 0.184, 0.019, 0.007, 0.005, 0.008, 0.019, 0.003, 0.009, 0.007, 0.025, 0.004, 0.049, 0.001, 0.018, 0.002, 0.008, 0.279, 0.015, 0.004, 0.013, 0.004, 0.003, 0.007, 0.001, 0.046, 0.006, 0.007, 0.005, 0.006, 0.004, 0.006, 0.001, 0.0001, 0.0001, 0.452, 0.166, 0.097, 0.047, 0.001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.031, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.019, 0.018, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.017, 0.012, 0.008, 0.002, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ch":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.587, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.467, 0.008, 0.286, 0.0001, 0.0001, 0.018, 0.0001, 1.077, 0.189, 0.189, 0.0001, 0.0001, 1.14, 0.532, 1.257, 0.007, 0.648, 0.639, 0.504, 0.182, 0.3, 0.173, 0.195, 0.169, 0.204, 0.218, 0.042, 0.013, 0.0001, 0.001, 0.0001, 0.005, 0.0001, 0.26, 0.146, 0.257, 0.104, 0.401, 0.111, 0.564, 0.173, 0.223, 0.038, 0.106, 0.097, 0.317, 0.12, 0.025, 0.199, 0.01, 0.074, 0.256, 0.153, 0.279, 0.066, 0.06, 0.002, 0.047, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 10.968, 0.472, 0.524, 1.575, 4.239, 0.44, 3.776, 1.808, 6.943, 0.028, 1.21, 2.019, 1.749, 8.291, 5.798, 1.592, 0.018, 1.795, 5.81, 3.872, 3.565, 0.141, 0.106, 0.012, 0.845, 0.055, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.016, 0.008, 0.003, 0.001, 0.002, 0.01, 0.001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.0001, 0.0001, 0.006, 0.001, 0.0001, 0.005, 0.003, 0.0001, 0.0001, 0.001, 0.0001, 0.011, 0.0001, 0.003, 0.001, 0.0001, 0.002, 0.001, 0.0001, 0.02, 0.002, 0.003, 0.001, 0.974, 0.0001, 0.004, 0.003, 0.012, 0.0001, 0.0001, 0.001, 0.009, 0.0001, 0.0001, 0.002, 0.432, 0.0001, 0.044, 0.0001, 0.001, 0.003, 0.001, 0.002, 0.002, 0.006, 0.007, 0.002, 0.002, 0.001, 0.003, 0.0001, 0.0001, 0.001, 1.51, 0.004, 0.013, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.002, 0.0001, 0.0001, 0.005, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.002, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.012, 0.0001, 0.0001, 0.003, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cho": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.477, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.446, 0.089, 1.242, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.621, 0.621, 0.0001, 0.0001, 0.799, 0.0001, 0.532, 0.0001, 0.0001, 0.177, 0.089, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.355, 0.266, 0.0001, 0.0001, 0.0001, 0.089, 0.0001, 0.444, 0.0001, 1.154, 0.0001, 0.0001, 0.0001, 0.089, 0.799, 0.177, 0.0001, 0.177, 0.0001, 0.355, 0.177, 0.177, 0.444, 0.0001, 0.0001, 0.355, 0.0001, 0.0001, 0.089, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.955, 1.154, 0.799, 0.0001, 2.839, 0.177, 0.621, 7.365, 8.252, 0.0001, 5.146, 2.662, 3.549, 3.727, 5.413, 1.597, 0.0001, 0.799, 3.638, 5.146, 1.597, 1.065, 0.089, 0.0001, 1.331, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.154, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.266, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.177, 0.0001, 0.0001, 0.0001, 1.154, 0.0001, 0.089, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "chr": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.394, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.115, 0.002, 0.174, 0.0001, 0.001, 0.005, 0.001, 0.018, 0.095, 0.095, 0.0001, 0.001, 0.499, 0.081, 0.439, 0.009, 0.086, 0.076, 0.045, 0.025, 0.02, 0.027, 0.02, 0.018, 0.025, 0.029, 0.03, 0.019, 0.002, 0.001, 0.003, 0.002, 0.0001, 0.037, 0.02, 0.038, 0.014, 0.023, 0.017, 0.012, 0.014, 0.019, 0.011, 0.01, 0.013, 0.028, 0.014, 0.01, 0.02, 0.002, 0.016, 0.034, 0.027, 0.013, 0.008, 0.015, 0.002, 0.005, 0.003, 0.065, 0.0001, 0.065, 0.0001, 0.004, 0.0001, 0.692, 0.092, 0.264, 0.31, 0.823, 0.092, 0.184, 0.209, 0.663, 0.01, 0.064, 0.374, 0.188, 0.502, 0.498, 0.163, 0.016, 0.479, 0.482, 0.523, 0.235, 0.107, 0.076, 0.023, 0.123, 0.021, 0.0001, 0.028, 0.0001, 0.0001, 0.0001, 0.027, 0.355, 0.722, 0.213, 0.313, 0.628, 0.115, 0.06, 0.021, 0.056, 0.084, 0.04, 0.154, 1.876, 13.554, 13.952, 0.082, 0.032, 0.441, 0.837, 0.268, 0.161, 0.041, 1.986, 0.138, 0.561, 0.191, 0.664, 0.014, 0.045, 0.005, 0.13, 2.057, 0.126, 1.445, 0.138, 1.031, 0.39, 0.904, 0.381, 0.457, 1.048, 0.569, 0.458, 0.748, 0.433, 0.062, 1.427, 0.213, 0.207, 0.29, 0.574, 0.831, 0.687, 0.218, 0.077, 0.387, 0.051, 0.016, 0.01, 0.004, 0.004, 1.405, 0.134, 0.0001, 0.0001, 0.009, 0.006, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 27.238, 0.017, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "chy": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.992, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.662, 0.002, 0.655, 0.0001, 0.0001, 0.0001, 0.0001, 4.281, 0.488, 0.49, 0.012, 0.039, 1.209, 0.935, 1.193, 0.009, 0.099, 0.186, 0.07, 0.039, 0.048, 0.046, 0.051, 0.032, 0.087, 0.113, 0.294, 0.06, 0.044, 0.012, 0.043, 0.009, 0.0001, 0.28, 0.143, 0.271, 0.068, 0.058, 0.046, 0.056, 0.705, 0.041, 0.084, 0.094, 0.075, 0.71, 0.203, 0.133, 0.21, 0.01, 0.123, 0.333, 0.369, 0.109, 0.326, 0.043, 0.02, 0.015, 0.015, 0.017, 0.0001, 0.017, 0.0001, 0.0001, 0.005, 5.694, 0.454, 0.435, 0.594, 8.431, 0.195, 0.654, 4.544, 1.753, 0.053, 1.313, 1.118, 1.931, 4.523, 6.14, 0.553, 0.043, 1.203, 5.097, 4.735, 0.637, 1.842, 0.224, 0.461, 0.27, 0.08, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.024, 0.014, 0.009, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.113, 0.0001, 0.002, 0.007, 0.005, 0.0001, 0.0001, 0.005, 0.002, 0.0001, 0.058, 0.012, 0.0001, 0.003, 0.029, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.002, 0.002, 0.044, 1.384, 0.696, 0.009, 0.027, 0.002, 0.002, 0.039, 0.005, 3.484, 0.98, 0.162, 0.003, 0.009, 0.002, 0.017, 0.009, 0.003, 0.005, 1.282, 0.993, 0.003, 0.142, 0.0001, 0.017, 0.0001, 0.002, 0.009, 0.007, 0.0001, 0.007, 0.005, 0.0001, 0.0001, 0.014, 8.846, 0.043, 0.545, 0.0001, 0.005, 0.046, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.031, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.019, 0.003, 0.017, 0.0001, 0.0001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ckb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.676, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.035, 0.002, 0.062, 0.0001, 0.0001, 0.003, 0.0001, 0.002, 0.131, 0.13, 0.001, 0.001, 0.011, 0.034, 0.374, 0.013, 0.01, 0.014, 0.008, 0.005, 0.004, 0.004, 0.004, 0.004, 0.005, 0.007, 0.05, 0.0001, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.009, 0.006, 0.007, 0.006, 0.004, 0.004, 0.004, 0.004, 0.005, 0.002, 0.003, 0.004, 0.007, 0.005, 0.003, 0.007, 0.001, 0.005, 0.01, 0.007, 0.002, 0.002, 0.003, 0.001, 0.001, 0.001, 0.004, 0.0001, 0.004, 0.0001, 0.003, 0.0001, 0.058, 0.008, 0.018, 0.017, 0.063, 0.009, 0.012, 0.017, 0.048, 0.001, 0.008, 0.031, 0.019, 0.043, 0.045, 0.012, 0.001, 0.045, 0.029, 0.036, 0.019, 0.006, 0.008, 0.003, 0.011, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.386, 0.193, 0.124, 0.067, 1.187, 1.207, 3.947, 0.41, 3.556, 0.028, 0.015, 0.002, 5.576, 0.003, 1.191, 0.005, 0.006, 0.005, 0.002, 0.004, 0.001, 6.665, 0.001, 0.002, 0.236, 0.001, 0.002, 0.008, 0.002, 0.002, 0.001, 0.006, 0.161, 0.192, 0.114, 0.062, 0.112, 0.064, 0.707, 4.366, 1.564, 2.13, 1.551, 0.015, 0.253, 0.092, 0.303, 2.261, 0.008, 2.411, 0.524, 1.151, 0.651, 0.531, 0.001, 0.004, 0.003, 0.092, 0.048, 0.036, 0.003, 0.003, 0.823, 0.003, 0.0001, 0.0001, 0.028, 0.007, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 15.514, 10.978, 4.45, 13.188, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.375, 0.002, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.063, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "co":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.449, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.862, 0.008, 0.387, 0.0001, 0.0001, 0.006, 0.001, 0.763, 0.212, 0.212, 0.003, 0.001, 0.925, 0.075, 0.859, 0.019, 0.189, 0.28, 0.146, 0.097, 0.087, 0.101, 0.081, 0.085, 0.107, 0.132, 0.097, 0.026, 0.009, 0.003, 0.01, 0.004, 0.0001, 0.325, 0.102, 0.335, 0.094, 0.091, 0.089, 0.126, 0.077, 0.208, 0.025, 0.02, 0.156, 0.189, 0.082, 0.052, 0.201, 0.016, 0.093, 0.268, 0.121, 0.17, 0.078, 0.019, 0.022, 0.005, 0.013, 0.032, 0.0001, 0.032, 0.0001, 0.016, 0.0001, 8.602, 0.557, 3.322, 3.101, 4.329, 0.784, 1.174, 1.381, 10.092, 0.419, 0.069, 2.83, 1.864, 5.457, 2.618, 1.888, 0.179, 4.342, 3.458, 4.676, 6.626, 0.877, 0.033, 0.017, 0.063, 0.595, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.058, 0.006, 0.004, 0.002, 0.003, 0.001, 0.001, 0.001, 0.004, 0.001, 0.001, 0.0001, 0.002, 0.002, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.039, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.789, 0.005, 0.002, 0.002, 0.004, 0.001, 0.002, 0.004, 0.94, 0.016, 0.001, 0.007, 0.251, 0.004, 0.001, 0.002, 0.005, 0.006, 0.189, 0.011, 0.005, 0.003, 0.002, 0.024, 0.003, 0.252, 0.004, 0.007, 0.006, 0.005, 0.002, 0.004, 0.0001, 0.0001, 0.05, 2.469, 0.006, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.032, 0.015, 0.008, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.004, 0.04, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.443, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.088, 0.004, 0.073, 0.0001, 0.0001, 0.02, 0.0001, 0.023, 0.121, 0.12, 0.0001, 0.002, 0.629, 0.081, 0.971, 0.012, 0.119, 0.193, 0.101, 0.064, 0.076, 0.066, 0.061, 0.066, 0.062, 0.105, 0.063, 0.027, 0.0001, 0.0001, 0.0001, 0.015, 0.0001, 0.161, 0.04, 0.143, 0.045, 0.195, 0.034, 0.029, 0.053, 0.081, 0.084, 0.151, 0.056, 0.235, 0.167, 0.103, 0.138, 0.009, 0.033, 0.115, 0.119, 0.03, 0.034, 0.067, 0.012, 0.01, 0.004, 0.05, 0.0001, 0.047, 0.0001, 0.014, 0.0001, 9.914, 0.233, 4.69, 1.145, 5.906, 0.235, 0.326, 1.052, 10.924, 0.134, 6.149, 1.256, 2.551, 4.689, 5.033, 1.928, 0.073, 2.706, 3.099, 5.744, 0.924, 0.192, 2.967, 0.038, 0.312, 0.067, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.088, 0.031, 0.077, 0.099, 0.046, 0.115, 0.007, 0.048, 0.054, 0.011, 0.091, 0.103, 0.074, 0.037, 0.073, 0.005, 0.766, 0.405, 0.312, 0.295, 0.175, 0.052, 0.036, 0.009, 0.01, 0.038, 0.001, 0.005, 0.002, 0.0001, 0.001, 0.021, 0.037, 0.111, 0.205, 0.026, 0.084, 0.087, 0.065, 0.093, 0.076, 0.063, 0.057, 0.032, 0.002, 0.144, 0.111, 0.096, 0.017, 0.078, 0.065, 0.232, 0.037, 0.005, 0.0001, 0.0001, 0.021, 0.005, 0.022, 0.02, 0.014, 0.002, 0.005, 0.005, 0.0001, 0.0001, 0.046, 0.821, 0.023, 0.077, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 1.861, 0.08, 0.005, 0.002, 0.009, 0.005, 0.0001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "crh": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.666, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.545, 0.003, 0.2, 0.0001, 0.003, 0.006, 0.0001, 0.011, 0.498, 0.498, 0.001, 0.003, 0.581, 0.375, 1.265, 0.029, 0.54, 0.844, 0.447, 0.25, 0.254, 0.244, 0.225, 0.224, 0.237, 0.353, 0.036, 0.017, 0.017, 0.002, 0.017, 0.003, 0.0001, 0.292, 0.227, 0.115, 0.122, 0.258, 0.045, 0.081, 0.079, 0.299, 0.014, 0.172, 0.079, 0.19, 0.068, 0.102, 0.074, 0.317, 0.092, 0.196, 0.162, 0.157, 0.161, 0.003, 0.13, 0.089, 0.035, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 7.42, 1.383, 0.39, 2.173, 6.493, 0.253, 0.439, 0.324, 6.527, 0.039, 1.974, 3.301, 1.629, 5.164, 1.476, 0.486, 0.955, 4.625, 3.637, 2.416, 1.149, 1.071, 0.013, 0.004, 1.959, 0.598, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.415, 0.022, 0.015, 0.022, 0.008, 0.007, 0.005, 0.065, 0.008, 0.005, 0.004, 0.008, 0.007, 0.007, 0.003, 0.008, 0.005, 0.004, 0.004, 0.069, 0.234, 0.004, 0.026, 0.004, 0.006, 0.008, 0.008, 0.005, 0.067, 0.049, 0.094, 1.497, 0.026, 0.01, 0.278, 0.006, 0.008, 0.006, 0.005, 0.416, 0.004, 0.006, 0.005, 0.014, 0.004, 0.007, 0.006, 0.006, 0.149, 5.025, 0.014, 0.011, 0.012, 0.067, 0.295, 0.006, 0.022, 0.01, 0.019, 0.017, 0.605, 0.022, 0.039, 0.006, 0.0001, 0.0001, 0.035, 2.796, 4.495, 1.1, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.003, 0.002, 0.256, 0.079, 0.004, 0.002, 0.0001, 0.004, 0.008, 0.013, 0.021, 0.017, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.015, 0.009, 0.398, 0.007, 0.004, 0.019, 0.009, 0.005, 0.004, 0.004, 0.003, 0.002, 0.004, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "csb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.825, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.296, 0.002, 0.584, 0.0001, 0.0001, 0.003, 0.001, 0.009, 0.331, 0.334, 0.002, 0.0001, 0.877, 0.236, 1.256, 0.065, 0.271, 0.637, 0.291, 0.193, 0.181, 0.174, 0.153, 0.187, 0.256, 0.339, 0.093, 0.04, 0.024, 0.004, 0.024, 0.003, 0.0001, 0.093, 0.136, 0.203, 0.135, 0.053, 0.045, 0.141, 0.038, 0.163, 0.132, 0.28, 0.122, 0.184, 0.116, 0.024, 0.275, 0.002, 0.1, 0.23, 0.118, 0.014, 0.056, 0.218, 0.119, 0.003, 0.085, 0.006, 0.0001, 0.007, 0.0001, 0.002, 0.0001, 4.612, 0.986, 3.096, 2.007, 3.546, 0.161, 1.136, 0.946, 4.255, 1.343, 2.142, 1.634, 1.571, 3.378, 2.668, 1.384, 0.004, 3.469, 3.152, 2.405, 0.834, 0.037, 2.89, 0.011, 0.614, 4.079, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.169, 0.025, 0.879, 0.003, 0.332, 0.515, 0.031, 0.005, 0.001, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.005, 0.001, 0.013, 0.102, 0.134, 0.005, 0.002, 0.001, 0.001, 0.003, 0.049, 0.005, 0.012, 0.006, 0.026, 0.025, 0.003, 0.016, 0.006, 0.006, 0.677, 0.002, 0.001, 0.001, 0.001, 0.003, 1.17, 0.001, 2.19, 0.001, 0.003, 0.0001, 0.002, 0.009, 0.003, 2.322, 0.76, 1.31, 0.003, 0.004, 0.001, 0.007, 0.615, 0.005, 0.077, 0.465, 0.007, 0.003, 0.002, 0.0001, 0.0001, 0.14, 9.122, 0.543, 1.724, 0.0001, 0.001, 0.002, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.006, 0.002, 0.024, 0.023, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.197, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cu":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.095, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.137, 0.0001, 0.05, 0.0001, 0.001, 0.001, 0.0001, 0.002, 0.026, 0.026, 0.001, 0.0001, 0.049, 0.014, 0.024, 0.015, 0.131, 0.259, 0.12, 0.082, 0.083, 0.082, 0.076, 0.078, 0.096, 0.129, 0.009, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.003, 0.001, 0.006, 0.001, 0.001, 0.001, 0.0001, 0.006, 0.004, 0.0001, 0.001, 0.001, 0.002, 0.002, 0.004, 0.001, 0.0001, 0.002, 0.004, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.003, 0.0001, 0.023, 0.002, 0.008, 0.007, 0.018, 0.001, 0.005, 0.004, 0.017, 0.005, 0.009, 0.01, 0.003, 0.016, 0.015, 0.003, 0.001, 0.01, 0.011, 0.009, 0.011, 0.004, 0.0001, 0.002, 0.002, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.938, 4.019, 2.29, 0.582, 0.265, 0.184, 0.28, 0.33, 0.175, 0.126, 2.698, 0.002, 1.962, 0.002, 0.135, 0.0001, 0.124, 0.906, 0.12, 0.072, 1.561, 0.0001, 0.139, 0.857, 0.034, 2.179, 0.103, 0.119, 0.097, 0.099, 0.095, 0.124, 0.126, 0.438, 0.049, 1.297, 0.06, 0.96, 0.01, 0.295, 0.011, 0.359, 0.005, 0.236, 0.002, 0.101, 0.019, 0.025, 3.114, 0.623, 1.373, 0.62, 1.221, 0.086, 0.518, 0.573, 2.627, 0.002, 1.325, 1.567, 0.924, 2.121, 2.823, 0.585, 0.0001, 0.0001, 0.514, 0.003, 0.006, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.001, 0.408, 0.0001, 0.016, 0.012, 21.25, 18.718, 0.249, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.51, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 1.747, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.247, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.093, 0.001, 0.059, 0.0001, 0.0001, 0.007, 0.0001, 0.003, 0.152, 0.151, 0.0001, 0.002, 0.478, 0.273, 0.79, 0.011, 0.204, 0.309, 0.183, 0.104, 0.101, 0.1, 0.081, 0.081, 0.096, 0.17, 0.076, 0.008, 0.002, 0.002, 0.002, 0.003, 0.0001, 0.004, 0.003, 0.005, 0.002, 0.002, 0.002, 0.002, 0.002, 0.019, 0.001, 0.001, 0.002, 0.003, 0.002, 0.002, 0.003, 0.0001, 0.003, 0.005, 0.003, 0.002, 0.006, 0.001, 0.01, 0.001, 0.0001, 0.013, 0.0001, 0.013, 0.0001, 0.001, 0.0001, 0.027, 0.004, 0.007, 0.008, 0.027, 0.004, 0.006, 0.007, 0.02, 0.001, 0.004, 0.016, 0.009, 0.019, 0.018, 0.006, 0.0001, 0.019, 0.014, 0.015, 0.011, 0.003, 0.002, 0.002, 0.004, 0.001, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 3.257, 1.78, 2.381, 2.851, 0.156, 1.36, 0.178, 0.773, 1.001, 0.006, 0.006, 0.869, 0.319, 0.035, 0.373, 0.165, 0.161, 0.088, 0.098, 0.049, 0.312, 2.25, 0.007, 0.017, 0.069, 0.007, 0.174, 0.039, 0.101, 0.06, 0.095, 0.155, 0.212, 0.157, 0.129, 0.054, 0.061, 0.066, 0.005, 1.16, 0.101, 0.002, 0.0001, 0.045, 0.001, 0.021, 0.156, 0.041, 4.16, 0.372, 1.295, 0.368, 0.304, 3.139, 0.041, 0.13, 2.185, 0.64, 1.311, 1.785, 0.994, 3.619, 1.18, 1.135, 0.0001, 0.0001, 0.101, 1.175, 3.79, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.0001, 0.002, 0.001, 24.733, 13.586, 0.004, 0.088, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.282, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "cy":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.628, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.48, 0.003, 0.545, 0.0001, 0.001, 0.007, 0.002, 0.872, 0.259, 0.258, 0.001, 0.001, 0.777, 0.194, 0.96, 0.016, 0.363, 0.487, 0.244, 0.138, 0.133, 0.135, 0.125, 0.126, 0.164, 0.239, 0.149, 0.081, 0.022, 0.001, 0.022, 0.003, 0.0001, 0.36, 0.242, 0.56, 0.267, 0.155, 0.163, 0.331, 0.126, 0.112, 0.06, 0.033, 0.279, 0.433, 0.133, 0.073, 0.238, 0.004, 0.18, 0.303, 0.196, 0.061, 0.026, 0.092, 0.003, 0.167, 0.006, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 7.082, 0.905, 1.506, 6.475, 6.263, 2.165, 2.494, 2.4, 4.773, 0.015, 0.114, 3.901, 1.419, 6.217, 4.277, 0.556, 0.008, 5.57, 2.092, 2.13, 1.941, 0.086, 2.82, 0.025, 5.712, 0.034, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.074, 0.005, 0.003, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.021, 0.002, 0.001, 0.0001, 0.001, 0.002, 0.033, 0.001, 0.001, 0.007, 0.009, 0.001, 0.001, 0.033, 0.007, 0.059, 0.003, 0.003, 0.001, 0.001, 0.003, 0.004, 0.015, 0.016, 0.004, 0.001, 0.004, 0.01, 0.012, 0.004, 0.003, 0.003, 0.004, 0.074, 0.043, 0.005, 0.016, 0.003, 0.006, 0.003, 0.002, 0.004, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.036, 0.221, 0.003, 0.06, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.007, 0.004, 0.014, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.072, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "din": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.698, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.927, 0.0001, 0.06, 0.0001, 0.003, 0.013, 0.0001, 0.015, 0.171, 0.17, 0.0001, 0.0001, 0.878, 0.077, 0.901, 0.027, 0.297, 0.229, 0.151, 0.055, 0.064, 0.078, 0.053, 0.048, 0.049, 0.126, 0.018, 0.013, 0.002, 0.0001, 0.002, 0.005, 0.0001, 0.424, 0.153, 0.093, 0.101, 0.075, 0.019, 0.074, 0.021, 0.051, 0.069, 0.324, 0.085, 0.16, 0.163, 0.021, 0.306, 0.002, 0.087, 0.062, 0.288, 0.034, 0.007, 0.069, 0.0001, 0.136, 0.003, 0.027, 0.0001, 0.027, 0.0001, 0.0001, 0.0001, 5.438, 0.999, 2.9, 1.603, 4.394, 0.024, 0.521, 1.912, 3.749, 0.362, 4.818, 2.02, 1.512, 4.26, 1.668, 1.035, 0.003, 2.29, 0.155, 3.595, 3.428, 0.022, 0.527, 0.011, 2.005, 0.013, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.036, 0.001, 0.001, 0.0001, 0.027, 0.0001, 0.026, 0.0001, 1.487, 0.0001, 0.005, 1.04, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 2.319, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 1.678, 0.006, 0.006, 0.0001, 0.0001, 0.01, 0.0001, 0.0001, 0.222, 1.181, 0.0001, 0.0001, 0.001, 0.004, 0.001, 0.0001, 3.25, 0.0001, 0.001, 0.006, 1.508, 0.003, 0.002, 0.006, 0.002, 0.0001, 0.011, 1.021, 0.001, 0.004, 0.002, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.016, 6.971, 0.0001, 1.041, 0.02, 0.0001, 0.0001, 4.193, 0.0001, 0.0001, 1.487, 0.0001, 0.027, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.062, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "diq": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.719, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.354, 0.008, 0.4, 0.0001, 0.0001, 0.009, 0.0001, 0.031, 0.299, 0.3, 0.001, 0.003, 0.98, 0.165, 1.27, 0.045, 0.227, 0.302, 0.162, 0.087, 0.08, 0.089, 0.076, 0.082, 0.096, 0.17, 0.156, 0.035, 0.026, 0.008, 0.027, 0.01, 0.0001, 0.309, 0.187, 0.135, 0.206, 0.243, 0.108, 0.12, 0.188, 0.05, 0.033, 0.209, 0.106, 0.271, 0.167, 0.06, 0.167, 0.062, 0.13, 0.271, 0.259, 0.059, 0.085, 0.06, 0.052, 0.088, 0.128, 0.014, 0.0001, 0.014, 0.0001, 0.002, 0.001, 7.586, 1.293, 0.911, 2.514, 8.148, 0.439, 0.62, 0.759, 4.61, 0.11, 2.125, 1.599, 2.095, 4.93, 3.468, 0.588, 0.377, 4.808, 2.018, 2.359, 1.695, 0.626, 1.106, 0.479, 3.36, 1.081, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.078, 0.018, 0.011, 0.012, 0.02, 0.015, 0.019, 0.078, 0.016, 0.004, 0.018, 0.002, 0.014, 0.004, 0.014, 0.003, 0.006, 0.005, 0.003, 0.011, 0.005, 0.006, 0.005, 0.002, 0.009, 0.023, 0.002, 0.004, 0.022, 0.016, 0.065, 0.865, 0.032, 0.01, 0.013, 0.005, 0.007, 0.004, 0.006, 0.242, 0.014, 0.032, 2.716, 0.012, 0.007, 0.008, 0.29, 0.015, 0.191, 2.379, 0.013, 0.015, 0.01, 0.006, 0.021, 0.004, 0.009, 0.01, 0.007, 0.128, 0.093, 0.009, 0.008, 0.006, 0.0001, 0.0001, 0.039, 3.563, 2.668, 0.816, 0.0001, 0.001, 0.0001, 0.005, 0.003, 0.002, 0.002, 0.0001, 0.03, 0.013, 0.034, 0.014, 0.001, 0.0001, 0.001, 0.012, 0.005, 0.037, 0.126, 0.091, 0.007, 0.013, 0.003, 0.0001, 0.0001, 0.0001, 0.019, 0.012, 0.072, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "dsb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.783, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.853, 0.003, 0.608, 0.0001, 0.0001, 0.007, 0.002, 0.016, 0.311, 0.311, 0.022, 0.002, 0.839, 0.138, 1.194, 0.023, 0.287, 0.411, 0.214, 0.128, 0.124, 0.131, 0.109, 0.104, 0.125, 0.201, 0.084, 0.035, 0.006, 0.007, 0.007, 0.003, 0.0001, 0.155, 0.168, 0.123, 0.122, 0.077, 0.058, 0.102, 0.068, 0.054, 0.115, 0.164, 0.108, 0.197, 0.144, 0.038, 0.256, 0.004, 0.113, 0.246, 0.119, 0.042, 0.025, 0.244, 0.005, 0.007, 0.075, 0.008, 0.0001, 0.008, 0.0001, 0.002, 0.0001, 6.833, 1.047, 1.719, 1.818, 5.619, 0.234, 0.977, 0.835, 3.647, 3.795, 2.962, 1.965, 2.079, 4.006, 5.923, 1.615, 0.008, 3.224, 3.399, 2.803, 2.458, 0.071, 3.327, 0.021, 1.623, 1.195, 0.0001, 0.003, 0.0001, 0.001, 0.0001, 0.148, 0.049, 0.931, 0.01, 0.22, 0.006, 0.005, 0.266, 0.005, 0.002, 0.002, 0.002, 0.017, 0.029, 0.002, 0.002, 0.007, 0.003, 0.004, 0.026, 0.004, 0.064, 0.004, 0.004, 0.009, 0.024, 0.008, 1.886, 0.043, 0.009, 0.04, 0.009, 0.064, 0.625, 0.008, 0.004, 0.017, 0.003, 0.003, 0.004, 0.006, 0.017, 0.003, 0.004, 0.001, 0.008, 0.001, 0.002, 0.019, 0.008, 0.014, 1.225, 0.005, 0.009, 0.011, 0.005, 0.012, 0.012, 0.395, 0.009, 0.027, 0.02, 0.616, 0.016, 0.0001, 0.0001, 0.039, 1.311, 1.431, 3.692, 0.0001, 0.0001, 0.001, 0.004, 0.001, 0.001, 0.001, 0.0001, 0.017, 0.009, 0.074, 0.029, 0.001, 0.0001, 0.0001, 0.002, 0.007, 0.043, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.012, 0.141, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "dty": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.724, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.716, 0.001, 0.019, 0.0001, 0.0001, 0.003, 0.0001, 0.008, 0.063, 0.066, 0.001, 0.0001, 0.189, 0.033, 0.052, 0.008, 0.003, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.027, 0.004, 0.012, 0.001, 0.012, 0.001, 0.0001, 0.002, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.017, 0.012, 0.004, 0.005, 0.014, 0.002, 0.003, 0.006, 0.016, 0.001, 0.004, 0.007, 0.008, 0.013, 0.011, 0.003, 0.0001, 0.019, 0.008, 0.009, 0.004, 0.001, 0.003, 0.0001, 0.003, 0.001, 0.0001, 0.015, 0.0001, 0.0001, 0.0001, 0.87, 0.744, 0.354, 0.069, 0.0001, 0.295, 0.114, 1.106, 0.404, 0.216, 0.006, 1.008, 0.08, 2.434, 0.0001, 0.171, 0.009, 0.001, 0.001, 0.025, 0.014, 1.53, 0.174, 0.539, 0.045, 0.068, 0.25, 0.269, 0.443, 0.023, 0.04, 0.304, 0.083, 0.214, 0.028, 0.182, 24.937, 7.5, 0.641, 0.298, 1.687, 0.033, 0.816, 0.129, 0.459, 0.371, 1.179, 1.062, 2.109, 0.002, 1.084, 0.0001, 0.0001, 0.578, 0.275, 0.191, 1.004, 0.659, 0.001, 0.0001, 0.01, 0.01, 3.197, 1.534, 0.0001, 0.0001, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.897, 0.0001, 0.034, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "dv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.449, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.782, 0.003, 0.057, 0.0001, 0.0001, 0.005, 0.0001, 0.005, 0.068, 0.068, 0.0001, 0.001, 0.01, 0.02, 0.58, 0.003, 0.08, 0.111, 0.068, 0.041, 0.031, 0.037, 0.03, 0.031, 0.035, 0.052, 0.01, 0.001, 0.003, 0.002, 0.003, 0.0001, 0.0001, 0.003, 0.001, 0.005, 0.002, 0.002, 0.002, 0.001, 0.003, 0.003, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.003, 0.0001, 0.002, 0.003, 0.005, 0.001, 0.001, 0.003, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.004, 0.0001, 0.069, 0.013, 0.026, 0.027, 0.096, 0.015, 0.017, 0.033, 0.065, 0.001, 0.006, 0.037, 0.021, 0.063, 0.061, 0.016, 0.001, 0.05, 0.05, 0.064, 0.025, 0.009, 0.011, 0.002, 0.014, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.961, 0.592, 2.65, 1.657, 0.723, 0.269, 1.597, 3.461, 1.72, 1.651, 0.757, 0.977, 1.223, 0.768, 1.538, 0.011, 0.778, 0.359, 0.094, 0.266, 0.255, 0.126, 0.187, 0.051, 0.006, 0.076, 0.047, 0.004, 0.004, 0.086, 0.041, 0.008, 0.02, 0.003, 0.091, 0.008, 0.069, 0.003, 5.331, 1.558, 2.986, 0.988, 3.164, 0.17, 3.662, 0.439, 0.51, 0.17, 3.636, 0.006, 0.014, 0.003, 0.002, 0.002, 0.001, 0.014, 0.001, 0.004, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.201, 0.101, 0.0001, 0.002, 0.0001, 0.0001, 45.417, 0.0001, 0.002, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.02, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "dz":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.39, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.815, 0.0001, 0.004, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.023, 0.023, 0.0001, 0.002, 0.003, 0.013, 0.008, 0.001, 0.017, 0.015, 0.012, 0.006, 0.005, 0.004, 0.005, 0.004, 0.004, 0.004, 0.001, 0.0001, 0.007, 0.0001, 0.007, 0.001, 0.0001, 0.002, 0.004, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.003, 0.001, 0.0001, 0.004, 0.0001, 0.002, 0.003, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.03, 0.011, 0.006, 0.008, 0.024, 0.002, 0.006, 0.009, 0.021, 0.002, 0.004, 0.014, 0.011, 0.019, 0.021, 0.004, 0.0001, 0.02, 0.011, 0.013, 0.01, 0.002, 0.002, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.269, 0.247, 1.794, 0.002, 1.18, 0.189, 0.19, 0.052, 0.002, 0.102, 0.016, 7.859, 0.051, 0.549, 0.008, 0.12, 0.301, 1.592, 0.28, 1.053, 0.694, 0.157, 1.278, 0.061, 0.824, 0.093, 0.2, 0.068, 0.006, 0.019, 0.267, 0.283, 0.898, 0.517, 1.238, 0.954, 0.214, 0.015, 2.251, 0.029, 0.117, 0.081, 0.001, 0.058, 0.0001, 0.012, 0.002, 0.0001, 0.002, 0.89, 2.149, 0.094, 1.08, 0.001, 0.0001, 0.053, 0.001, 0.0001, 0.926, 0.001, 10.076, 21.494, 2.583, 0.002, 0.0001, 0.0001, 0.002, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 32.733, 0.0001, 0.016, 0.005, 0.001, 0.002, 0.002, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ee":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.047, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.659, 0.001, 0.347, 0.0001, 0.001, 0.004, 0.004, 0.044, 0.199, 0.199, 0.001, 0.0001, 0.713, 0.054, 1.348, 0.005, 0.312, 0.38, 0.219, 0.115, 0.09, 0.132, 0.118, 0.118, 0.109, 0.211, 0.064, 0.006, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.552, 0.172, 0.134, 0.182, 0.397, 0.085, 0.215, 0.112, 0.083, 0.04, 0.209, 0.217, 0.202, 0.168, 0.043, 0.117, 0.006, 0.112, 0.229, 0.176, 0.053, 0.059, 0.177, 0.021, 0.139, 0.02, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 7.214, 1.62, 0.258, 2.122, 10.212, 0.557, 1.427, 0.62, 4.11, 0.028, 2.137, 3.419, 2.267, 3.348, 4.663, 0.886, 0.007, 1.264, 2.303, 2.327, 2.541, 0.557, 2.031, 0.389, 1.697, 0.84, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.058, 0.011, 0.016, 0.109, 0.004, 0.002, 0.009, 0.001, 0.003, 0.01, 0.044, 0.61, 0.005, 0.002, 0.0001, 0.003, 0.018, 0.018, 1.229, 0.009, 2.883, 0.003, 1.23, 0.001, 0.002, 0.008, 0.003, 0.085, 0.02, 0.018, 0.001, 0.001, 0.052, 0.01, 0.004, 0.485, 0.002, 0.0001, 0.002, 0.004, 0.005, 0.042, 0.003, 0.002, 0.003, 0.025, 0.002, 0.002, 0.007, 0.009, 0.047, 0.01, 0.005, 0.003, 0.005, 0.003, 0.006, 0.005, 0.14, 0.007, 0.005, 0.138, 0.008, 0.004, 0.0001, 0.0001, 0.039, 0.487, 0.018, 0.548, 1.276, 0.0001, 0.004, 4.335, 0.128, 0.004, 0.106, 0.013, 0.028, 0.013, 0.041, 0.016, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.138, 0.051, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "eml": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.684, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.039, 0.004, 0.415, 0.0001, 0.0001, 0.004, 0.001, 1.632, 0.216, 0.216, 0.001, 0.001, 0.746, 0.069, 0.997, 0.011, 0.415, 0.659, 0.408, 0.216, 0.231, 0.235, 0.226, 0.213, 0.215, 0.256, 0.061, 0.026, 0.05, 0.006, 0.05, 0.003, 0.0001, 0.44, 0.139, 0.4, 0.112, 0.078, 0.095, 0.114, 0.018, 0.424, 0.019, 0.012, 0.251, 0.226, 0.059, 0.026, 0.233, 0.016, 0.153, 0.231, 0.099, 0.036, 0.164, 0.011, 0.127, 0.003, 0.015, 0.004, 0.0001, 0.004, 0.0001, 0.002, 0.0001, 7.63, 0.549, 2.301, 3.601, 3.529, 0.617, 1.263, 0.808, 5.22, 0.113, 0.052, 4.92, 1.657, 5.406, 1.72, 1.353, 0.118, 3.957, 2.689, 3.146, 2.026, 0.904, 0.024, 0.02, 0.047, 0.34, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.239, 0.003, 0.006, 0.003, 0.004, 0.052, 0.002, 0.003, 0.008, 0.003, 0.006, 0.001, 0.002, 0.193, 0.002, 0.001, 0.002, 0.002, 0.003, 0.098, 0.002, 0.001, 0.001, 0.001, 0.033, 0.188, 0.003, 0.047, 0.006, 0.006, 0.001, 0.078, 0.562, 0.025, 0.617, 0.129, 0.182, 0.072, 0.003, 0.005, 1.444, 0.829, 0.895, 0.057, 0.235, 0.011, 0.346, 0.001, 0.004, 0.003, 0.664, 0.345, 0.314, 0.007, 0.019, 0.001, 0.003, 0.275, 0.004, 0.186, 0.062, 0.002, 0.002, 0.006, 0.0001, 0.0001, 0.011, 6.936, 0.1, 0.325, 0.0001, 0.004, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.003, 0.002, 0.007, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.192, 0.237, 0.003, 0.002, 0.005, 0.003, 0.003, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "eo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.154, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.737, 0.006, 0.429, 0.0001, 0.0001, 0.01, 0.001, 0.015, 0.235, 0.235, 0.001, 0.003, 0.936, 0.306, 0.916, 0.015, 0.284, 0.481, 0.226, 0.14, 0.134, 0.143, 0.121, 0.123, 0.155, 0.273, 0.072, 0.027, 0.012, 0.007, 0.013, 0.002, 0.0001, 0.209, 0.154, 0.114, 0.106, 0.232, 0.094, 0.127, 0.102, 0.106, 0.077, 0.183, 0.354, 0.184, 0.118, 0.083, 0.187, 0.004, 0.116, 0.241, 0.149, 0.061, 0.074, 0.035, 0.004, 0.009, 0.024, 0.021, 0.0001, 0.021, 0.0001, 0.004, 0.0001, 9.544, 0.784, 0.841, 2.534, 6.934, 0.706, 0.989, 0.423, 6.212, 2.407, 2.868, 4.302, 1.963, 5.456, 7.143, 1.699, 0.009, 4.617, 4.113, 4.222, 2.31, 1.083, 0.045, 0.017, 0.108, 0.46, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.058, 0.01, 0.009, 0.008, 0.004, 0.004, 0.002, 0.003, 0.037, 0.239, 0.001, 0.001, 0.004, 0.007, 0.001, 0.002, 0.002, 0.007, 0.002, 0.018, 0.008, 0.001, 0.002, 0.002, 0.002, 0.012, 0.003, 0.005, 0.093, 0.609, 0.008, 0.005, 0.021, 0.066, 0.005, 0.003, 0.01, 0.029, 0.002, 0.005, 0.005, 0.035, 0.002, 0.007, 0.002, 0.34, 0.001, 0.002, 0.012, 0.007, 0.011, 0.025, 0.006, 0.093, 0.016, 0.003, 0.007, 0.003, 0.008, 0.009, 0.016, 0.009, 0.009, 0.003, 0.0001, 0.0001, 0.038, 0.2, 0.946, 0.502, 0.0001, 0.001, 0.005, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.012, 0.006, 0.045, 0.015, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.003, 0.006, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.003, 0.056, 0.002, 0.001, 0.003, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "eu":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.418, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.177, 0.001, 0.297, 0.0001, 0.0001, 0.006, 0.001, 0.01, 0.167, 0.167, 0.0001, 0.001, 1.097, 0.307, 1.039, 0.006, 0.582, 0.665, 0.539, 0.263, 0.232, 0.207, 0.196, 0.233, 0.193, 0.297, 0.077, 0.037, 0.019, 0.004, 0.019, 0.001, 0.0001, 0.228, 0.197, 0.105, 0.074, 0.177, 0.09, 0.111, 0.131, 0.123, 0.048, 0.077, 0.106, 0.134, 0.065, 0.059, 0.121, 0.005, 0.05, 0.134, 0.08, 0.034, 0.046, 0.019, 0.022, 0.008, 0.029, 0.005, 0.0001, 0.005, 0.0001, 0.002, 0.0001, 11.924, 1.97, 0.229, 2.409, 9.817, 0.3, 1.545, 0.915, 6.874, 0.162, 4.015, 2.508, 1.08, 6.457, 4.385, 0.883, 0.011, 6.261, 2.025, 5.706, 3.55, 0.077, 0.032, 0.337, 0.117, 3.463, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.014, 0.003, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.003, 0.003, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.003, 0.003, 0.0001, 0.001, 0.008, 0.009, 0.002, 0.002, 0.004, 0.001, 0.001, 0.003, 0.009, 0.023, 0.001, 0.012, 0.001, 0.01, 0.001, 0.001, 0.003, 0.012, 0.007, 0.008, 0.006, 0.001, 0.003, 0.001, 0.002, 0.001, 0.004, 0.012, 0.004, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.039, 0.094, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.003, 0.007, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.013, 0.001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ext": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.183, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.144, 0.002, 0.474, 0.0001, 0.0001, 0.008, 0.001, 0.271, 0.191, 0.19, 0.004, 0.002, 1.06, 0.101, 0.854, 0.021, 0.249, 0.293, 0.188, 0.105, 0.088, 0.096, 0.085, 0.084, 0.099, 0.161, 0.072, 0.026, 0.008, 0.003, 0.006, 0.002, 0.0001, 0.241, 0.103, 0.248, 0.095, 0.369, 0.065, 0.091, 0.071, 0.128, 0.047, 0.018, 0.238, 0.161, 0.09, 0.062, 0.171, 0.026, 0.094, 0.188, 0.116, 0.077, 0.083, 0.02, 0.035, 0.009, 0.011, 0.028, 0.0001, 0.029, 0.0001, 0.001, 0.002, 8.822, 0.896, 2.974, 2.338, 7.586, 0.409, 0.951, 0.639, 6.63, 0.18, 0.079, 4.794, 1.966, 5.508, 3.713, 1.742, 0.451, 4.358, 5.625, 3.427, 5.456, 0.664, 0.026, 0.047, 0.265, 0.205, 0.0001, 0.012, 0.0001, 0.001, 0.0001, 0.09, 0.042, 0.016, 0.012, 0.021, 0.01, 0.009, 0.007, 0.019, 0.01, 0.009, 0.002, 0.007, 0.012, 0.003, 0.002, 0.01, 0.006, 0.003, 0.011, 0.012, 0.003, 0.002, 0.002, 0.004, 0.049, 0.003, 0.005, 0.01, 0.009, 0.003, 0.003, 0.02, 0.663, 0.003, 0.009, 0.008, 0.004, 0.004, 0.092, 0.006, 0.332, 0.01, 0.012, 0.009, 0.354, 0.005, 0.01, 0.016, 0.295, 0.019, 0.537, 0.024, 0.015, 0.008, 0.008, 0.011, 0.017, 0.174, 0.02, 0.041, 0.023, 0.01, 0.023, 0.0001, 0.0001, 0.078, 2.453, 0.012, 0.009, 0.0001, 0.0001, 0.0001, 0.019, 0.008, 0.025, 0.005, 0.0001, 0.151, 0.073, 0.021, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.005, 0.034, 0.026, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.021, 0.082, 0.001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ff":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.229, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.756, 0.003, 0.154, 0.0001, 0.0001, 0.002, 0.0001, 0.07, 0.321, 0.324, 0.004, 0.001, 1.19, 0.201, 1.011, 0.039, 0.221, 0.281, 0.197, 0.076, 0.082, 0.099, 0.098, 0.084, 0.101, 0.154, 0.153, 0.028, 0.04, 0.008, 0.04, 0.01, 0.0001, 0.371, 0.159, 0.12, 0.097, 0.095, 0.198, 0.111, 0.12, 0.065, 0.102, 0.267, 0.138, 0.299, 0.201, 0.095, 0.085, 0.014, 0.046, 0.262, 0.159, 0.061, 0.013, 0.059, 0.003, 0.066, 0.008, 0.007, 0.0001, 0.007, 0.0001, 0.006, 0.0001, 10.449, 0.928, 0.343, 3.119, 8.063, 0.727, 1.432, 1.127, 6.432, 0.966, 2.387, 3.274, 2.586, 6.222, 6.89, 0.484, 0.058, 2.623, 1.086, 2.251, 3.239, 0.048, 1.581, 0.013, 1.385, 0.042, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.25, 0.043, 0.0001, 0.008, 0.007, 0.001, 0.007, 0.002, 0.023, 0.003, 0.031, 0.086, 0.001, 0.0001, 0.009, 0.005, 0.007, 0.017, 0.003, 1.378, 0.001, 0.001, 0.0001, 1.485, 0.01, 0.123, 0.001, 0.002, 0.036, 0.035, 0.003, 0.0001, 0.06, 0.009, 0.0001, 0.003, 0.0001, 0.002, 0.02, 0.011, 0.007, 0.04, 0.001, 0.024, 0.001, 0.003, 0.0001, 0.0001, 0.006, 0.154, 0.006, 0.01, 0.135, 0.002, 0.002, 0.009, 0.004, 0.002, 0.004, 0.025, 0.02, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.111, 0.229, 0.005, 0.088, 0.202, 0.0001, 0.0001, 2.86, 0.02, 0.001, 0.001, 0.0001, 0.003, 0.0001, 0.017, 0.011, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.023, 0.044, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.248, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fj":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.647, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.11, 0.005, 0.222, 0.0001, 0.0001, 0.0001, 0.002, 0.182, 0.39, 0.39, 0.002, 0.003, 0.665, 0.202, 1.418, 0.055, 0.382, 0.504, 0.342, 0.179, 0.168, 0.196, 0.159, 0.133, 0.129, 0.164, 0.07, 0.04, 0.02, 0.002, 0.02, 0.013, 0.002, 0.352, 0.212, 0.246, 0.146, 0.319, 0.066, 0.096, 0.061, 0.166, 0.217, 0.277, 0.179, 0.262, 0.29, 0.095, 0.254, 0.022, 0.118, 0.377, 0.355, 0.066, 0.534, 0.043, 0.003, 0.05, 0.01, 0.0001, 0.0001, 0.005, 0.0001, 0.008, 0.0001, 13.891, 0.708, 1.055, 1.505, 4.909, 0.352, 0.936, 0.685, 8.96, 0.075, 2.998, 2.827, 2.182, 5.506, 3.831, 0.546, 0.257, 2.747, 2.722, 3.592, 4.532, 2.046, 0.526, 0.045, 0.71, 0.111, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.267, 0.01, 0.007, 0.005, 0.007, 0.002, 0.0001, 0.002, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.008, 0.0001, 0.0001, 0.013, 0.192, 0.0001, 0.003, 0.002, 0.0001, 0.008, 0.002, 0.0001, 0.023, 0.022, 0.002, 0.0001, 0.007, 0.005, 0.0001, 0.01, 0.003, 0.0001, 0.002, 0.005, 0.002, 0.003, 0.0001, 0.0001, 0.0001, 0.01, 0.0001, 0.0001, 0.008, 0.0001, 0.003, 0.022, 0.003, 0.002, 0.005, 0.0001, 0.008, 0.0001, 0.01, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.013, 0.065, 0.0001, 0.023, 0.002, 0.0001, 0.0001, 0.013, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.002, 0.01, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.26, 0.0001, 0.003, 0.002, 0.0001, 0.003, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.171, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.257, 0.003, 0.223, 0.0001, 0.0001, 0.01, 0.002, 0.015, 0.14, 0.141, 0.001, 0.001, 0.983, 0.292, 1.473, 0.021, 0.562, 0.624, 0.361, 0.197, 0.187, 0.183, 0.177, 0.172, 0.176, 0.285, 0.092, 0.018, 0.015, 0.005, 0.015, 0.002, 0.0001, 0.162, 0.14, 0.076, 0.088, 0.116, 0.204, 0.083, 0.244, 0.058, 0.087, 0.291, 0.108, 0.161, 0.114, 0.065, 0.092, 0.004, 0.082, 0.312, 0.204, 0.061, 0.078, 0.034, 0.003, 0.012, 0.006, 0.004, 0.0001, 0.004, 0.0001, 0.002, 0.0001, 6.488, 0.752, 0.145, 1.557, 3.939, 1.383, 2.415, 1.123, 6.407, 0.628, 2.151, 3.099, 2.563, 5.616, 2.172, 0.663, 0.005, 6.541, 3.536, 4.094, 3.571, 2.164, 0.044, 0.017, 0.862, 0.025, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.054, 0.049, 0.002, 0.002, 0.002, 0.004, 0.003, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.091, 0.001, 0.001, 0.001, 0.001, 0.001, 0.033, 0.002, 0.001, 0.002, 0.001, 0.017, 0.004, 0.006, 0.001, 0.006, 0.009, 0.001, 0.001, 0.01, 0.939, 0.001, 0.001, 0.016, 0.008, 0.277, 0.003, 0.006, 0.007, 0.001, 0.002, 0.001, 1.13, 0.001, 0.004, 1.899, 0.003, 0.003, 0.718, 0.002, 0.002, 0.014, 0.001, 0.801, 0.002, 0.333, 0.003, 0.004, 0.203, 0.003, 0.002, 0.0001, 0.0001, 0.022, 6.504, 0.004, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.012, 0.005, 0.009, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.004, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.053, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "frp": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.788, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.014, 0.012, 0.659, 0.001, 0.0001, 0.001, 0.001, 0.361, 0.368, 0.368, 0.001, 0.0001, 0.743, 0.467, 0.873, 0.02, 0.214, 0.426, 0.274, 0.128, 0.113, 0.117, 0.113, 0.107, 0.116, 0.228, 0.11, 0.019, 0.081, 0.005, 0.081, 0.002, 0.0001, 0.35, 0.279, 0.333, 0.142, 0.141, 0.152, 0.135, 0.066, 0.159, 0.087, 0.033, 0.593, 0.22, 0.099, 0.082, 0.206, 0.019, 0.236, 0.314, 0.121, 0.062, 0.179, 0.013, 0.025, 0.027, 0.009, 0.022, 0.0001, 0.022, 0.0001, 0.006, 0.0001, 6.3, 0.639, 2.237, 2.924, 6.953, 0.549, 0.996, 0.581, 3.639, 0.252, 0.124, 3.838, 1.505, 5.552, 4.982, 1.442, 0.366, 4.363, 4.487, 4.4, 2.763, 0.919, 0.029, 0.168, 0.501, 0.132, 0.0001, 0.008, 0.0001, 0.001, 0.0001, 0.591, 0.012, 0.04, 0.026, 0.003, 0.003, 0.002, 0.002, 0.077, 0.083, 0.002, 0.003, 0.004, 0.003, 0.002, 0.005, 0.007, 0.003, 0.002, 0.023, 0.039, 0.002, 0.001, 0.002, 0.013, 0.56, 0.002, 0.002, 0.004, 0.004, 0.002, 0.004, 0.079, 0.014, 0.761, 0.004, 0.005, 0.003, 0.004, 0.044, 1.724, 0.994, 0.451, 0.049, 0.014, 0.007, 0.008, 0.004, 0.024, 0.005, 0.02, 0.03, 0.411, 0.012, 0.002, 0.176, 0.006, 0.01, 0.014, 0.089, 0.007, 0.007, 0.007, 0.005, 0.0001, 0.0001, 0.277, 4.789, 0.008, 0.018, 0.001, 0.0001, 0.0001, 0.008, 0.004, 0.004, 0.003, 0.001, 0.014, 0.004, 0.075, 0.032, 0.0001, 0.0001, 0.001, 0.005, 0.001, 0.004, 0.007, 0.005, 0.0001, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.004, 0.024, 0.586, 0.003, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "frr": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.212, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.548, 0.003, 0.682, 0.0001, 0.001, 0.008, 0.0001, 0.237, 0.407, 0.407, 0.015, 0.002, 0.738, 0.264, 1.349, 0.032, 0.426, 0.487, 0.285, 0.155, 0.131, 0.142, 0.153, 0.132, 0.154, 0.213, 0.163, 0.033, 0.094, 0.019, 0.094, 0.014, 0.0001, 0.424, 0.235, 0.114, 0.463, 0.142, 0.219, 0.132, 0.243, 0.123, 0.143, 0.217, 0.156, 0.239, 0.202, 0.1, 0.178, 0.008, 0.163, 0.493, 0.169, 0.107, 0.04, 0.158, 0.005, 0.006, 0.018, 0.02, 0.0001, 0.02, 0.0001, 0.015, 0.0001, 7.38, 1.026, 0.694, 2.643, 7.751, 1.48, 1.329, 1.414, 5.143, 0.835, 1.946, 2.506, 1.658, 6.635, 2.847, 0.757, 0.017, 4.866, 3.953, 4.835, 3.559, 0.125, 1.078, 0.025, 0.13, 0.078, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.121, 0.04, 0.005, 0.007, 0.011, 0.01, 0.004, 0.003, 0.014, 0.003, 0.007, 0.003, 0.003, 0.004, 0.004, 0.002, 0.006, 0.041, 0.003, 0.029, 0.004, 0.002, 0.039, 0.002, 0.005, 0.057, 0.003, 0.003, 0.033, 0.001, 0.015, 0.005, 0.043, 0.01, 0.008, 0.004, 0.702, 0.24, 0.006, 0.008, 0.007, 0.041, 0.006, 0.01, 0.003, 0.013, 0.002, 0.004, 0.015, 0.008, 0.014, 0.009, 0.006, 0.008, 0.971, 0.003, 0.022, 0.007, 0.006, 0.005, 0.964, 0.005, 0.004, 0.003, 0.0001, 0.0001, 0.041, 3.039, 0.101, 0.012, 0.0001, 0.001, 0.0001, 0.016, 0.008, 0.014, 0.003, 0.0001, 0.014, 0.006, 0.019, 0.007, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.018, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.024, 0.122, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fur": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.465, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.803, 0.002, 0.385, 0.0001, 0.0001, 0.006, 0.001, 0.135, 0.204, 0.203, 0.001, 0.001, 0.945, 0.084, 1.045, 0.015, 0.262, 0.474, 0.24, 0.16, 0.15, 0.158, 0.149, 0.15, 0.168, 0.219, 0.076, 0.046, 0.024, 0.003, 0.006, 0.002, 0.0001, 0.268, 0.102, 0.337, 0.116, 0.078, 0.115, 0.121, 0.022, 0.278, 0.048, 0.02, 0.218, 0.154, 0.07, 0.05, 0.172, 0.005, 0.086, 0.217, 0.131, 0.073, 0.108, 0.016, 0.024, 0.002, 0.027, 0.004, 0.001, 0.004, 0.0001, 0.016, 0.0001, 6.873, 0.54, 3.119, 3.521, 7.672, 0.855, 0.912, 0.901, 8.131, 0.838, 0.065, 4.486, 1.745, 5.361, 3.491, 1.873, 0.016, 4.269, 4.833, 4.488, 2.566, 1.056, 0.024, 0.012, 0.029, 0.497, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.039, 0.005, 0.002, 0.002, 0.002, 0.001, 0.001, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.007, 0.001, 0.0001, 0.001, 0.0001, 0.002, 0.013, 0.0001, 0.001, 0.008, 0.008, 0.0001, 0.002, 0.187, 0.005, 0.973, 0.001, 0.004, 0.001, 0.001, 0.127, 0.268, 0.009, 0.161, 0.005, 0.069, 0.003, 0.185, 0.001, 0.033, 0.003, 0.05, 0.006, 0.254, 0.001, 0.005, 0.001, 0.001, 0.015, 0.003, 0.208, 0.005, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.042, 2.523, 0.01, 0.009, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.01, 0.005, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.038, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "fy":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.82, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.701, 0.001, 0.398, 0.0001, 0.001, 0.014, 0.003, 0.455, 0.166, 0.166, 0.001, 0.0001, 0.747, 0.192, 0.908, 0.008, 0.277, 0.415, 0.181, 0.098, 0.101, 0.113, 0.107, 0.108, 0.145, 0.219, 0.052, 0.025, 0.005, 0.001, 0.005, 0.002, 0.0001, 0.213, 0.183, 0.091, 0.343, 0.093, 0.196, 0.109, 0.18, 0.193, 0.088, 0.132, 0.122, 0.161, 0.156, 0.11, 0.106, 0.003, 0.108, 0.302, 0.13, 0.038, 0.048, 0.113, 0.006, 0.12, 0.009, 0.007, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 6.027, 1.091, 0.691, 3.439, 11.73, 1.889, 1.306, 1.373, 5.412, 1.009, 2.464, 2.808, 1.837, 7.368, 3.471, 1.074, 0.006, 5.381, 4.264, 5.226, 1.268, 0.226, 1.241, 0.017, 1.595, 0.254, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.043, 0.003, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.007, 0.001, 0.0001, 0.001, 0.001, 0.004, 0.021, 0.0001, 0.001, 0.004, 0.003, 0.001, 0.001, 0.013, 0.004, 0.263, 0.001, 0.008, 0.001, 0.001, 0.002, 0.002, 0.01, 0.258, 0.016, 0.001, 0.003, 0.001, 0.013, 0.005, 0.004, 0.006, 0.003, 0.084, 0.002, 0.006, 0.001, 0.003, 0.002, 0.21, 0.348, 0.006, 0.003, 0.002, 0.001, 0.0001, 0.0001, 0.02, 1.229, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.006, 0.003, 0.015, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.042, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ga":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.234, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.249, 0.002, 0.288, 0.0001, 0.001, 0.013, 0.002, 0.109, 0.15, 0.15, 0.0001, 0.002, 0.872, 0.193, 0.872, 0.017, 0.241, 0.359, 0.187, 0.093, 0.09, 0.096, 0.095, 0.093, 0.117, 0.202, 0.044, 0.013, 0.002, 0.003, 0.002, 0.002, 0.0001, 0.26, 0.338, 0.441, 0.188, 0.097, 0.15, 0.18, 0.066, 0.279, 0.041, 0.036, 0.154, 0.249, 0.121, 0.062, 0.138, 0.005, 0.145, 0.311, 0.272, 0.036, 0.033, 0.04, 0.007, 0.009, 0.007, 0.033, 0.0001, 0.031, 0.0001, 0.002, 0.0001, 11.315, 1.29, 2.859, 2.236, 4.184, 0.692, 2.117, 5.503, 7.212, 0.011, 0.093, 2.991, 1.605, 6.018, 2.868, 0.471, 0.007, 4.409, 3.653, 3.32, 1.715, 0.088, 0.061, 0.021, 0.135, 0.028, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.063, 0.032, 0.004, 0.003, 0.002, 0.001, 0.001, 0.002, 0.002, 0.059, 0.001, 0.002, 0.002, 0.009, 0.001, 0.001, 0.001, 0.001, 0.001, 0.044, 0.003, 0.001, 0.001, 0.001, 0.005, 0.023, 0.008, 0.001, 0.006, 0.006, 0.0001, 0.001, 0.02, 1.278, 0.008, 0.002, 0.005, 0.001, 0.001, 0.002, 0.002, 1.021, 0.001, 0.002, 0.002, 1.343, 0.001, 0.001, 0.006, 0.004, 0.004, 0.674, 0.002, 0.003, 0.005, 0.001, 0.004, 0.002, 0.624, 0.002, 0.005, 0.004, 0.003, 0.002, 0.0001, 0.0001, 0.022, 5.087, 0.004, 0.005, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.011, 0.005, 0.021, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.061, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gag": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.391, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.28, 0.016, 0.1, 0.0001, 0.001, 0.011, 0.0001, 0.023, 0.153, 0.154, 0.0001, 0.0001, 0.918, 0.454, 1.065, 0.029, 0.183, 0.22, 0.131, 0.062, 0.067, 0.072, 0.06, 0.06, 0.062, 0.143, 0.13, 0.023, 0.015, 0.004, 0.015, 0.028, 0.0001, 0.378, 0.403, 0.048, 0.156, 0.135, 0.049, 0.237, 0.117, 0.049, 0.029, 0.415, 0.105, 0.242, 0.108, 0.187, 0.134, 0.003, 0.159, 0.189, 0.383, 0.092, 0.136, 0.005, 0.011, 0.079, 0.04, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 9.932, 1.463, 0.503, 2.949, 4.34, 0.314, 1.11, 0.547, 5.816, 0.052, 2.859, 4.285, 1.983, 5.174, 2.034, 0.659, 0.007, 5.297, 2.304, 2.138, 2.154, 0.741, 0.006, 0.007, 1.825, 1.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.33, 0.02, 0.014, 0.017, 0.012, 0.006, 0.003, 0.09, 0.004, 0.002, 0.005, 0.006, 0.005, 0.012, 0.002, 0.007, 0.002, 0.004, 0.002, 0.034, 0.066, 0.003, 0.036, 0.002, 0.004, 0.038, 0.004, 0.03, 0.138, 0.106, 0.046, 1.073, 0.005, 0.011, 0.07, 0.054, 1.028, 0.001, 0.006, 0.629, 0.003, 0.006, 0.12, 0.006, 0.002, 0.001, 0.004, 0.003, 0.193, 2.957, 0.016, 0.012, 0.008, 0.03, 0.347, 0.009, 0.023, 0.005, 0.018, 0.016, 1.278, 0.014, 0.03, 0.005, 0.0001, 0.0001, 0.012, 3.601, 3.155, 1.149, 0.001, 0.0001, 0.029, 0.03, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.224, 0.106, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.003, 0.019, 0.015, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.005, 0.317, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gan": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.76, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.481, 0.002, 0.018, 0.0001, 0.0001, 0.024, 0.003, 0.008, 0.023, 0.024, 0.002, 0.006, 0.023, 0.038, 0.047, 0.01, 0.315, 0.585, 0.297, 0.204, 0.191, 0.202, 0.185, 0.171, 0.182, 0.259, 0.005, 0.001, 0.007, 0.003, 0.007, 0.002, 0.0001, 0.033, 0.019, 0.032, 0.016, 0.012, 0.012, 0.016, 0.021, 0.015, 0.011, 0.012, 0.015, 0.023, 0.016, 0.01, 0.022, 0.002, 0.018, 0.031, 0.022, 0.006, 0.009, 0.014, 0.002, 0.005, 0.001, 0.008, 0.001, 0.008, 0.0001, 0.002, 0.0001, 0.219, 0.03, 0.061, 0.069, 0.246, 0.023, 0.046, 0.084, 0.187, 0.005, 0.034, 0.116, 0.064, 0.184, 0.17, 0.035, 0.003, 0.158, 0.115, 0.138, 0.085, 0.023, 0.019, 0.007, 0.04, 0.007, 0.001, 0.001, 0.001, 0.0001, 0.0001, 3.244, 1.279, 2.127, 0.643, 0.387, 0.883, 0.435, 0.848, 1.431, 1.201, 0.629, 1.296, 2.258, 1.163, 0.623, 0.808, 0.937, 0.395, 0.26, 0.593, 0.667, 0.608, 1.031, 1.999, 0.578, 0.845, 0.936, 0.665, 1.536, 0.644, 0.439, 0.928, 0.498, 0.603, 0.631, 0.704, 0.585, 0.768, 0.515, 0.538, 0.76, 0.649, 0.365, 0.712, 0.597, 1.095, 0.882, 0.565, 2.328, 1.119, 0.438, 0.543, 1.012, 0.372, 0.441, 0.708, 1.829, 1.205, 1.47, 1.203, 2.219, 1.044, 0.843, 1.251, 0.0001, 0.0001, 0.055, 0.02, 0.005, 0.006, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.002, 0.0001, 0.018, 0.009, 0.031, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.007, 0.036, 0.032, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.029, 0.011, 0.055, 2.062, 3.549, 9.312, 4.838, 3.056, 2.889, 2.67, 0.003, 0.005, 0.013, 0.003, 0.002, 1.772, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gd":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.483, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.829, 0.001, 0.374, 0.0001, 0.0001, 0.027, 0.001, 0.653, 0.225, 0.224, 0.001, 0.001, 0.74, 0.732, 0.959, 0.016, 0.275, 0.512, 0.251, 0.163, 0.143, 0.16, 0.146, 0.151, 0.187, 0.234, 0.126, 0.023, 0.004, 0.003, 0.004, 0.002, 0.0001, 0.399, 0.354, 0.494, 0.195, 0.121, 0.114, 0.226, 0.061, 0.158, 0.033, 0.034, 0.204, 0.239, 0.107, 0.062, 0.151, 0.004, 0.164, 0.477, 0.402, 0.038, 0.033, 0.037, 0.023, 0.009, 0.008, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.001, 13.191, 1.481, 2.674, 2.933, 4.722, 0.55, 2.044, 6.832, 6.396, 0.019, 0.13, 2.757, 1.684, 7.147, 2.433, 0.32, 0.014, 3.962, 3.004, 2.554, 2.054, 0.073, 0.068, 0.016, 0.125, 0.044, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.262, 0.013, 0.005, 0.005, 0.004, 0.003, 0.002, 0.002, 0.031, 0.005, 0.004, 0.001, 0.011, 0.003, 0.002, 0.001, 0.006, 0.003, 0.012, 0.014, 0.004, 0.002, 0.002, 0.002, 0.027, 0.178, 0.003, 0.005, 0.012, 0.011, 0.001, 0.003, 0.677, 0.029, 0.002, 0.003, 0.009, 0.003, 0.004, 0.005, 0.218, 0.029, 0.004, 0.003, 0.303, 0.022, 0.002, 0.003, 0.018, 0.008, 0.323, 0.026, 0.004, 0.004, 0.01, 0.002, 0.006, 0.223, 0.01, 0.003, 0.014, 0.004, 0.005, 0.002, 0.0001, 0.0001, 0.041, 1.912, 0.009, 0.011, 0.0001, 0.0001, 0.0001, 0.018, 0.01, 0.015, 0.003, 0.0001, 0.015, 0.007, 0.02, 0.006, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.008, 0.009, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.004, 0.244, 0.002, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gl":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.812, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.39, 0.002, 0.342, 0.0001, 0.0001, 0.01, 0.001, 0.013, 0.144, 0.144, 0.001, 0.002, 1.02, 0.075, 0.726, 0.01, 0.251, 0.326, 0.181, 0.093, 0.083, 0.092, 0.082, 0.082, 0.102, 0.185, 0.047, 0.021, 0.003, 0.002, 0.002, 0.001, 0.0001, 0.331, 0.122, 0.257, 0.114, 0.192, 0.107, 0.104, 0.065, 0.139, 0.039, 0.03, 0.104, 0.167, 0.127, 0.177, 0.186, 0.007, 0.111, 0.187, 0.12, 0.054, 0.074, 0.026, 0.055, 0.009, 0.01, 0.005, 0.0001, 0.005, 0.0001, 0.003, 0.0001, 9.121, 0.85, 3.271, 4.11, 8.668, 0.721, 0.784, 0.524, 5.185, 0.017, 0.092, 2.548, 2.069, 5.528, 7.673, 1.889, 0.464, 5.046, 5.357, 3.627, 2.8, 0.704, 0.036, 0.564, 0.085, 0.291, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.015, 0.013, 0.003, 0.002, 0.002, 0.001, 0.002, 0.002, 0.001, 0.012, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.006, 0.003, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.003, 0.003, 0.0001, 0.001, 0.028, 0.396, 0.001, 0.002, 0.002, 0.001, 0.001, 0.003, 0.003, 0.383, 0.003, 0.007, 0.001, 0.442, 0.001, 0.001, 0.005, 0.193, 0.006, 0.599, 0.002, 0.002, 0.003, 0.001, 0.003, 0.002, 0.219, 0.007, 0.008, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.05, 2.267, 0.004, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.011, 0.005, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.014, 0.003, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "glk": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.405, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.911, 0.005, 0.048, 0.0001, 0.0001, 0.001, 0.0001, 0.017, 0.104, 0.105, 0.0001, 0.001, 0.019, 0.086, 0.553, 0.019, 0.043, 0.074, 0.037, 0.051, 0.028, 0.037, 0.027, 0.021, 0.025, 0.021, 0.078, 0.0001, 0.005, 0.006, 0.007, 0.001, 0.0001, 0.004, 0.003, 0.008, 0.003, 0.003, 0.002, 0.002, 0.002, 0.003, 0.001, 0.001, 0.002, 0.004, 0.002, 0.001, 0.003, 0.0001, 0.003, 0.005, 0.008, 0.006, 0.001, 0.002, 0.0001, 0.001, 0.001, 0.003, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.177, 0.041, 0.015, 0.061, 0.102, 0.013, 0.028, 0.036, 0.115, 0.015, 0.035, 0.047, 0.057, 0.128, 0.083, 0.024, 0.008, 0.098, 0.063, 0.07, 0.058, 0.027, 0.009, 0.013, 0.021, 0.022, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 0.159, 0.386, 0.313, 0.148, 1.28, 2.09, 3.65, 3.311, 2.644, 0.084, 1.185, 0.003, 3.768, 0.001, 0.042, 0.015, 0.057, 0.007, 0.001, 0.006, 0.005, 0.0001, 0.0001, 0.0001, 0.027, 0.07, 0.004, 0.009, 0.002, 0.001, 0.0001, 0.024, 0.001, 0.005, 0.174, 0.185, 0.526, 0.0001, 0.349, 5.779, 1.561, 0.992, 2.058, 0.045, 0.725, 0.235, 0.5, 2.399, 0.083, 3.048, 0.622, 2.068, 1.214, 0.15, 0.072, 0.14, 0.046, 0.343, 0.079, 0.014, 0.001, 0.0001, 0.271, 0.0001, 0.0001, 0.0001, 0.044, 0.065, 0.002, 0.021, 0.0001, 0.003, 0.0001, 0.068, 0.0001, 0.285, 0.001, 0.0001, 0.001, 0.0001, 0.005, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 21.901, 14.77, 1.833, 3.683, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.141, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.37, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.685, 0.002, 0.175, 0.0001, 0.0001, 0.007, 0.0001, 0.625, 0.171, 0.173, 0.001, 0.002, 1.108, 0.288, 0.925, 0.011, 0.221, 0.306, 0.165, 0.094, 0.084, 0.088, 0.078, 0.08, 0.105, 0.172, 0.107, 0.053, 0.096, 0.001, 0.096, 0.001, 0.0001, 0.314, 0.094, 0.194, 0.053, 0.124, 0.074, 0.125, 0.128, 0.14, 0.097, 0.171, 0.086, 0.202, 0.077, 0.188, 0.312, 0.005, 0.119, 0.136, 0.188, 0.098, 0.072, 0.01, 0.015, 0.073, 0.005, 0.013, 0.0001, 0.013, 0.0001, 0.0001, 0.003, 10.368, 1.134, 1.037, 1.076, 7.653, 0.097, 1.575, 2.931, 4.208, 1.013, 1.951, 0.867, 2.302, 2.015, 5.216, 3.17, 0.036, 4.342, 1.438, 2.887, 4.01, 2.242, 0.013, 0.023, 2.052, 0.103, 0.002, 0.004, 0.0001, 0.0001, 0.0001, 1.036, 0.019, 0.002, 0.069, 0.003, 0.001, 0.001, 0.001, 0.002, 0.002, 0.001, 0.003, 0.001, 0.004, 0.0001, 0.001, 0.002, 0.06, 0.001, 0.011, 0.003, 0.002, 0.001, 0.001, 0.002, 0.862, 0.001, 0.001, 0.08, 0.08, 0.0001, 0.001, 0.04, 0.787, 0.003, 0.72, 0.019, 0.002, 0.003, 0.003, 0.016, 1.383, 0.003, 0.016, 0.003, 0.256, 0.002, 0.013, 0.007, 0.786, 0.011, 0.399, 0.027, 0.172, 0.004, 0.001, 0.003, 0.065, 0.529, 0.066, 0.013, 0.527, 0.003, 0.003, 0.0001, 0.0001, 0.078, 4.545, 0.392, 0.1, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.002, 0.065, 0.0001, 0.004, 0.002, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.015, 0.405, 1.034, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gom": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.459, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.409, 0.004, 0.032, 0.0001, 0.0001, 0.004, 0.001, 0.065, 0.082, 0.086, 0.0001, 0.001, 0.31, 0.092, 0.614, 0.025, 0.037, 0.076, 0.035, 0.025, 0.021, 0.026, 0.02, 0.018, 0.022, 0.033, 0.044, 0.02, 0.0001, 0.003, 0.0001, 0.007, 0.0001, 0.043, 0.023, 0.024, 0.027, 0.017, 0.011, 0.022, 0.028, 0.023, 0.016, 0.018, 0.014, 0.036, 0.017, 0.018, 0.033, 0.001, 0.019, 0.035, 0.046, 0.011, 0.011, 0.005, 0.004, 0.003, 0.004, 0.001, 0.0001, 0.001, 0.0001, 0.007, 0.001, 1.398, 0.134, 0.264, 0.41, 0.83, 0.062, 0.182, 0.613, 0.742, 0.041, 0.372, 0.505, 0.457, 0.862, 0.987, 0.203, 0.002, 0.568, 0.367, 0.732, 0.344, 0.233, 0.034, 0.102, 0.093, 0.096, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 1.014, 0.337, 1.717, 0.03, 0.0001, 0.2, 0.49, 1.091, 0.031, 0.184, 0.021, 0.587, 0.017, 1.766, 0.002, 0.057, 0.002, 0.019, 0.005, 0.016, 0.001, 1.048, 0.146, 0.534, 0.083, 0.03, 0.659, 0.006, 0.381, 0.026, 0.006, 0.26, 0.031, 0.261, 0.004, 0.294, 21.971, 5.237, 0.529, 0.24, 0.912, 0.021, 0.699, 0.107, 0.285, 0.131, 0.613, 1.017, 1.508, 0.008, 1.629, 0.499, 0.008, 0.864, 0.313, 0.067, 0.93, 0.419, 0.0001, 0.006, 0.002, 0.0001, 3.471, 0.661, 0.0001, 0.0001, 0.008, 0.024, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 26.65, 0.0001, 0.078, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "got": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.339, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.094, 0.003, 1.291, 0.0001, 0.0001, 0.0001, 0.0001, 0.038, 0.115, 0.115, 0.004, 0.002, 1.558, 0.264, 1.449, 0.007, 0.147, 0.29, 0.265, 0.261, 0.158, 0.118, 0.082, 0.102, 0.128, 0.101, 0.042, 0.039, 0.006, 0.003, 0.008, 0.017, 0.0001, 0.013, 0.006, 0.028, 0.006, 0.126, 0.004, 0.142, 0.123, 0.192, 0.004, 0.003, 0.01, 0.007, 0.004, 0.248, 0.007, 0.0001, 0.011, 0.012, 0.024, 0.014, 0.037, 0.008, 0.0001, 0.001, 0.001, 0.004, 0.0001, 0.004, 0.0001, 0.0001, 0.001, 1.416, 0.252, 0.277, 0.447, 1.622, 0.345, 0.341, 0.482, 1.005, 0.151, 0.167, 0.523, 0.441, 1.296, 0.895, 0.224, 0.019, 0.998, 0.984, 0.975, 0.495, 0.221, 0.388, 0.018, 0.135, 0.029, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.227, 0.027, 0.943, 1.525, 0.623, 0.417, 0.281, 0.024, 0.043, 0.442, 0.017, 0.001, 12.517, 4.391, 0.0001, 0.004, 16.904, 0.001, 0.002, 0.005, 0.001, 0.003, 0.004, 0.004, 0.0001, 0.003, 0.047, 0.043, 0.003, 0.003, 0.004, 0.002, 1.424, 0.066, 0.105, 0.001, 0.004, 0.001, 0.009, 0.002, 0.017, 0.005, 0.002, 0.005, 0.002, 0.035, 0.001, 0.003, 3.235, 0.309, 0.439, 0.698, 0.617, 0.042, 0.143, 0.379, 0.509, 2.203, 0.495, 0.498, 0.573, 1.215, 0.432, 0.907, 0.0001, 0.0001, 1.432, 0.164, 0.007, 0.004, 0.001, 0.002, 0.0001, 0.004, 0.018, 0.002, 0.024, 0.0001, 0.034, 0.019, 0.033, 0.012, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.003, 0.002, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.003, 0.09, 0.108, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.902, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "gv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.271, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.449, 0.001, 0.421, 0.002, 0.001, 0.012, 0.004, 0.833, 0.218, 0.217, 0.001, 0.004, 1.036, 0.572, 0.962, 0.016, 0.26, 0.327, 0.165, 0.089, 0.084, 0.093, 0.087, 0.088, 0.107, 0.189, 0.087, 0.045, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.277, 0.194, 0.324, 0.123, 0.14, 0.124, 0.191, 0.146, 0.079, 0.067, 0.068, 0.129, 0.207, 0.152, 0.084, 0.138, 0.02, 0.179, 0.402, 0.526, 0.063, 0.198, 0.043, 0.005, 0.079, 0.006, 0.009, 0.0001, 0.009, 0.0001, 0.001, 0.0001, 8.563, 0.792, 1.691, 1.903, 8.594, 0.377, 2.885, 5.368, 3.902, 0.512, 0.598, 3.599, 1.506, 6.663, 4.174, 0.394, 0.032, 4.839, 4.581, 2.625, 1.233, 0.647, 0.279, 0.018, 6.112, 0.053, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.052, 0.016, 0.011, 0.007, 0.006, 0.002, 0.002, 0.033, 0.003, 0.008, 0.001, 0.001, 0.002, 0.004, 0.001, 0.003, 0.004, 0.002, 0.001, 0.008, 0.005, 0.001, 0.002, 0.002, 0.007, 0.032, 0.003, 0.002, 0.002, 0.002, 0.001, 0.001, 0.013, 0.034, 0.001, 0.003, 0.006, 0.002, 0.002, 0.259, 0.003, 0.024, 0.003, 0.005, 0.003, 0.021, 0.001, 0.003, 0.016, 0.006, 0.011, 0.021, 0.003, 0.006, 0.005, 0.006, 0.011, 0.008, 0.015, 0.005, 0.007, 0.005, 0.006, 0.004, 0.0001, 0.0001, 0.024, 0.446, 0.012, 0.021, 0.0001, 0.001, 0.0001, 0.006, 0.003, 0.004, 0.002, 0.0001, 0.012, 0.007, 0.044, 0.019, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.006, 0.05, 0.002, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ha":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.755, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.253, 0.006, 0.093, 0.001, 0.0001, 0.003, 0.001, 0.233, 0.264, 0.267, 0.0001, 0.001, 0.745, 0.202, 0.904, 0.054, 0.25, 0.351, 0.185, 0.101, 0.092, 0.11, 0.102, 0.101, 0.107, 0.226, 0.077, 0.015, 0.009, 0.002, 0.009, 0.006, 0.001, 0.703, 0.295, 0.111, 0.155, 0.055, 0.098, 0.113, 0.13, 0.318, 0.133, 0.225, 0.088, 0.271, 0.163, 0.048, 0.074, 0.005, 0.115, 0.268, 0.173, 0.05, 0.018, 0.079, 0.003, 0.091, 0.042, 0.023, 0.0001, 0.025, 0.0001, 0.021, 0.001, 18.747, 1.651, 0.919, 2.906, 2.679, 0.906, 1.302, 1.831, 6.455, 0.467, 3.514, 2.109, 2.474, 6.749, 1.839, 0.213, 0.023, 4.031, 3.401, 2.21, 3.388, 0.067, 1.617, 0.015, 2.266, 0.447, 0.001, 0.001, 0.003, 0.002, 0.0001, 0.116, 0.007, 0.003, 0.001, 0.009, 0.005, 0.003, 0.003, 0.007, 0.002, 0.01, 0.001, 0.001, 0.0001, 0.003, 0.001, 0.002, 0.002, 0.001, 0.029, 0.002, 0.001, 0.0001, 0.094, 0.018, 0.242, 0.0001, 0.001, 0.01, 0.009, 0.001, 0.004, 0.02, 0.005, 0.002, 0.006, 0.0001, 0.001, 0.003, 0.015, 0.004, 0.013, 0.004, 0.002, 0.002, 0.004, 0.002, 0.003, 0.004, 0.011, 0.001, 0.005, 0.011, 0.003, 0.002, 0.003, 0.002, 0.004, 0.002, 0.001, 0.003, 0.003, 0.0001, 0.002, 0.0001, 0.0001, 0.03, 0.04, 0.008, 0.004, 0.18, 0.0001, 0.0001, 0.118, 0.001, 0.004, 0.0001, 0.0001, 0.001, 0.001, 0.011, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.044, 0.043, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.115, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hak": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.757, 0.001, 0.06, 0.0001, 0.0001, 0.014, 0.0001, 0.007, 0.281, 0.281, 0.0001, 0.001, 0.836, 6.558, 0.681, 0.018, 0.337, 0.407, 0.278, 0.148, 0.134, 0.137, 0.13, 0.123, 0.136, 0.173, 0.065, 0.014, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.079, 0.057, 0.378, 0.035, 0.025, 0.133, 0.042, 0.182, 0.024, 0.017, 0.335, 0.169, 0.185, 0.174, 0.026, 0.169, 0.016, 0.027, 0.334, 0.366, 0.012, 0.069, 0.025, 0.002, 0.166, 0.009, 0.007, 0.0001, 0.007, 0.0001, 0.003, 0.0001, 1.796, 0.086, 1.609, 0.16, 2.657, 0.577, 2.978, 5.312, 4.077, 0.022, 2.986, 0.978, 0.836, 6.046, 1.214, 0.924, 0.006, 0.355, 1.925, 2.85, 1.719, 0.417, 0.063, 0.011, 1.033, 0.05, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.624, 0.347, 0.489, 0.127, 0.115, 0.217, 0.079, 0.128, 0.198, 0.185, 0.153, 0.188, 0.325, 0.981, 0.111, 0.161, 0.125, 0.071, 0.062, 0.072, 0.143, 0.099, 0.164, 0.124, 0.154, 0.164, 0.132, 0.118, 0.284, 0.116, 0.086, 0.138, 0.587, 0.284, 0.798, 0.114, 0.117, 0.11, 0.089, 0.096, 0.66, 0.449, 0.294, 0.11, 0.804, 0.308, 1.169, 0.118, 0.192, 0.187, 0.596, 1.486, 0.608, 0.076, 0.115, 0.115, 0.317, 1.436, 0.679, 1.022, 0.36, 0.128, 0.129, 0.134, 0.0001, 0.0001, 0.018, 7.409, 0.013, 0.036, 0.003, 0.005, 0.0001, 0.003, 0.001, 0.002, 1.194, 0.0001, 0.01, 0.005, 0.045, 0.02, 0.001, 0.001, 0.0001, 0.0001, 0.003, 0.013, 0.006, 0.004, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 1.011, 0.064, 0.254, 0.448, 1.333, 0.785, 0.602, 0.451, 0.439, 0.002, 0.004, 0.008, 0.003, 0.0001, 0.269, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "haw": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.221, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.294, 0.012, 0.203, 0.0001, 0.0001, 0.0001, 0.001, 0.132, 0.34, 0.352, 0.0001, 0.001, 1.505, 0.111, 0.979, 0.007, 0.17, 0.218, 0.129, 0.06, 0.059, 0.065, 0.059, 0.085, 0.093, 0.096, 0.074, 0.017, 0.01, 0.0001, 0.01, 0.007, 0.0001, 0.393, 0.447, 0.582, 0.062, 0.097, 0.065, 0.079, 0.798, 0.153, 0.05, 0.341, 0.594, 0.369, 0.112, 0.254, 0.296, 0.019, 0.112, 0.703, 0.122, 0.065, 0.176, 0.058, 0.005, 0.01, 0.09, 0.005, 0.0001, 0.006, 0.0001, 0.003, 0.006, 12.798, 0.268, 0.355, 0.813, 5.652, 0.089, 0.569, 1.324, 6.125, 0.081, 4.131, 4.145, 2.483, 4.121, 5.223, 1.895, 0.028, 1.627, 1.407, 0.928, 3.376, 0.134, 0.71, 0.014, 0.137, 0.178, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 1.411, 1.393, 0.012, 0.01, 0.008, 0.004, 0.003, 0.003, 0.01, 0.001, 0.006, 0.004, 0.027, 0.239, 0.001, 0.004, 0.011, 0.006, 0.002, 0.111, 0.01, 0.006, 0.002, 0.004, 1.323, 0.019, 0.006, 0.004, 0.007, 0.006, 0.005, 0.004, 0.006, 0.059, 0.005, 0.006, 0.006, 0.004, 0.001, 0.013, 0.01, 0.035, 0.014, 0.268, 0.004, 0.047, 0.003, 0.004, 0.012, 0.061, 0.008, 0.113, 0.006, 0.007, 0.004, 0.005, 0.011, 0.005, 0.014, 1.288, 0.011, 0.01, 0.006, 0.004, 0.0001, 0.0001, 0.011, 0.331, 1.585, 0.461, 0.0001, 0.008, 0.0001, 0.011, 1.285, 0.01, 0.001, 0.0001, 0.031, 0.013, 0.031, 0.011, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.043, 0.02, 0.017, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.013, 1.362, 0.0001, 0.001, 0.006, 0.003, 0.002, 0.001, 0.001, 0.002, 0.007, 0.008, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hif": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.441, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.1, 0.004, 0.114, 0.0001, 0.001, 0.009, 0.004, 0.035, 0.174, 0.173, 0.001, 0.001, 0.931, 0.131, 1.205, 0.021, 0.405, 0.564, 0.33, 0.156, 0.134, 0.134, 0.137, 0.132, 0.161, 0.312, 0.075, 0.009, 0.003, 0.005, 0.003, 0.001, 0.0001, 0.456, 0.322, 0.355, 0.197, 0.18, 0.293, 0.19, 0.17, 0.372, 0.16, 0.181, 0.207, 0.307, 0.247, 0.078, 0.33, 0.012, 0.201, 0.529, 0.239, 0.168, 0.085, 0.095, 0.004, 0.174, 0.017, 0.016, 0.0001, 0.016, 0.0001, 0.019, 0.0001, 12.241, 1.338, 1.486, 1.704, 7.791, 0.593, 1.202, 3.829, 6.515, 0.754, 3.146, 2.684, 2.468, 4.596, 2.829, 0.958, 0.04, 4.362, 3.289, 3.315, 2.328, 0.443, 0.421, 0.04, 0.804, 0.089, 0.0001, 0.003, 0.001, 0.0001, 0.0001, 0.026, 0.089, 0.009, 0.002, 0.003, 0.005, 0.002, 0.014, 0.008, 0.001, 0.001, 0.004, 0.002, 0.02, 0.003, 0.003, 0.003, 0.002, 0.001, 0.042, 0.003, 0.007, 0.001, 0.003, 0.002, 0.005, 0.002, 0.011, 0.004, 0.001, 0.001, 0.013, 0.018, 0.009, 0.001, 0.005, 0.072, 0.024, 0.007, 0.009, 0.007, 0.012, 0.004, 0.029, 0.003, 0.013, 0.009, 0.007, 0.017, 0.022, 0.012, 0.006, 0.004, 0.005, 0.014, 0.002, 0.01, 0.032, 0.005, 0.004, 0.012, 0.003, 0.011, 0.006, 0.0001, 0.0001, 0.028, 0.074, 0.148, 0.035, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.005, 0.003, 0.0001, 0.004, 0.001, 0.016, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.009, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.105, 0.033, 0.022, 0.002, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ho":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.445, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.244, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.681, 0.84, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.042, 0.0001, 0.0001, 0.0001, 0.84, 1.681, 0.0001, 0.84, 0.0001, 0.0001, 1.681, 1.681, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.966, 0.84, 0.0001, 0.0001, 7.563, 0.0001, 0.84, 0.0001, 5.042, 0.0001, 0.0001, 2.521, 1.681, 5.882, 12.605, 1.681, 0.0001, 3.361, 1.681, 1.681, 0.0001, 1.681, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hsb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.885, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.708, 0.001, 0.633, 0.0001, 0.0001, 0.02, 0.001, 0.006, 0.349, 0.351, 0.019, 0.001, 0.617, 0.09, 1.156, 0.027, 0.335, 0.549, 0.233, 0.161, 0.153, 0.173, 0.141, 0.134, 0.184, 0.278, 0.061, 0.039, 0.002, 0.004, 0.002, 0.001, 0.0001, 0.124, 0.184, 0.091, 0.139, 0.058, 0.04, 0.059, 0.112, 0.043, 0.103, 0.189, 0.142, 0.204, 0.143, 0.038, 0.227, 0.002, 0.151, 0.281, 0.088, 0.037, 0.024, 0.263, 0.004, 0.003, 0.079, 0.006, 0.0001, 0.006, 0.0001, 0.001, 0.0001, 6.697, 1.107, 1.636, 2.081, 6.467, 0.188, 0.301, 1.756, 3.527, 3.654, 2.787, 1.99, 1.872, 3.895, 5.864, 1.456, 0.004, 3.313, 3.645, 2.804, 2.176, 0.063, 3.44, 0.018, 1.507, 1.152, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.124, 0.059, 0.706, 0.009, 0.065, 0.003, 0.009, 0.741, 0.003, 0.001, 0.002, 0.003, 0.046, 0.476, 0.001, 0.004, 0.004, 0.003, 0.003, 0.038, 0.003, 0.003, 0.002, 0.005, 0.005, 0.336, 0.005, 1.33, 0.03, 0.004, 0.03, 0.009, 0.055, 0.752, 0.003, 0.003, 0.008, 0.003, 0.002, 0.003, 0.003, 0.008, 0.001, 0.003, 0.001, 0.011, 0.001, 0.003, 0.025, 0.008, 0.015, 0.5, 0.006, 0.013, 0.014, 0.003, 0.014, 0.005, 0.495, 0.011, 0.018, 0.033, 0.697, 0.004, 0.0001, 0.0001, 0.023, 0.573, 2.597, 3.085, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.0001, 0.001, 0.0001, 0.01, 0.005, 0.144, 0.055, 0.001, 0.001, 0.001, 0.011, 0.004, 0.015, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.004, 0.112, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ht":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.728, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.82, 0.005, 0.108, 0.0001, 0.0001, 0.002, 0.001, 0.044, 0.203, 0.204, 0.004, 0.0001, 1.177, 0.16, 1.277, 0.006, 1.128, 0.316, 0.385, 0.25, 0.229, 0.141, 0.146, 0.14, 0.135, 0.218, 0.285, 0.009, 0.002, 0.002, 0.002, 0.073, 0.0001, 0.308, 0.114, 0.31, 0.14, 0.272, 0.082, 0.233, 0.111, 0.459, 0.083, 0.546, 0.457, 0.33, 0.177, 0.124, 0.284, 0.004, 0.107, 0.215, 0.201, 0.017, 0.07, 0.083, 0.002, 0.054, 0.009, 0.003, 0.0001, 0.009, 0.003, 0.007, 0.0001, 8.338, 0.713, 0.405, 1.058, 6.922, 0.493, 1.121, 0.484, 5.37, 0.359, 1.684, 3.389, 1.726, 9.14, 4.981, 1.524, 0.032, 2.005, 4.201, 3.104, 1.485, 1.12, 1.14, 0.06, 3.565, 0.552, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.036, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.007, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.023, 0.0001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.04, 0.015, 0.008, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.815, 0.088, 0.005, 0.004, 0.0001, 0.019, 0.002, 0.004, 0.0001, 0.005, 0.398, 0.011, 0.003, 0.0001, 0.001, 0.0001, 0.001, 0.002, 0.004, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.04, 1.397, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.036, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hy":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.597, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.8, 0.0001, 0.032, 0.0001, 0.0001, 0.007, 0.0001, 0.004, 0.144, 0.144, 0.0001, 0.001, 0.586, 0.166, 0.08, 0.013, 0.201, 0.284, 0.165, 0.087, 0.08, 0.086, 0.077, 0.075, 0.09, 0.155, 0.113, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.01, 0.007, 0.01, 0.005, 0.005, 0.004, 0.004, 0.004, 0.018, 0.002, 0.002, 0.005, 0.008, 0.005, 0.004, 0.006, 0.001, 0.005, 0.011, 0.008, 0.002, 0.006, 0.004, 0.006, 0.001, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.003, 0.016, 0.046, 0.006, 0.015, 0.015, 0.053, 0.008, 0.009, 0.013, 0.038, 0.001, 0.005, 0.027, 0.014, 0.034, 0.04, 0.008, 0.001, 0.036, 0.026, 0.029, 0.018, 0.005, 0.004, 0.002, 0.008, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.414, 0.639, 1.93, 0.109, 0.557, 0.081, 0.132, 0.316, 0.027, 0.402, 0.05, 0.015, 0.042, 0.09, 0.048, 0.034, 0.002, 0.004, 0.002, 0.02, 0.024, 0.021, 0.028, 0.001, 0.0001, 0.001, 0.001, 0.003, 0.001, 0.123, 0.001, 0.0001, 0.042, 6.697, 0.434, 0.595, 0.616, 2.608, 0.304, 0.502, 0.621, 0.7, 0.114, 2.81, 0.856, 0.261, 0.333, 1.634, 0.66, 0.29, 0.576, 0.139, 1.799, 1.257, 4.145, 0.346, 3.356, 0.233, 0.437, 0.354, 0.321, 1.025, 1.123, 1.395, 0.0001, 0.0001, 0.223, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.029, 0.01, 0.0001, 0.0001, 0.652, 36.534, 7.186, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.026, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "hz":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 25.0, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.667, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 8.333, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ia":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.646, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.141, 0.002, 0.329, 0.0001, 0.0001, 0.008, 0.001, 0.021, 0.18, 0.18, 0.001, 0.003, 1.016, 0.166, 0.808, 0.013, 0.184, 0.275, 0.136, 0.08, 0.078, 0.085, 0.074, 0.075, 0.088, 0.135, 0.073, 0.032, 0.002, 0.003, 0.002, 0.002, 0.0001, 0.226, 0.141, 0.296, 0.096, 0.165, 0.073, 0.096, 0.067, 0.304, 0.056, 0.03, 0.329, 0.174, 0.087, 0.056, 0.186, 0.016, 0.106, 0.253, 0.115, 0.079, 0.078, 0.031, 0.017, 0.009, 0.016, 0.018, 0.0001, 0.018, 0.0001, 0.006, 0.0001, 7.783, 0.726, 2.95, 2.701, 11.264, 0.558, 0.916, 0.687, 6.773, 0.133, 0.102, 4.659, 2.226, 5.924, 5.837, 2.221, 0.5, 4.607, 4.777, 5.188, 3.143, 1.186, 0.054, 0.19, 0.205, 0.091, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.033, 0.014, 0.005, 0.003, 0.003, 0.002, 0.001, 0.002, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.001, 0.01, 0.005, 0.001, 0.001, 0.002, 0.003, 0.007, 0.001, 0.002, 0.005, 0.005, 0.001, 0.001, 0.01, 0.02, 0.003, 0.006, 0.005, 0.003, 0.002, 0.005, 0.004, 0.021, 0.003, 0.011, 0.003, 0.018, 0.002, 0.002, 0.005, 0.011, 0.005, 0.019, 0.002, 0.002, 0.005, 0.002, 0.004, 0.005, 0.009, 0.012, 0.006, 0.004, 0.003, 0.005, 0.0001, 0.0001, 0.026, 0.115, 0.014, 0.006, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.019, 0.01, 0.009, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.009, 0.031, 0.001, 0.001, 0.004, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ie":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.521, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.809, 0.001, 0.247, 0.0001, 0.0001, 0.004, 0.006, 0.019, 0.181, 0.18, 0.001, 0.0001, 0.349, 1.199, 1.232, 0.019, 0.563, 0.838, 0.612, 0.225, 0.226, 0.24, 0.207, 0.206, 0.219, 0.382, 0.048, 0.005, 0.006, 0.001, 0.006, 0.0001, 0.0001, 0.309, 0.238, 0.27, 0.166, 0.167, 0.136, 0.184, 0.184, 0.432, 0.068, 0.108, 0.44, 0.245, 0.141, 0.106, 0.224, 0.013, 0.149, 0.37, 0.155, 0.122, 0.083, 0.052, 0.007, 0.021, 0.046, 0.016, 0.0001, 0.017, 0.0001, 0.0001, 0.0001, 6.871, 0.75, 2.348, 2.768, 8.937, 0.584, 0.855, 0.832, 7.829, 0.174, 0.338, 4.504, 1.934, 5.948, 3.473, 1.445, 0.281, 4.186, 4.435, 5.077, 2.881, 0.926, 0.103, 0.2, 0.235, 0.149, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.011, 0.008, 0.006, 0.012, 0.004, 0.002, 0.001, 0.038, 0.001, 0.002, 0.0001, 0.001, 0.004, 0.016, 0.001, 0.001, 0.004, 0.003, 0.001, 0.004, 0.002, 0.001, 0.002, 0.003, 0.002, 0.005, 0.002, 0.003, 0.002, 0.002, 0.002, 0.01, 0.018, 0.107, 0.002, 0.003, 0.013, 0.003, 0.002, 0.004, 0.007, 0.34, 0.002, 0.007, 0.001, 0.079, 0.001, 0.001, 0.011, 0.009, 0.004, 0.067, 0.003, 0.005, 0.02, 0.001, 0.012, 0.001, 0.025, 0.006, 0.034, 0.009, 0.011, 0.002, 0.0001, 0.0001, 0.012, 0.714, 0.064, 0.047, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.007, 0.004, 0.047, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.01, 0.006, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ig":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.656, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.908, 0.0001, 0.773, 0.0001, 0.001, 0.002, 0.004, 0.323, 0.322, 0.321, 0.001, 0.0001, 0.801, 0.348, 0.989, 0.009, 0.349, 0.514, 0.434, 0.211, 0.141, 0.149, 0.14, 0.138, 0.136, 0.236, 0.05, 0.04, 0.002, 0.002, 0.002, 0.002, 0.001, 0.437, 0.139, 0.167, 0.124, 0.161, 0.093, 0.162, 0.08, 0.189, 0.162, 0.069, 0.102, 0.246, 0.453, 0.257, 0.111, 0.006, 0.081, 0.231, 0.102, 0.09, 0.023, 0.104, 0.006, 0.033, 0.007, 0.018, 0.0001, 0.018, 0.0001, 0.003, 0.003, 8.644, 2.249, 1.219, 1.734, 6.313, 0.761, 1.664, 1.979, 4.558, 0.257, 2.436, 1.453, 1.97, 6.1, 4.008, 0.966, 0.02, 3.332, 1.739, 2.408, 2.889, 0.225, 1.517, 0.034, 1.16, 0.258, 0.0001, 0.007, 0.001, 0.0001, 0.0001, 0.13, 0.024, 0.004, 0.004, 0.007, 0.006, 0.001, 0.002, 0.001, 0.004, 0.004, 0.623, 0.118, 0.962, 0.001, 0.0001, 0.002, 0.001, 0.004, 0.022, 0.005, 0.001, 0.031, 0.002, 0.001, 0.082, 0.001, 0.002, 0.004, 0.003, 0.0001, 0.0001, 0.34, 0.233, 0.001, 0.001, 0.012, 1.142, 0.001, 0.003, 0.105, 0.2, 0.002, 0.014, 0.036, 0.183, 0.028, 0.019, 0.004, 0.004, 0.029, 0.095, 0.002, 0.001, 0.009, 0.001, 0.003, 0.143, 0.201, 2.836, 0.03, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.146, 1.356, 0.009, 0.021, 0.0001, 0.014, 0.039, 0.003, 0.0001, 0.001, 0.029, 0.0001, 0.005, 0.001, 0.008, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 2.927, 0.109, 0.003, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ii":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.208, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.146, 0.0001, 1.029, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.343, 0.343, 0.0001, 0.0001, 0.0001, 0.0001, 0.686, 0.0001, 4.803, 0.172, 2.401, 0.0001, 0.0001, 0.172, 0.343, 0.686, 0.686, 0.343, 0.0001, 0.0001, 0.0001, 0.0001, 0.515, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.172, 0.0001, 0.172, 0.0001, 0.0001, 0.0001, 0.172, 0.858, 0.0001, 0.343, 0.343, 0.0001, 0.0001, 0.0001, 0.172, 0.0001, 0.0001, 0.0001, 0.343, 0.343, 0.343, 0.172, 0.0001, 0.172, 0.343, 0.0001, 0.515, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.172, 0.0001, 0.172, 0.0001, 0.0001, 2.573, 1.887, 1.544, 0.858, 2.058, 0.343, 1.715, 0.343, 0.343, 0.0001, 0.172, 1.201, 1.887, 1.029, 1.372, 0.172, 0.343, 1.029, 0.686, 0.172, 0.172, 0.172, 0.686, 0.858, 0.686, 0.172, 0.343, 0.0001, 0.515, 0.172, 0.343, 0.686, 0.343, 0.172, 0.0001, 0.0001, 0.343, 0.0001, 0.172, 1.544, 0.172, 0.343, 0.686, 0.858, 0.686, 0.858, 0.343, 0.686, 0.172, 0.343, 0.343, 0.515, 2.744, 0.172, 0.0001, 0.343, 0.858, 2.916, 0.343, 0.686, 0.343, 0.0001, 0.343, 0.172, 0.0001, 0.0001, 1.372, 0.0001, 0.172, 0.172, 0.0001, 0.0001, 0.0001, 0.172, 0.172, 0.0001, 0.515, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.515, 1.029, 4.631, 1.029, 0.686, 0.0001, 0.858, 10.635, 0.0001, 0.0001, 0.0001, 0.0001, 0.343, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ik":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.089, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.203, 0.0001, 1.053, 0.0001, 0.0001, 0.002, 0.004, 0.019, 1.043, 1.038, 0.0001, 0.004, 0.489, 0.212, 1.246, 0.04, 0.17, 0.264, 0.162, 0.084, 0.065, 0.055, 0.132, 0.065, 0.076, 0.124, 0.136, 0.025, 0.013, 0.002, 0.002, 0.0001, 0.0001, 0.824, 0.109, 0.155, 0.065, 0.052, 0.025, 0.048, 0.076, 0.634, 0.073, 0.409, 0.111, 0.352, 0.545, 0.069, 0.285, 0.321, 0.155, 0.436, 0.755, 0.409, 0.076, 0.046, 0.008, 0.044, 0.002, 0.01, 0.0001, 0.015, 0.0001, 0.008, 0.0001, 9.867, 0.308, 1.051, 0.799, 2.404, 0.327, 1.605, 1.011, 7.743, 0.063, 2.113, 2.725, 1.552, 3.495, 1.668, 1.101, 3.682, 2.526, 3.015, 4.496, 6.747, 0.801, 0.229, 0.065, 0.707, 0.103, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.097, 0.073, 0.183, 0.023, 0.006, 0.031, 0.013, 0.103, 0.05, 0.008, 0.004, 0.952, 0.019, 0.055, 0.002, 0.023, 0.013, 0.008, 0.0001, 0.002, 0.008, 0.067, 0.013, 0.015, 0.004, 0.002, 0.04, 0.01, 0.017, 0.01, 0.002, 0.019, 0.013, 0.843, 0.013, 0.029, 1.206, 0.281, 0.025, 0.006, 0.067, 0.006, 0.055, 0.004, 0.038, 0.017, 0.078, 0.034, 0.172, 0.862, 0.059, 0.01, 0.01, 0.067, 0.008, 0.187, 0.269, 0.134, 0.055, 0.021, 0.038, 0.019, 0.189, 0.046, 0.0001, 0.0001, 0.008, 0.944, 0.835, 1.051, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.006, 0.008, 0.0001, 0.025, 0.023, 0.42, 0.185, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.408, 0.191, 0.013, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ilo": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.301, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.793, 0.111, 0.271, 0.0001, 0.001, 0.008, 0.001, 0.018, 0.172, 0.172, 0.003, 0.003, 0.83, 0.304, 0.649, 0.008, 0.21, 0.325, 0.155, 0.097, 0.09, 0.094, 0.083, 0.085, 0.1, 0.166, 0.048, 0.03, 0.005, 0.001, 0.005, 0.001, 0.0001, 0.349, 0.225, 0.111, 0.217, 0.099, 0.09, 0.118, 0.1, 0.241, 0.035, 0.175, 0.118, 0.232, 0.163, 0.054, 0.279, 0.009, 0.088, 0.233, 0.461, 0.072, 0.035, 0.029, 0.006, 0.014, 0.012, 0.036, 0.0001, 0.036, 0.0001, 0.001, 0.0001, 16.461, 1.586, 0.189, 2.856, 3.734, 0.046, 3.664, 0.306, 10.008, 0.024, 3.303, 2.245, 1.909, 7.066, 3.393, 2.095, 0.012, 2.409, 3.066, 6.078, 2.307, 0.067, 0.549, 0.014, 1.361, 0.048, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.043, 0.006, 0.003, 0.001, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.004, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.033, 0.004, 0.001, 0.001, 0.001, 0.001, 0.004, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.015, 0.007, 0.001, 0.001, 0.003, 0.002, 0.001, 0.003, 0.002, 0.009, 0.002, 0.004, 0.001, 0.006, 0.002, 0.001, 0.013, 0.007, 0.003, 0.007, 0.004, 0.001, 0.003, 0.001, 0.004, 0.002, 0.004, 0.002, 0.003, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.027, 0.049, 0.008, 0.009, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.007, 0.004, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.004, 0.043, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "io":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.24, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.878, 0.001, 0.226, 0.0001, 0.145, 0.683, 0.001, 0.155, 0.19, 0.19, 0.002, 0.001, 1.502, 0.399, 2.049, 0.127, 1.123, 1.116, 0.845, 0.565, 0.593, 0.586, 0.516, 0.395, 0.578, 0.496, 0.056, 0.039, 0.008, 0.001, 0.008, 0.001, 0.0001, 0.262, 0.119, 0.082, 0.063, 0.203, 0.065, 0.07, 0.089, 0.077, 0.042, 0.131, 0.387, 0.144, 0.086, 0.052, 0.185, 0.006, 0.067, 0.238, 0.064, 0.127, 0.06, 0.027, 0.002, 0.018, 0.007, 0.008, 0.0001, 0.008, 0.0001, 0.001, 0.0001, 8.848, 0.82, 0.665, 2.487, 7.044, 0.592, 0.8, 0.82, 7.741, 0.211, 1.575, 3.65, 2.435, 4.587, 5.58, 1.431, 0.273, 4.484, 4.553, 2.971, 2.4, 1.482, 0.052, 0.098, 0.514, 0.499, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.011, 0.008, 0.004, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.003, 0.002, 0.001, 0.0001, 0.002, 0.001, 0.004, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.096, 0.01, 0.001, 0.003, 0.003, 0.001, 0.001, 0.003, 0.002, 0.015, 0.001, 0.004, 0.001, 0.008, 0.001, 0.001, 0.005, 0.003, 0.253, 0.006, 0.002, 0.002, 0.003, 0.001, 0.003, 0.002, 0.004, 0.005, 0.003, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.356, 0.056, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.005, 0.003, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.01, 0.006, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "is":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.97, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.586, 0.001, 0.206, 0.0001, 0.0001, 0.008, 0.002, 0.007, 0.124, 0.124, 0.0001, 0.001, 0.499, 0.123, 1.026, 0.011, 0.247, 0.371, 0.182, 0.092, 0.087, 0.097, 0.09, 0.091, 0.11, 0.206, 0.046, 0.01, 0.008, 0.003, 0.008, 0.001, 0.0001, 0.15, 0.156, 0.07, 0.071, 0.123, 0.128, 0.099, 0.219, 0.049, 0.069, 0.114, 0.102, 0.145, 0.086, 0.033, 0.076, 0.003, 0.091, 0.275, 0.089, 0.03, 0.081, 0.029, 0.009, 0.012, 0.006, 0.003, 0.0001, 0.003, 0.0001, 0.004, 0.0001, 6.834, 0.671, 0.132, 1.35, 4.383, 2.01, 2.575, 1.208, 5.351, 0.788, 2.14, 3.293, 2.549, 5.87, 1.807, 0.609, 0.005, 6.508, 4.164, 3.597, 3.384, 1.444, 0.036, 0.05, 0.678, 0.023, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.079, 0.071, 0.003, 0.002, 0.002, 0.001, 0.004, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.093, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.031, 0.002, 0.001, 0.01, 0.001, 0.001, 0.001, 0.008, 0.001, 0.029, 0.003, 0.162, 0.001, 0.007, 1.147, 0.001, 0.001, 0.003, 0.001, 0.559, 0.002, 0.002, 0.221, 0.001, 0.001, 0.001, 1.194, 0.001, 0.001, 2.653, 0.002, 0.003, 0.801, 0.002, 0.002, 0.588, 0.001, 0.004, 0.002, 0.419, 0.002, 0.005, 0.183, 0.658, 0.001, 0.0001, 0.0001, 0.014, 8.751, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.004, 0.011, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.079, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "iu":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.678, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.59, 0.002, 0.2, 0.001, 0.003, 0.002, 0.003, 0.015, 0.174, 0.173, 0.0001, 0.002, 0.361, 0.227, 0.729, 0.057, 0.148, 0.113, 0.088, 0.041, 0.055, 0.057, 0.038, 0.037, 0.041, 0.075, 0.086, 0.012, 0.034, 0.007, 0.037, 0.002, 0.001, 0.071, 0.015, 0.028, 0.016, 0.027, 0.021, 0.01, 0.09, 0.106, 0.011, 0.032, 0.025, 0.039, 0.045, 0.015, 0.038, 0.016, 0.009, 0.047, 0.052, 0.031, 0.008, 0.018, 0.002, 0.004, 0.0001, 0.042, 0.001, 0.038, 0.0001, 0.005, 0.0001, 3.23, 0.132, 0.269, 0.325, 0.963, 0.137, 0.72, 0.796, 3.179, 0.121, 0.846, 1.083, 0.73, 1.94, 0.65, 0.417, 0.773, 0.909, 0.645, 2.138, 2.24, 0.378, 0.125, 0.018, 0.352, 0.009, 0.063, 0.016, 0.064, 0.0001, 0.0001, 0.195, 0.104, 0.722, 2.277, 0.527, 2.845, 0.283, 0.577, 0.292, 0.16, 1.105, 0.322, 0.08, 0.107, 0.751, 0.183, 5.797, 4.461, 2.284, 5.133, 1.078, 2.99, 2.939, 0.631, 0.063, 0.2, 0.245, 0.066, 0.007, 0.045, 0.004, 0.005, 0.011, 0.073, 0.039, 0.009, 0.018, 0.411, 1.389, 0.092, 0.103, 0.04, 1.113, 0.086, 0.008, 0.439, 0.043, 0.681, 0.098, 0.475, 0.284, 0.202, 0.231, 0.064, 0.012, 0.003, 0.129, 0.126, 0.009, 0.223, 0.007, 0.015, 0.041, 0.125, 0.0001, 0.0001, 0.1, 0.024, 0.004, 0.006, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.004, 0.002, 0.0001, 0.0001, 0.001, 0.131, 0.044, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.009, 0.014, 0.007, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 21.14, 0.181, 0.003, 0.004, 0.004, 0.004, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "jam": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.114, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.204, 0.0001, 0.437, 0.0001, 0.002, 0.015, 0.001, 0.02, 0.246, 0.245, 0.0001, 0.001, 1.286, 0.238, 0.848, 0.011, 0.225, 0.262, 0.144, 0.074, 0.072, 0.083, 0.075, 0.089, 0.083, 0.132, 0.047, 0.038, 0.014, 0.002, 0.014, 0.002, 0.0001, 0.33, 0.18, 0.107, 0.273, 0.095, 0.097, 0.106, 0.027, 0.375, 0.206, 0.243, 0.122, 0.218, 0.135, 0.052, 0.177, 0.004, 0.132, 0.376, 0.091, 0.038, 0.042, 0.116, 0.003, 0.09, 0.009, 0.033, 0.0001, 0.032, 0.0001, 0.0001, 0.0001, 11.751, 1.124, 0.777, 3.113, 4.403, 1.279, 0.882, 3.07, 11.71, 0.554, 2.318, 3.164, 2.271, 5.696, 3.101, 1.655, 0.006, 3.113, 3.636, 3.486, 2.985, 0.575, 1.126, 0.207, 0.616, 0.737, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.054, 0.013, 0.006, 0.006, 0.011, 0.005, 0.006, 0.002, 0.011, 0.004, 0.003, 0.002, 0.005, 0.004, 0.004, 0.001, 0.005, 0.003, 0.003, 0.026, 0.009, 0.002, 0.001, 0.003, 0.001, 0.005, 0.001, 0.003, 0.008, 0.008, 0.001, 0.0001, 0.066, 0.009, 0.001, 0.004, 0.01, 0.005, 0.003, 0.008, 0.009, 0.009, 0.007, 0.005, 0.005, 0.009, 0.003, 0.008, 0.005, 0.014, 0.004, 0.031, 0.002, 0.005, 0.002, 0.003, 0.006, 0.009, 0.008, 0.006, 0.007, 0.007, 0.004, 0.009, 0.0001, 0.0001, 0.069, 0.057, 0.012, 0.006, 0.0001, 0.001, 0.0001, 0.011, 0.004, 0.01, 0.001, 0.0001, 0.049, 0.024, 0.009, 0.004, 0.0001, 0.0001, 0.0001, 0.007, 0.002, 0.004, 0.025, 0.021, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.017, 0.008, 0.051, 0.0001, 0.002, 0.004, 0.003, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "jbo": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 18.83, 0.0001, 0.137, 0.0001, 0.0001, 0.001, 0.0001, 3.634, 0.016, 0.016, 0.0001, 0.0001, 0.047, 0.014, 2.419, 0.008, 0.161, 0.318, 0.2, 0.098, 0.09, 0.087, 0.087, 0.089, 0.108, 0.172, 0.004, 0.001, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.035, 0.022, 0.028, 0.013, 0.017, 0.01, 0.012, 0.013, 0.015, 0.011, 0.012, 0.019, 0.022, 0.019, 0.015, 0.017, 0.001, 0.021, 0.029, 0.019, 0.006, 0.006, 0.008, 0.002, 0.002, 0.004, 0.002, 0.0001, 0.003, 0.0001, 0.003, 0.0001, 7.616, 1.669, 2.73, 1.665, 6.288, 0.906, 1.481, 0.055, 8.607, 1.326, 2.153, 5.868, 2.296, 4.062, 6.049, 1.389, 0.004, 2.945, 3.13, 2.492, 4.934, 0.679, 0.018, 0.589, 0.829, 0.701, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.01, 0.006, 0.006, 0.011, 0.005, 0.003, 0.004, 0.008, 0.003, 0.008, 0.003, 0.003, 0.002, 0.002, 0.002, 0.003, 0.002, 0.001, 0.003, 0.004, 0.004, 0.001, 0.002, 0.001, 0.006, 0.001, 0.002, 0.002, 0.002, 0.001, 0.002, 0.003, 0.006, 0.002, 0.003, 0.005, 0.003, 0.006, 0.013, 0.006, 0.012, 0.005, 0.003, 0.005, 0.008, 0.003, 0.005, 0.01, 0.01, 0.007, 0.008, 0.003, 0.008, 0.002, 0.003, 0.01, 0.005, 0.017, 0.007, 0.005, 0.005, 0.004, 0.003, 0.0001, 0.0001, 0.004, 0.032, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.006, 0.004, 0.004, 0.002, 0.0001, 0.016, 0.008, 0.046, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.007, 0.035, 0.026, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.02, 0.006, 0.007, 0.005, 0.001, 0.004, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "jv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.393, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.972, 0.002, 0.376, 0.0001, 0.001, 0.006, 0.002, 0.021, 0.174, 0.174, 0.0001, 0.002, 1.038, 0.248, 0.884, 0.027, 0.242, 0.269, 0.17, 0.077, 0.072, 0.083, 0.071, 0.073, 0.084, 0.157, 0.063, 0.014, 0.005, 0.004, 0.005, 0.001, 0.0001, 0.298, 0.282, 0.139, 0.175, 0.052, 0.071, 0.122, 0.1, 0.279, 0.193, 0.491, 0.137, 0.261, 0.155, 0.054, 0.404, 0.008, 0.134, 0.426, 0.252, 0.063, 0.039, 0.13, 0.007, 0.04, 0.014, 0.024, 0.0001, 0.024, 0.0001, 0.001, 0.0001, 13.56, 1.271, 0.49, 2.178, 3.499, 0.191, 4.675, 1.646, 6.887, 0.617, 3.65, 2.625, 2.089, 9.349, 2.312, 1.888, 0.011, 3.362, 3.296, 3.086, 3.992, 0.185, 1.217, 0.022, 0.792, 0.054, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.032, 0.007, 0.005, 0.003, 0.005, 0.004, 0.003, 0.003, 0.005, 0.012, 0.002, 0.001, 0.001, 0.004, 0.005, 0.002, 0.003, 0.002, 0.003, 0.01, 0.003, 0.002, 0.001, 0.001, 0.003, 0.007, 0.001, 0.003, 0.006, 0.006, 0.001, 0.001, 0.039, 0.004, 0.002, 0.003, 0.006, 0.005, 0.004, 0.006, 0.531, 1.186, 0.005, 0.003, 0.002, 0.005, 0.002, 0.002, 0.007, 0.006, 0.005, 0.006, 0.002, 0.002, 0.002, 0.001, 0.008, 0.005, 0.003, 0.002, 0.004, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.048, 1.757, 0.01, 0.01, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.007, 0.003, 0.009, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.015, 0.024, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.005, 0.031, 0.001, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ka":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.399, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.467, 0.006, 0.05, 0.004, 0.0001, 0.005, 0.0001, 0.002, 0.08, 0.08, 0.0001, 0.001, 0.389, 0.146, 0.411, 0.004, 0.126, 0.165, 0.095, 0.049, 0.046, 0.05, 0.043, 0.044, 0.053, 0.091, 0.022, 0.014, 0.001, 0.004, 0.001, 0.001, 0.0001, 0.009, 0.008, 0.011, 0.006, 0.004, 0.005, 0.005, 0.004, 0.036, 0.001, 0.002, 0.005, 0.008, 0.005, 0.004, 0.007, 0.001, 0.005, 0.01, 0.01, 0.002, 0.01, 0.003, 0.012, 0.001, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.058, 0.009, 0.021, 0.031, 0.066, 0.012, 0.013, 0.018, 0.047, 0.001, 0.009, 0.032, 0.017, 0.042, 0.05, 0.013, 0.001, 0.044, 0.039, 0.037, 0.025, 0.006, 0.004, 0.003, 0.014, 0.002, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.133, 0.001, 0.001, 30.616, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.203, 1.072, 0.617, 1.27, 2.811, 0.901, 0.251, 0.727, 3.846, 0.492, 1.468, 1.522, 1.35, 1.641, 0.3, 0.022, 1.853, 2.198, 0.578, 0.828, 0.251, 0.28, 0.142, 0.172, 0.523, 0.1, 0.36, 0.118, 0.305, 0.039, 0.412, 0.073, 0.048, 0.001, 0.009, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.002, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.023, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.013, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 30.616, 0.133, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kaa": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.138, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.976, 0.002, 0.25, 0.0001, 0.0001, 0.021, 0.001, 3.483, 0.188, 0.189, 0.0001, 0.005, 0.857, 0.333, 1.07, 0.015, 0.261, 0.313, 0.191, 0.105, 0.096, 0.102, 0.088, 0.083, 0.102, 0.171, 0.072, 0.024, 0.027, 0.002, 0.027, 0.004, 0.0001, 0.278, 0.237, 0.043, 0.064, 0.083, 0.059, 0.068, 0.047, 0.074, 0.082, 0.125, 0.051, 0.155, 0.064, 0.155, 0.081, 0.145, 0.091, 0.217, 0.129, 0.07, 0.048, 0.02, 0.093, 0.029, 0.015, 0.006, 0.0001, 0.006, 0.0001, 0.001, 0.174, 10.672, 1.536, 0.082, 2.608, 4.846, 0.175, 1.712, 1.492, 6.03, 0.848, 1.882, 4.983, 2.176, 5.612, 2.458, 1.187, 1.824, 4.354, 3.602, 3.336, 1.517, 0.264, 0.926, 0.28, 1.912, 0.845, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.223, 0.008, 0.008, 0.008, 0.005, 0.003, 0.002, 0.003, 0.003, 0.001, 0.002, 0.004, 0.002, 0.002, 0.002, 0.003, 0.008, 0.001, 0.002, 0.024, 0.089, 0.003, 0.024, 0.001, 0.007, 0.086, 0.002, 0.002, 0.011, 0.008, 0.005, 0.018, 0.016, 0.006, 0.003, 0.007, 0.007, 0.002, 0.001, 0.006, 0.003, 0.004, 0.002, 0.012, 0.002, 0.006, 0.001, 0.009, 0.142, 4.113, 0.008, 0.007, 0.005, 0.049, 0.005, 0.002, 0.01, 0.003, 0.009, 0.024, 0.009, 0.012, 0.029, 0.003, 0.0001, 0.0001, 0.053, 0.053, 4.15, 0.014, 0.0001, 0.002, 0.0001, 0.001, 0.004, 0.0001, 0.0001, 0.0001, 0.009, 0.003, 0.235, 0.052, 0.004, 0.001, 0.001, 0.003, 0.0001, 0.002, 0.013, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.006, 0.216, 0.0001, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kab": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.63, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.911, 0.026, 0.321, 0.0001, 0.0001, 0.004, 0.001, 0.063, 0.372, 0.372, 0.003, 0.002, 1.155, 1.35, 0.936, 0.038, 0.266, 0.305, 0.202, 0.119, 0.101, 0.107, 0.102, 0.091, 0.121, 0.181, 0.18, 0.035, 0.01, 0.019, 0.017, 0.012, 0.0001, 0.467, 0.145, 0.091, 0.191, 0.038, 0.078, 0.046, 0.038, 0.257, 0.025, 0.056, 0.211, 0.226, 0.079, 0.026, 0.046, 0.014, 0.063, 0.175, 0.468, 0.109, 0.016, 0.087, 0.018, 0.161, 0.062, 0.014, 0.0001, 0.014, 0.0001, 0.076, 0.0001, 9.2, 0.865, 0.672, 3.845, 7.769, 0.911, 1.711, 0.292, 5.622, 0.162, 1.372, 2.877, 2.842, 6.403, 0.445, 0.148, 0.603, 3.37, 3.532, 4.882, 2.937, 0.09, 1.486, 0.181, 2.148, 0.956, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 0.079, 0.004, 0.004, 0.003, 0.012, 0.007, 0.008, 0.003, 0.006, 0.01, 0.01, 0.0001, 0.003, 0.341, 0.007, 0.006, 0.04, 0.002, 0.003, 0.156, 0.017, 0.001, 0.002, 0.001, 0.004, 0.031, 0.007, 0.42, 0.023, 0.016, 0.002, 0.009, 0.025, 0.004, 0.015, 1.13, 0.027, 0.269, 0.019, 0.099, 0.024, 0.093, 0.007, 0.018, 0.013, 0.276, 0.002, 0.011, 0.014, 0.01, 0.006, 0.129, 0.025, 0.113, 0.001, 0.005, 0.586, 0.529, 0.15, 0.021, 0.004, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.056, 0.149, 0.056, 0.01, 0.05, 0.088, 0.0001, 1.3, 0.001, 0.001, 0.003, 0.0001, 0.203, 0.001, 0.007, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.008, 0.063, 0.048, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 1.257, 0.143, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kbd": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.877, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.792, 0.0001, 0.075, 0.0001, 0.001, 0.011, 0.0001, 0.003, 0.141, 0.14, 0.0001, 0.003, 0.771, 0.205, 0.683, 0.004, 0.136, 0.165, 0.088, 0.057, 0.049, 0.062, 0.042, 0.043, 0.05, 0.078, 0.043, 0.017, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.002, 0.001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.304, 0.0001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.003, 0.0001, 0.001, 0.002, 0.001, 0.001, 0.008, 0.001, 0.014, 0.0001, 0.0001, 0.008, 0.0001, 0.008, 0.0001, 0.0001, 0.0001, 0.03, 0.003, 0.008, 0.007, 0.025, 0.002, 0.005, 0.007, 0.023, 0.001, 0.003, 0.195, 0.01, 0.018, 0.016, 0.007, 0.001, 0.016, 0.015, 0.015, 0.012, 0.002, 0.002, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.555, 0.735, 0.993, 2.937, 0.251, 2.403, 0.222, 0.078, 0.401, 1.056, 2.895, 2.991, 0.659, 6.305, 0.005, 0.318, 0.209, 0.078, 0.019, 0.048, 0.196, 0.035, 0.012, 0.035, 0.104, 0.003, 0.214, 0.042, 0.072, 0.044, 0.009, 0.068, 0.044, 0.057, 0.06, 0.06, 0.029, 0.076, 0.012, 0.013, 0.04, 0.033, 0.001, 0.031, 0.001, 0.015, 0.002, 0.026, 2.293, 0.626, 0.124, 1.164, 0.956, 0.819, 0.63, 0.861, 1.412, 0.263, 1.894, 1.024, 2.202, 1.111, 0.541, 0.996, 0.0001, 0.0001, 0.089, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.007, 0.001, 0.005, 0.002, 18.347, 24.31, 0.001, 1.322, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.003, 0.144, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kbp": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.616, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.57, 0.002, 0.071, 0.0001, 0.0001, 0.002, 0.0001, 0.006, 0.116, 0.115, 0.0001, 0.002, 0.667, 0.749, 0.841, 0.003, 0.17, 0.215, 0.112, 0.059, 0.061, 0.065, 0.059, 0.058, 0.083, 0.103, 0.043, 0.02, 0.0001, 0.014, 0.0001, 0.002, 0.0001, 0.165, 0.045, 0.077, 0.029, 0.079, 0.067, 0.032, 0.069, 0.044, 0.029, 0.18, 0.072, 0.11, 0.068, 0.032, 0.297, 0.002, 0.044, 0.127, 0.122, 0.017, 0.016, 0.035, 0.003, 0.036, 0.006, 0.025, 0.0001, 0.025, 0.0001, 0.0001, 0.0001, 8.914, 0.693, 0.409, 0.775, 2.26, 0.236, 0.534, 0.509, 1.986, 0.346, 2.598, 2.297, 1.559, 3.608, 1.061, 1.995, 0.02, 0.616, 1.735, 2.888, 0.861, 0.082, 0.914, 0.015, 2.587, 0.783, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.019, 0.022, 0.002, 0.003, 0.114, 0.001, 0.002, 0.001, 0.002, 0.067, 2.039, 2.33, 0.002, 0.002, 0.001, 0.0001, 0.179, 0.013, 0.001, 0.001, 2.735, 0.001, 1.381, 0.0001, 0.001, 0.007, 0.0001, 5.08, 0.004, 0.003, 0.0001, 0.0001, 0.004, 0.008, 0.005, 1.151, 0.003, 0.001, 0.001, 0.004, 0.013, 4.529, 0.002, 0.019, 0.001, 0.004, 0.003, 0.007, 0.003, 0.207, 0.003, 0.002, 0.003, 0.002, 0.003, 0.001, 0.006, 0.003, 0.002, 0.02, 0.003, 0.002, 0.002, 0.006, 0.0001, 0.0001, 0.035, 0.33, 0.004, 1.069, 0.247, 0.001, 0.0001, 14.815, 3.312, 0.001, 0.128, 0.0001, 0.011, 0.006, 0.009, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.007, 0.013, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kg":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.239, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.886, 0.007, 0.336, 0.0001, 0.0001, 0.012, 0.002, 0.078, 0.324, 0.324, 0.001, 0.002, 0.656, 0.558, 1.416, 0.029, 0.129, 0.236, 0.138, 0.114, 0.084, 0.072, 0.081, 0.089, 0.086, 0.136, 0.151, 0.002, 0.018, 0.006, 0.018, 0.004, 0.0001, 0.545, 0.514, 0.255, 0.205, 0.262, 0.18, 0.131, 0.1, 0.152, 0.059, 0.708, 0.293, 0.752, 0.533, 0.094, 0.176, 0.01, 0.241, 0.354, 0.199, 0.108, 0.106, 0.05, 0.006, 0.156, 0.054, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.001, 12.562, 2.277, 0.331, 1.547, 5.722, 0.691, 1.741, 0.399, 6.386, 0.118, 3.863, 3.599, 2.582, 6.478, 2.883, 0.88, 0.049, 1.355, 2.305, 2.139, 4.827, 0.445, 0.58, 0.051, 3.145, 1.337, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.071, 0.012, 0.003, 0.002, 0.01, 0.004, 0.003, 0.001, 0.004, 0.005, 0.003, 0.002, 0.002, 0.003, 0.001, 0.001, 0.006, 0.001, 0.001, 0.062, 0.006, 0.002, 0.0001, 0.001, 0.001, 0.009, 0.002, 0.004, 0.002, 0.001, 0.001, 0.001, 0.037, 0.129, 0.127, 0.006, 0.006, 0.002, 0.017, 0.018, 0.037, 0.077, 0.018, 0.027, 0.002, 0.034, 0.066, 0.007, 0.011, 0.023, 0.005, 0.025, 0.156, 0.002, 0.001, 0.003, 0.003, 0.002, 0.014, 0.066, 0.013, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.038, 0.798, 0.007, 0.01, 0.0001, 0.0001, 0.0001, 0.013, 0.006, 0.004, 0.0001, 0.001, 0.012, 0.004, 0.009, 0.006, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.021, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.016, 0.007, 0.069, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ki":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.157, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.033, 0.001, 0.1, 0.001, 0.0001, 0.002, 0.005, 0.051, 0.116, 0.115, 0.0001, 0.0001, 0.384, 0.122, 1.505, 0.04, 0.182, 0.215, 0.151, 0.09, 0.071, 0.091, 0.067, 0.064, 0.059, 0.089, 0.065, 0.003, 0.006, 0.002, 0.008, 0.01, 0.0001, 0.273, 0.233, 0.763, 0.125, 0.089, 0.072, 0.168, 0.139, 0.145, 0.105, 0.364, 0.123, 0.376, 0.291, 0.066, 0.138, 0.046, 0.111, 0.252, 0.278, 0.132, 0.045, 0.093, 0.048, 0.079, 0.058, 0.02, 0.0001, 0.02, 0.0001, 0.015, 0.0001, 10.33, 0.786, 1.541, 0.901, 3.961, 0.163, 2.604, 2.426, 8.641, 0.326, 1.916, 0.633, 2.867, 6.576, 3.627, 0.309, 0.068, 4.492, 0.79, 3.938, 2.525, 0.104, 1.397, 0.085, 2.472, 0.26, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.05, 0.003, 0.002, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.037, 0.01, 0.0001, 0.001, 0.003, 0.0001, 0.001, 0.001, 0.013, 0.005, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.06, 4.809, 0.0001, 0.069, 0.002, 0.006, 0.001, 0.004, 0.016, 0.25, 0.0001, 0.009, 0.003, 0.002, 0.0001, 0.009, 0.003, 0.001, 0.003, 0.026, 0.003, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.021, 0.032, 2.884, 2.321, 0.0001, 0.002, 0.0001, 0.002, 0.021, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.009, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.048, 0.0001, 0.0001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kj":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.677, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.71, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.323, 0.323, 0.0001, 0.0001, 0.645, 0.0001, 0.323, 0.0001, 1.613, 0.0001, 0.323, 0.323, 0.0001, 0.0001, 0.645, 0.323, 0.0001, 0.0001, 0.968, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.0001, 0.0001, 0.645, 0.0001, 0.0001, 0.323, 0.0001, 2.903, 0.323, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.097, 3.226, 0.0001, 6.774, 6.774, 1.935, 1.29, 2.903, 10.645, 0.0001, 3.226, 3.226, 6.129, 2.581, 8.065, 0.323, 0.0001, 0.0001, 0.968, 0.323, 1.935, 0.645, 2.581, 0.0001, 0.323, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kk":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.706, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.721, 0.001, 0.107, 0.001, 0.0001, 0.006, 0.0001, 0.002, 0.151, 0.153, 0.001, 0.003, 0.51, 0.246, 0.686, 0.019, 0.221, 0.251, 0.17, 0.104, 0.092, 0.099, 0.09, 0.082, 0.09, 0.149, 0.04, 0.015, 0.002, 0.003, 0.002, 0.002, 0.0001, 0.007, 0.006, 0.025, 0.003, 0.016, 0.005, 0.021, 0.002, 0.024, 0.001, 0.002, 0.003, 0.008, 0.018, 0.004, 0.01, 0.0001, 0.004, 0.021, 0.003, 0.003, 0.004, 0.004, 0.005, 0.001, 0.003, 0.003, 0.0001, 0.003, 0.0001, 0.002, 0.0001, 0.029, 0.005, 0.01, 0.008, 0.028, 0.004, 0.005, 0.007, 0.042, 0.0001, 0.004, 0.014, 0.009, 0.02, 0.023, 0.007, 0.0001, 0.021, 0.015, 0.017, 0.01, 0.003, 0.003, 0.001, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.667, 1.698, 2.42, 0.825, 0.088, 0.097, 0.101, 0.026, 0.573, 0.004, 0.008, 3.51, 0.092, 0.043, 0.029, 0.226, 0.151, 0.105, 0.034, 0.789, 0.175, 0.042, 2.128, 0.009, 0.046, 0.288, 0.211, 1.223, 0.099, 0.041, 0.082, 0.052, 0.093, 0.104, 0.084, 0.646, 0.032, 0.032, 0.003, 0.007, 0.066, 0.363, 0.0001, 0.047, 0.001, 0.014, 0.013, 0.268, 5.447, 1.26, 0.188, 0.541, 1.919, 3.092, 0.668, 0.583, 0.903, 0.603, 1.169, 2.242, 1.309, 3.102, 1.26, 0.548, 0.0001, 0.0001, 0.145, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 25.612, 14.266, 3.356, 0.697, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.197, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kl":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.635, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.811, 0.001, 0.341, 0.0001, 0.0001, 0.006, 0.002, 0.029, 0.34, 0.34, 0.002, 0.001, 0.7, 0.575, 1.063, 0.02, 0.29, 0.457, 0.333, 0.207, 0.125, 0.141, 0.14, 0.129, 0.134, 0.187, 0.161, 0.026, 0.008, 0.005, 0.008, 0.0001, 0.0001, 0.28, 0.09, 0.107, 0.095, 0.075, 0.086, 0.052, 0.078, 0.183, 0.108, 0.243, 0.074, 0.184, 0.254, 0.06, 0.118, 0.079, 0.062, 0.257, 0.166, 0.177, 0.059, 0.032, 0.002, 0.012, 0.015, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 12.711, 0.276, 0.184, 0.432, 3.419, 0.731, 1.669, 0.268, 10.409, 0.217, 2.335, 4.586, 2.81, 6.778, 2.817, 1.735, 2.883, 5.126, 5.68, 6.217, 6.675, 0.652, 0.054, 0.023, 0.144, 0.055, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.031, 0.035, 0.014, 0.011, 0.03, 0.017, 0.013, 0.007, 0.019, 0.003, 0.015, 0.002, 0.011, 0.008, 0.02, 0.004, 0.014, 0.011, 0.007, 0.013, 0.007, 0.01, 0.01, 0.01, 0.008, 0.016, 0.005, 0.005, 0.009, 0.004, 0.004, 0.006, 0.015, 0.021, 0.01, 0.011, 0.03, 0.03, 0.045, 0.038, 0.021, 0.031, 0.02, 0.014, 0.007, 0.018, 0.005, 0.016, 0.035, 0.027, 0.022, 0.024, 0.011, 0.015, 0.016, 0.009, 0.078, 0.018, 0.016, 0.011, 0.01, 0.011, 0.018, 0.012, 0.0001, 0.0001, 0.012, 0.201, 0.031, 0.017, 0.001, 0.001, 0.0001, 0.004, 0.004, 0.004, 0.004, 0.0001, 0.016, 0.008, 0.085, 0.031, 0.0001, 0.001, 0.0001, 0.0001, 0.024, 0.053, 0.13, 0.094, 0.007, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.065, 0.039, 0.022, 0.0001, 0.001, 0.011, 0.007, 0.001, 0.001, 0.003, 0.001, 0.002, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "km":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.234, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.565, 0.004, 0.038, 0.0001, 0.0001, 0.004, 0.0001, 0.009, 0.049, 0.049, 0.0001, 0.001, 0.07, 0.028, 0.072, 0.003, 0.02, 0.022, 0.013, 0.008, 0.007, 0.007, 0.006, 0.006, 0.007, 0.012, 0.008, 0.003, 0.007, 0.012, 0.008, 0.004, 0.0001, 0.018, 0.012, 0.02, 0.008, 0.012, 0.009, 0.007, 0.009, 0.013, 0.004, 0.006, 0.012, 0.012, 0.009, 0.006, 0.011, 0.001, 0.009, 0.018, 0.02, 0.004, 0.004, 0.006, 0.002, 0.002, 0.001, 0.022, 0.0001, 0.022, 0.0001, 0.004, 0.0001, 0.403, 0.068, 0.154, 0.173, 0.554, 0.096, 0.093, 0.2, 0.358, 0.005, 0.025, 0.201, 0.122, 0.348, 0.339, 0.093, 0.005, 0.306, 0.292, 0.378, 0.132, 0.051, 0.059, 0.012, 0.073, 0.006, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.652, 0.801, 0.696, 0.139, 1.351, 0.735, 0.591, 0.836, 0.083, 0.299, 0.563, 1.859, 0.05, 0.041, 0.223, 1.134, 0.273, 0.671, 2.897, 1.707, 1.359, 0.097, 0.57, 0.24, 1.039, 0.72, 1.493, 0.708, 0.482, 0.006, 22.614, 7.802, 0.292, 0.16, 0.416, 0.027, 0.02, 0.041, 0.016, 0.053, 0.015, 0.021, 0.003, 0.006, 0.019, 0.002, 0.004, 0.041, 0.002, 0.021, 0.047, 0.001, 0.001, 0.001, 2.388, 0.829, 0.388, 0.131, 0.053, 0.602, 0.318, 0.199, 0.385, 0.021, 0.0001, 0.0001, 0.017, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 29.306, 1.288, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.22, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.193, 0.001, 0.077, 0.0001, 0.001, 0.006, 0.001, 0.024, 0.05, 0.05, 0.001, 0.001, 0.263, 0.039, 0.387, 0.008, 0.055, 0.048, 0.031, 0.015, 0.013, 0.017, 0.014, 0.014, 0.016, 0.027, 0.012, 0.01, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.007, 0.004, 0.007, 0.004, 0.004, 0.003, 0.002, 0.002, 0.007, 0.001, 0.001, 0.002, 0.005, 0.003, 0.003, 0.004, 0.0001, 0.003, 0.008, 0.004, 0.004, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.005, 0.0001, 0.005, 0.0001, 0.001, 0.001, 0.019, 0.003, 0.007, 0.007, 0.022, 0.004, 0.004, 0.008, 0.016, 0.001, 0.002, 0.009, 0.007, 0.014, 0.015, 0.005, 0.0001, 0.014, 0.012, 0.015, 0.006, 0.002, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.377, 1.744, 1.056, 0.052, 0.0001, 0.294, 1.302, 0.476, 0.14, 0.07, 0.25, 0.184, 0.18, 3.237, 0.115, 0.016, 0.01, 0.0001, 0.076, 0.009, 0.004, 1.075, 0.058, 1.134, 0.019, 0.006, 0.205, 0.005, 0.214, 0.004, 0.012, 0.397, 0.02, 0.439, 0.004, 0.214, 1.341, 0.105, 1.57, 0.184, 1.477, 0.01, 0.553, 0.067, 0.408, 0.14, 0.772, 0.936, 1.909, 0.0001, 24.738, 8.223, 0.0001, 1.147, 0.212, 0.182, 0.975, 0.409, 0.0001, 0.0001, 0.001, 0.0001, 1.605, 2.364, 0.0001, 0.0001, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 31.178, 0.0001, 0.177, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "koi": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.5, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.538, 0.003, 0.105, 0.0001, 0.0001, 0.012, 0.0001, 0.066, 0.298, 0.299, 0.001, 0.003, 0.665, 0.135, 0.828, 0.022, 0.193, 0.238, 0.151, 0.096, 0.069, 0.095, 0.069, 0.062, 0.067, 0.14, 0.09, 0.011, 0.011, 0.003, 0.011, 0.012, 0.0001, 0.012, 0.004, 0.007, 0.003, 0.004, 0.002, 0.002, 0.009, 0.016, 0.003, 0.015, 0.007, 0.012, 0.004, 0.018, 0.015, 0.0001, 0.004, 0.016, 0.008, 0.003, 0.011, 0.001, 0.01, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.204, 0.031, 0.062, 0.036, 0.122, 0.007, 0.019, 0.037, 0.201, 0.009, 0.035, 0.077, 0.03, 0.109, 0.075, 0.025, 0.001, 0.099, 0.08, 0.059, 0.07, 0.013, 0.004, 0.003, 0.013, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.529, 2.707, 1.977, 1.216, 0.076, 0.086, 0.043, 0.37, 0.314, 0.008, 0.015, 2.496, 1.151, 0.356, 0.143, 0.67, 0.146, 0.176, 0.113, 0.201, 0.173, 0.031, 0.44, 0.017, 0.071, 0.03, 0.345, 0.07, 0.14, 0.058, 0.096, 0.178, 0.09, 0.17, 0.07, 0.048, 0.048, 0.034, 0.04, 1.604, 0.03, 0.001, 0.001, 0.041, 0.0001, 0.038, 0.042, 0.02, 3.314, 0.375, 1.556, 0.398, 1.638, 1.43, 1.24, 0.859, 1.906, 0.527, 1.802, 1.787, 1.504, 2.903, 2.273, 0.718, 0.0001, 0.0001, 0.079, 0.905, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.021, 0.0001, 0.001, 0.001, 25.357, 14.367, 0.001, 1.602, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.0001, 0.293, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 25.0, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.667, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 8.333, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 8.333, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "krc": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.633, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.611, 0.001, 0.078, 0.0001, 0.002, 0.011, 0.0001, 0.002, 0.139, 0.14, 0.0001, 0.001, 0.591, 0.21, 0.542, 0.004, 0.138, 0.24, 0.114, 0.076, 0.067, 0.073, 0.058, 0.056, 0.073, 0.12, 0.04, 0.013, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.004, 0.004, 0.006, 0.003, 0.003, 0.002, 0.003, 0.002, 0.03, 0.001, 0.002, 0.003, 0.004, 0.002, 0.002, 0.002, 0.001, 0.002, 0.005, 0.004, 0.001, 0.01, 0.002, 0.017, 0.0001, 0.001, 0.014, 0.0001, 0.014, 0.0001, 0.001, 0.0001, 0.038, 0.009, 0.012, 0.014, 0.044, 0.006, 0.01, 0.013, 0.029, 0.002, 0.006, 0.019, 0.015, 0.026, 0.027, 0.008, 0.001, 0.031, 0.024, 0.024, 0.014, 0.007, 0.003, 0.002, 0.005, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.564, 1.141, 1.579, 1.633, 0.148, 0.303, 0.164, 0.503, 0.456, 0.003, 1.306, 2.454, 0.134, 0.294, 0.686, 0.418, 0.168, 0.334, 0.032, 0.057, 0.2, 0.019, 0.006, 0.011, 0.056, 0.006, 0.17, 0.03, 0.076, 0.03, 0.044, 0.051, 0.143, 0.104, 0.049, 0.03, 0.04, 0.03, 0.005, 0.024, 0.052, 0.001, 0.001, 0.061, 0.001, 0.035, 0.01, 0.009, 6.121, 1.252, 0.294, 1.328, 2.359, 2.412, 0.507, 0.442, 2.779, 0.471, 1.691, 3.418, 1.039, 3.402, 1.301, 0.302, 0.0001, 0.0001, 0.209, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.0001, 0.006, 0.002, 30.423, 13.816, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.14, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ks":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.09, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.116, 0.0001, 0.395, 0.007, 0.0001, 0.0001, 0.004, 0.023, 0.126, 0.124, 0.0001, 0.001, 0.08, 0.153, 0.257, 0.005, 0.042, 0.08, 0.041, 0.04, 0.021, 0.031, 0.018, 0.019, 0.02, 0.048, 0.059, 0.003, 0.053, 0.167, 0.053, 0.0001, 0.0001, 0.005, 0.003, 0.008, 0.007, 0.01, 0.007, 0.001, 0.002, 0.016, 0.003, 0.003, 0.003, 0.004, 0.019, 0.008, 0.002, 0.0001, 0.005, 0.01, 0.013, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.198, 0.016, 0.127, 0.13, 0.17, 0.01, 0.074, 0.033, 0.171, 0.002, 0.016, 0.235, 0.092, 0.256, 0.101, 0.014, 0.0001, 0.218, 0.145, 0.254, 0.031, 0.065, 0.054, 0.0001, 0.012, 0.0001, 0.0001, 0.009, 0.0001, 0.0001, 0.0001, 0.451, 1.562, 0.534, 0.248, 1.154, 1.571, 2.211, 0.263, 1.281, 0.132, 0.341, 0.16, 1.683, 0.702, 0.993, 0.637, 0.623, 0.052, 0.37, 0.043, 0.331, 0.813, 0.313, 0.319, 0.042, 0.007, 0.092, 0.282, 0.326, 0.013, 0.006, 0.065, 0.245, 0.114, 0.179, 0.083, 8.684, 2.577, 0.461, 2.698, 1.217, 0.995, 1.306, 0.114, 0.545, 0.242, 0.666, 1.253, 0.811, 1.543, 0.848, 0.915, 0.434, 0.417, 0.188, 0.11, 0.481, 0.73, 0.156, 0.08, 0.312, 0.004, 2.104, 0.512, 0.0001, 0.0001, 0.188, 0.0001, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.031, 0.0001, 0.008, 0.002, 0.028, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.045, 10.482, 2.72, 3.264, 0.0001, 0.0001, 0.0001, 0.0001, 10.717, 0.001, 0.078, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.129, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ksh": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.3, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.544, 0.007, 0.195, 0.001, 0.0001, 0.006, 0.002, 0.064, 0.077, 0.076, 0.018, 0.0001, 0.951, 0.126, 1.237, 0.01, 0.258, 0.351, 0.176, 0.091, 0.09, 0.099, 0.083, 0.083, 0.116, 0.206, 0.046, 0.013, 0.003, 0.002, 0.003, 0.004, 0.0001, 0.29, 0.361, 0.086, 0.549, 0.218, 0.205, 0.059, 0.258, 0.102, 0.404, 0.343, 0.228, 0.359, 0.191, 0.138, 0.226, 0.009, 0.194, 0.601, 0.11, 0.081, 0.179, 0.232, 0.004, 0.009, 0.121, 0.015, 0.0001, 0.017, 0.0001, 0.132, 0.0001, 4.203, 0.771, 2.07, 4.046, 9.612, 0.707, 0.863, 3.367, 3.374, 1.363, 1.028, 2.762, 2.009, 5.309, 3.906, 0.798, 0.006, 4.302, 3.629, 3.86, 2.108, 1.453, 1.005, 0.049, 0.079, 0.749, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.123, 0.002, 0.002, 0.001, 0.082, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.005, 0.0001, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.005, 0.011, 0.0001, 0.082, 0.036, 0.001, 0.009, 0.0001, 0.001, 0.047, 0.002, 0.044, 0.413, 0.03, 0.004, 0.001, 0.001, 1.721, 0.001, 0.004, 0.002, 0.003, 0.013, 0.001, 0.084, 0.001, 0.002, 0.0001, 0.008, 0.003, 0.001, 0.005, 0.025, 0.002, 0.001, 1.538, 0.002, 0.001, 0.001, 0.002, 0.003, 0.531, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.029, 4.494, 0.051, 0.013, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.124, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ku":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.393, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.949, 0.005, 0.256, 0.0001, 0.0001, 0.006, 0.001, 0.113, 0.21, 0.21, 0.006, 0.003, 0.803, 0.095, 1.083, 0.023, 0.197, 0.26, 0.139, 0.073, 0.067, 0.078, 0.071, 0.068, 0.08, 0.162, 0.084, 0.022, 0.008, 0.003, 0.008, 0.006, 0.0001, 0.192, 0.235, 0.084, 0.253, 0.208, 0.064, 0.107, 0.175, 0.032, 0.081, 0.247, 0.144, 0.22, 0.115, 0.035, 0.144, 0.05, 0.104, 0.203, 0.123, 0.017, 0.026, 0.063, 0.071, 0.053, 0.061, 0.009, 0.0001, 0.009, 0.0001, 0.003, 0.002, 7.122, 1.894, 0.369, 2.998, 7.334, 0.296, 0.895, 1.263, 5.92, 0.932, 2.612, 1.973, 1.601, 5.257, 1.434, 0.604, 0.177, 4.384, 1.643, 2.193, 1.071, 1.195, 1.296, 0.608, 2.23, 0.722, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.075, 0.006, 0.004, 0.007, 0.009, 0.007, 0.012, 0.062, 0.012, 0.002, 0.018, 0.001, 0.015, 0.003, 0.068, 0.002, 0.001, 0.001, 0.002, 0.006, 0.001, 0.004, 0.003, 0.001, 0.007, 0.019, 0.001, 0.009, 0.013, 0.012, 0.074, 0.658, 0.013, 0.005, 0.004, 0.002, 0.003, 0.002, 0.004, 0.335, 0.008, 0.015, 3.587, 0.004, 0.003, 0.008, 2.776, 0.007, 0.009, 0.038, 0.007, 0.007, 0.016, 0.003, 0.007, 0.002, 0.005, 0.004, 0.005, 1.203, 0.019, 0.003, 0.004, 0.002, 0.0001, 0.0001, 0.03, 8.061, 0.042, 0.73, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.01, 0.005, 0.017, 0.006, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.003, 0.063, 0.061, 0.005, 0.012, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.006, 0.073, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.403, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.572, 0.002, 0.112, 0.0001, 0.0001, 0.004, 0.0001, 0.006, 0.253, 0.254, 0.001, 0.001, 0.652, 0.237, 0.796, 0.019, 0.199, 0.362, 0.176, 0.114, 0.094, 0.095, 0.085, 0.09, 0.105, 0.228, 0.059, 0.022, 0.003, 0.001, 0.004, 0.001, 0.0001, 0.008, 0.005, 0.011, 0.005, 0.005, 0.004, 0.005, 0.005, 0.02, 0.004, 0.011, 0.006, 0.006, 0.007, 0.004, 0.005, 0.002, 0.005, 0.011, 0.006, 0.004, 0.008, 0.003, 0.013, 0.003, 0.002, 0.039, 0.0001, 0.039, 0.0001, 0.0001, 0.0001, 0.115, 0.019, 0.016, 0.027, 0.069, 0.009, 0.022, 0.03, 0.155, 0.012, 0.027, 0.043, 0.035, 0.068, 0.048, 0.017, 0.005, 0.048, 0.041, 0.048, 0.044, 0.012, 0.013, 0.005, 0.014, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.292, 3.131, 1.711, 1.163, 0.094, 0.093, 0.068, 0.394, 0.316, 0.013, 0.283, 2.417, 0.996, 0.157, 0.138, 0.933, 0.185, 0.144, 0.161, 0.163, 0.266, 0.043, 0.288, 0.024, 0.072, 0.036, 0.257, 0.079, 0.116, 0.07, 0.051, 0.125, 0.115, 0.273, 0.082, 0.056, 0.037, 0.021, 0.028, 2.085, 0.052, 0.007, 0.009, 0.112, 0.005, 0.032, 0.025, 0.019, 3.353, 0.487, 1.618, 0.507, 1.803, 1.293, 0.994, 0.555, 1.916, 0.692, 1.739, 1.936, 1.41, 2.819, 2.119, 0.641, 0.0001, 0.0001, 0.211, 0.665, 0.015, 0.015, 0.006, 0.004, 0.002, 0.021, 0.019, 0.013, 0.017, 0.003, 0.013, 0.006, 25.09, 14.142, 0.006, 2.075, 0.003, 0.001, 0.001, 0.003, 0.01, 0.008, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.016, 0.023, 0.303, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "kw":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.271, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.698, 0.003, 0.428, 0.0001, 0.0001, 0.066, 0.001, 0.379, 0.501, 0.501, 0.001, 0.001, 1.043, 0.466, 1.444, 0.019, 0.602, 0.995, 0.55, 0.31, 0.288, 0.284, 0.283, 0.29, 0.333, 0.472, 0.076, 0.113, 0.013, 0.003, 0.013, 0.002, 0.0001, 0.413, 0.261, 0.217, 0.211, 0.211, 0.117, 0.24, 0.182, 0.089, 0.078, 0.627, 0.279, 0.264, 0.166, 0.089, 0.271, 0.018, 0.16, 0.497, 0.182, 0.13, 0.12, 0.167, 0.003, 0.325, 0.009, 0.005, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 7.247, 1.102, 0.333, 2.599, 6.963, 0.397, 1.527, 3.863, 2.585, 0.151, 1.745, 2.596, 1.638, 6.936, 4.292, 0.693, 0.011, 4.812, 4.449, 2.907, 1.185, 1.044, 2.706, 0.022, 4.495, 0.043, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.063, 0.013, 0.005, 0.007, 0.005, 0.004, 0.003, 0.003, 0.006, 0.002, 0.002, 0.001, 0.003, 0.005, 0.002, 0.002, 0.003, 0.003, 0.001, 0.033, 0.004, 0.002, 0.002, 0.004, 0.004, 0.02, 0.001, 0.001, 0.003, 0.002, 0.001, 0.002, 0.025, 0.014, 0.004, 0.005, 0.015, 0.004, 0.003, 0.007, 0.004, 0.018, 0.007, 0.005, 0.002, 0.009, 0.005, 0.004, 0.012, 0.007, 0.012, 0.009, 0.008, 0.009, 0.007, 0.003, 0.012, 0.008, 0.009, 0.008, 0.006, 0.007, 0.012, 0.004, 0.0001, 0.0001, 0.028, 0.1, 0.011, 0.012, 0.0001, 0.001, 0.0001, 0.007, 0.003, 0.004, 0.004, 0.0001, 0.012, 0.004, 0.062, 0.02, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.004, 0.013, 0.011, 0.001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.013, 0.007, 0.058, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ky":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.608, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.786, 0.001, 0.076, 0.0001, 0.0001, 0.007, 0.001, 0.001, 0.181, 0.185, 0.0001, 0.003, 0.592, 0.375, 0.793, 0.011, 0.212, 0.26, 0.154, 0.095, 0.087, 0.095, 0.083, 0.081, 0.088, 0.165, 0.05, 0.023, 0.024, 0.003, 0.024, 0.002, 0.0001, 0.006, 0.009, 0.006, 0.003, 0.002, 0.01, 0.002, 0.004, 0.023, 0.001, 0.001, 0.003, 0.004, 0.009, 0.004, 0.011, 0.0001, 0.003, 0.019, 0.005, 0.001, 0.003, 0.002, 0.004, 0.001, 0.0001, 0.007, 0.0001, 0.008, 0.0001, 0.002, 0.0001, 0.034, 0.028, 0.011, 0.01, 0.036, 0.004, 0.007, 0.01, 0.029, 0.001, 0.005, 0.017, 0.009, 0.023, 0.028, 0.008, 0.001, 0.047, 0.02, 0.022, 0.013, 0.003, 0.003, 0.002, 0.006, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 2.775, 1.184, 2.443, 1.907, 0.113, 0.08, 0.128, 0.561, 0.622, 0.004, 0.005, 2.414, 0.086, 0.264, 0.107, 0.368, 0.184, 0.149, 0.029, 0.1, 0.146, 0.009, 0.06, 0.008, 0.039, 0.003, 0.233, 0.023, 0.133, 0.051, 0.082, 0.045, 0.072, 0.109, 0.101, 0.162, 0.039, 0.017, 0.003, 0.031, 0.045, 0.843, 0.0001, 0.06, 0.001, 0.049, 0.011, 1.059, 5.238, 0.934, 0.249, 1.237, 1.665, 2.222, 0.662, 0.522, 2.314, 0.665, 2.431, 2.219, 1.157, 3.498, 1.756, 0.567, 0.0001, 0.0001, 0.135, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 28.842, 12.881, 1.192, 0.856, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.186, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "la":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.703, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.582, 0.002, 0.557, 0.0001, 0.0001, 0.004, 0.001, 0.038, 0.296, 0.296, 0.016, 0.002, 1.029, 0.127, 0.917, 0.01, 0.288, 0.518, 0.368, 0.158, 0.135, 0.172, 0.155, 0.139, 0.169, 0.292, 0.103, 0.058, 0.002, 0.004, 0.002, 0.002, 0.0001, 0.441, 0.179, 0.385, 0.16, 0.131, 0.176, 0.158, 0.144, 0.363, 0.023, 0.04, 0.184, 0.266, 0.121, 0.103, 0.293, 0.049, 0.202, 0.319, 0.152, 0.063, 0.122, 0.033, 0.022, 0.01, 0.013, 0.004, 0.0001, 0.004, 0.0001, 0.004, 0.0001, 7.718, 1.137, 2.983, 1.877, 7.832, 0.566, 0.934, 0.721, 8.862, 0.018, 0.079, 2.703, 3.638, 5.533, 4.661, 1.753, 0.47, 5.095, 5.379, 5.968, 5.347, 0.814, 0.036, 0.291, 0.205, 0.069, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.045, 0.018, 0.011, 0.014, 0.009, 0.005, 0.004, 0.005, 0.004, 0.018, 0.002, 0.002, 0.005, 0.007, 0.002, 0.002, 0.004, 0.003, 0.002, 0.014, 0.007, 0.002, 0.002, 0.002, 0.003, 0.004, 0.003, 0.002, 0.004, 0.003, 0.003, 0.004, 0.013, 0.011, 0.004, 0.003, 0.009, 0.004, 0.004, 0.006, 0.01, 0.02, 0.004, 0.013, 0.003, 0.007, 0.003, 0.005, 0.044, 0.013, 0.009, 0.008, 0.006, 0.012, 0.008, 0.005, 0.014, 0.01, 0.011, 0.011, 0.013, 0.014, 0.01, 0.01, 0.0001, 0.0001, 0.047, 0.083, 0.019, 0.012, 0.0001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.062, 0.03, 0.07, 0.024, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.005, 0.012, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 0.015, 0.037, 0.003, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lad": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.233, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.114, 0.001, 0.334, 0.0001, 0.0001, 0.009, 0.001, 0.032, 0.169, 0.169, 0.0001, 0.0001, 1.028, 0.087, 0.763, 0.008, 0.237, 0.25, 0.147, 0.074, 0.074, 0.086, 0.072, 0.065, 0.078, 0.138, 0.053, 0.043, 0.005, 0.001, 0.005, 0.001, 0.0001, 0.303, 0.15, 0.122, 0.124, 0.422, 0.07, 0.094, 0.073, 0.145, 0.052, 0.173, 0.269, 0.273, 0.076, 0.097, 0.169, 0.01, 0.114, 0.279, 0.178, 0.068, 0.08, 0.015, 0.026, 0.054, 0.024, 0.01, 0.0001, 0.01, 0.0001, 0.0001, 0.0001, 10.092, 0.654, 0.428, 4.329, 9.389, 0.52, 0.701, 0.524, 5.468, 0.466, 2.315, 4.475, 1.912, 5.533, 6.266, 1.56, 0.038, 4.367, 5.784, 3.223, 2.47, 1.069, 0.027, 0.043, 1.039, 0.561, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.02, 0.012, 0.005, 0.005, 0.009, 0.006, 0.004, 0.004, 0.005, 0.003, 0.006, 0.002, 0.002, 0.003, 0.003, 0.001, 0.015, 0.012, 0.004, 0.011, 0.018, 0.019, 0.004, 0.004, 0.007, 0.031, 0.002, 0.006, 0.016, 0.006, 0.01, 0.011, 0.017, 0.14, 0.008, 0.005, 0.006, 0.003, 0.003, 0.02, 0.02, 0.11, 0.013, 0.01, 0.003, 0.09, 0.002, 0.005, 0.012, 0.024, 0.019, 0.137, 0.008, 0.006, 0.005, 0.006, 0.008, 0.006, 0.03, 0.011, 0.011, 0.004, 0.004, 0.004, 0.0001, 0.0001, 0.044, 0.511, 0.018, 0.013, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.017, 0.007, 0.023, 0.009, 0.0001, 0.0001, 0.0001, 0.005, 0.02, 0.199, 0.037, 0.028, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.006, 0.018, 0.0001, 0.001, 0.003, 0.002, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lb":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.412, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.95, 0.002, 0.355, 0.0001, 0.0001, 0.008, 0.002, 0.417, 0.145, 0.146, 0.001, 0.003, 0.802, 0.307, 1.03, 0.016, 0.348, 0.52, 0.266, 0.139, 0.134, 0.143, 0.128, 0.134, 0.162, 0.294, 0.059, 0.012, 0.015, 0.003, 0.015, 0.001, 0.0001, 0.428, 0.324, 0.254, 0.594, 0.233, 0.259, 0.289, 0.233, 0.12, 0.196, 0.27, 0.284, 0.379, 0.192, 0.132, 0.314, 0.012, 0.243, 0.585, 0.165, 0.101, 0.142, 0.167, 0.006, 0.01, 0.098, 0.005, 0.0001, 0.005, 0.0001, 0.003, 0.0001, 4.931, 0.886, 1.95, 2.841, 11.151, 0.974, 2.202, 2.438, 4.449, 0.072, 0.85, 2.736, 2.142, 6.511, 2.976, 0.873, 0.044, 5.369, 4.192, 4.448, 3.418, 0.952, 0.815, 0.087, 0.179, 0.783, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.039, 0.004, 0.003, 0.002, 0.022, 0.001, 0.001, 0.002, 0.001, 0.016, 0.001, 0.02, 0.001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.012, 0.001, 0.0001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.01, 0.002, 0.008, 0.002, 0.053, 0.005, 0.005, 0.001, 0.485, 0.001, 0.003, 0.007, 0.029, 0.959, 0.004, 0.541, 0.001, 0.003, 0.002, 0.002, 0.009, 0.004, 0.006, 0.005, 0.01, 0.003, 0.01, 0.001, 0.004, 0.002, 0.003, 0.005, 0.046, 0.003, 0.003, 0.002, 0.0001, 0.0001, 0.061, 2.169, 0.003, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 0.004, 0.024, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.037, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lbe": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.255, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.9, 0.001, 0.252, 0.0001, 0.0001, 0.001, 0.0001, 0.011, 0.416, 0.416, 0.0001, 0.003, 0.481, 0.136, 0.815, 0.07, 0.265, 0.236, 0.199, 0.107, 0.105, 0.116, 0.098, 0.098, 0.121, 0.12, 0.136, 0.067, 0.071, 0.002, 0.067, 0.006, 0.0001, 0.016, 0.004, 0.021, 0.002, 0.004, 0.005, 0.004, 0.003, 0.485, 0.0001, 0.002, 0.006, 0.012, 0.003, 0.003, 0.014, 0.001, 0.005, 0.011, 0.004, 0.002, 0.006, 0.002, 0.003, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.216, 0.084, 0.071, 0.045, 0.128, 0.01, 0.022, 0.031, 0.155, 0.002, 0.014, 0.09, 0.049, 0.088, 0.086, 0.051, 0.003, 0.174, 0.114, 0.069, 0.102, 0.012, 0.003, 0.009, 0.024, 0.006, 0.001, 0.001, 0.001, 0.001, 0.0001, 3.391, 1.985, 1.311, 3.41, 0.076, 1.237, 0.309, 0.579, 0.377, 0.095, 0.645, 0.087, 1.158, 0.044, 0.125, 0.671, 0.313, 0.089, 0.058, 0.221, 0.212, 0.014, 0.015, 0.044, 0.077, 0.005, 0.185, 0.069, 0.144, 0.054, 0.029, 0.04, 0.037, 0.075, 0.123, 0.038, 0.018, 0.116, 0.052, 0.091, 0.05, 0.027, 0.003, 0.033, 0.004, 0.04, 0.009, 0.029, 7.018, 0.742, 1.169, 0.714, 1.012, 0.485, 0.137, 0.404, 2.976, 0.818, 1.445, 2.805, 1.012, 2.921, 0.476, 0.297, 0.0001, 0.0001, 0.062, 0.008, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 26.245, 14.532, 0.0001, 0.534, 0.0001, 0.001, 0.0001, 0.009, 0.088, 0.067, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.019, 0.318, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lez": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.788, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.917, 0.001, 0.11, 0.0001, 0.0001, 0.014, 0.0001, 0.0001, 0.118, 0.119, 0.0001, 0.001, 0.531, 0.18, 0.599, 0.004, 0.16, 0.227, 0.133, 0.076, 0.063, 0.071, 0.067, 0.062, 0.08, 0.115, 0.048, 0.01, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.003, 0.002, 0.005, 0.001, 0.002, 0.001, 0.001, 0.001, 0.351, 0.0001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.01, 0.001, 0.037, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.023, 0.003, 0.006, 0.005, 0.017, 0.002, 0.004, 0.004, 0.014, 0.001, 0.003, 0.017, 0.004, 0.011, 0.011, 0.005, 0.0001, 0.013, 0.01, 0.009, 0.009, 0.002, 0.001, 0.001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.387, 1.088, 1.449, 2.206, 0.197, 0.805, 0.228, 0.469, 0.264, 0.003, 0.696, 0.03, 1.797, 0.123, 0.075, 0.643, 0.214, 0.062, 0.057, 0.099, 0.243, 0.013, 0.01, 0.016, 0.072, 0.013, 0.197, 0.034, 0.095, 0.027, 0.015, 0.041, 0.182, 0.109, 0.046, 0.053, 0.025, 0.087, 0.024, 0.038, 0.035, 0.001, 0.001, 0.075, 0.001, 0.022, 0.007, 0.017, 6.908, 0.463, 1.591, 1.254, 1.853, 1.815, 0.195, 0.884, 4.344, 1.376, 1.744, 1.947, 0.879, 2.868, 0.597, 0.413, 0.0001, 0.0001, 0.264, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.007, 0.0001, 0.004, 0.002, 30.62, 13.045, 0.0001, 0.248, 0.0001, 0.001, 0.0001, 0.001, 0.004, 0.007, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.149, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lg":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.42, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.857, 0.039, 0.098, 0.0001, 0.0001, 0.008, 0.0001, 0.193, 0.619, 0.652, 0.006, 0.013, 0.576, 0.063, 0.759, 0.031, 0.142, 0.149, 0.106, 0.065, 0.048, 0.064, 0.043, 0.046, 0.035, 0.039, 0.112, 0.029, 0.002, 0.038, 0.003, 0.025, 0.0001, 0.202, 0.147, 0.077, 0.032, 0.406, 0.021, 0.082, 0.019, 0.071, 0.01, 0.184, 0.083, 0.172, 0.138, 0.35, 0.039, 0.002, 0.027, 0.089, 0.063, 0.041, 0.016, 0.06, 0.001, 0.036, 0.019, 0.012, 0.0001, 0.012, 0.0001, 0.01, 0.001, 11.513, 4.158, 0.451, 1.382, 6.569, 0.546, 2.789, 0.349, 6.274, 0.363, 4.548, 2.809, 3.269, 5.614, 5.854, 0.404, 0.013, 2.198, 2.205, 2.706, 6.16, 0.367, 2.254, 0.045, 2.427, 1.395, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 1.181, 0.001, 0.015, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.0001, 0.0001, 0.0001, 0.006, 0.008, 0.842, 0.001, 0.0001, 0.092, 0.085, 0.0001, 0.0001, 0.001, 0.001, 0.113, 0.0001, 0.0001, 0.0001, 0.017, 0.01, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.006, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.181, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.019, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "li":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.944, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.135, 0.002, 0.418, 0.0001, 0.0001, 0.017, 0.001, 1.033, 0.22, 0.22, 0.002, 0.001, 0.717, 0.245, 0.974, 0.02, 0.269, 0.322, 0.176, 0.093, 0.094, 0.096, 0.096, 0.091, 0.103, 0.161, 0.092, 0.054, 0.018, 0.002, 0.018, 0.001, 0.0001, 0.18, 0.177, 0.097, 0.347, 0.099, 0.066, 0.119, 0.148, 0.188, 0.05, 0.105, 0.134, 0.157, 0.158, 0.108, 0.098, 0.003, 0.104, 0.185, 0.093, 0.028, 0.141, 0.105, 0.003, 0.004, 0.083, 0.008, 0.001, 0.008, 0.0001, 0.0001, 0.001, 5.507, 1.139, 0.937, 3.64, 13.741, 0.575, 2.233, 1.264, 5.103, 1.163, 1.751, 2.989, 1.798, 6.008, 4.376, 1.144, 0.011, 4.793, 3.527, 4.666, 1.997, 1.767, 1.153, 0.045, 0.112, 0.704, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.031, 0.005, 0.002, 0.002, 0.002, 0.002, 0.001, 0.001, 0.004, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.01, 0.001, 0.0001, 0.002, 0.0001, 0.007, 0.018, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.002, 0.004, 0.007, 0.003, 0.002, 0.113, 0.003, 0.001, 0.003, 0.424, 0.024, 0.004, 0.246, 0.001, 0.004, 0.001, 0.014, 0.003, 0.003, 0.027, 0.238, 0.005, 0.002, 0.354, 0.001, 0.005, 0.003, 0.003, 0.002, 0.014, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.028, 1.471, 0.005, 0.005, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.001, 0.001, 0.0001, 0.01, 0.004, 0.009, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.031, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lij": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.115, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.653, 0.006, 0.425, 0.0001, 0.0001, 0.003, 0.001, 1.006, 0.211, 0.212, 0.0001, 0.0001, 1.079, 0.522, 0.689, 0.013, 0.183, 0.34, 0.145, 0.099, 0.1, 0.107, 0.089, 0.101, 0.099, 0.127, 0.107, 0.071, 0.08, 0.003, 0.08, 0.006, 0.0001, 0.288, 0.108, 0.216, 0.12, 0.091, 0.089, 0.116, 0.025, 0.235, 0.016, 0.018, 0.145, 0.148, 0.083, 0.128, 0.166, 0.021, 0.11, 0.224, 0.097, 0.053, 0.082, 0.012, 0.025, 0.004, 0.066, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.001, 7.807, 0.66, 2.955, 3.041, 7.727, 0.79, 1.509, 0.643, 7.15, 0.033, 0.062, 2.465, 2.057, 5.516, 6.662, 1.83, 0.232, 3.742, 3.269, 4.498, 2.078, 1.097, 0.032, 0.327, 0.052, 0.447, 0.0001, 0.015, 0.0001, 0.0001, 0.0001, 0.126, 0.011, 0.006, 0.006, 0.007, 0.003, 0.005, 0.015, 0.003, 0.004, 0.003, 0.001, 0.004, 0.006, 0.001, 0.002, 0.001, 0.002, 0.007, 0.108, 0.002, 0.001, 0.003, 0.001, 0.007, 0.097, 0.001, 0.002, 0.005, 0.004, 0.001, 0.001, 0.105, 0.013, 0.443, 0.002, 0.076, 0.002, 0.52, 0.668, 0.246, 0.118, 0.122, 0.032, 0.129, 0.012, 0.108, 0.033, 0.028, 0.081, 0.222, 0.058, 0.152, 0.005, 0.088, 0.002, 0.006, 0.059, 0.022, 0.1, 0.117, 0.007, 0.005, 0.013, 0.0001, 0.0001, 0.059, 3.444, 0.014, 0.118, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.003, 0.0001, 0.026, 0.013, 0.031, 0.013, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.016, 0.012, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.009, 0.005, 0.121, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lmo": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.694, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.003, 0.007, 0.496, 0.0001, 0.002, 0.011, 0.001, 1.536, 0.286, 0.286, 0.0001, 0.001, 1.048, 0.242, 0.905, 0.061, 0.214, 0.291, 0.19, 0.13, 0.124, 0.121, 0.109, 0.107, 0.118, 0.137, 0.12, 0.041, 0.23, 0.036, 0.23, 0.004, 0.0001, 0.256, 0.222, 0.29, 0.092, 0.333, 0.125, 0.138, 0.035, 0.151, 0.022, 0.022, 0.325, 0.213, 0.07, 0.062, 0.237, 0.013, 0.158, 0.284, 0.115, 0.042, 0.131, 0.03, 0.012, 0.005, 0.017, 0.008, 0.0001, 0.008, 0.0001, 0.008, 0.0001, 8.462, 0.843, 2.691, 3.762, 7.44, 0.686, 1.303, 1.109, 4.912, 0.086, 0.225, 4.93, 2.005, 5.17, 2.753, 1.529, 0.12, 4.31, 3.255, 3.626, 1.912, 0.803, 0.038, 0.028, 0.061, 0.501, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.065, 0.004, 0.005, 0.004, 0.003, 0.002, 0.001, 0.001, 0.002, 0.003, 0.001, 0.002, 0.002, 0.003, 0.001, 0.001, 0.001, 0.001, 0.002, 0.011, 0.002, 0.0001, 0.002, 0.001, 0.012, 0.042, 0.001, 0.001, 0.018, 0.002, 0.001, 0.003, 0.883, 0.012, 0.006, 0.001, 0.021, 0.001, 0.002, 0.005, 0.978, 0.311, 0.003, 0.015, 0.376, 0.025, 0.002, 0.002, 0.004, 0.005, 0.393, 0.184, 0.028, 0.003, 0.199, 0.002, 0.003, 0.227, 0.023, 0.007, 0.722, 0.004, 0.002, 0.004, 0.0001, 0.0001, 0.146, 4.287, 0.005, 0.015, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.019, 0.008, 0.013, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.061, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ln":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.397, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.893, 0.03, 0.25, 0.011, 0.001, 0.018, 0.001, 0.058, 0.196, 0.195, 0.019, 0.0001, 0.627, 0.278, 0.997, 0.017, 0.237, 0.316, 0.167, 0.077, 0.074, 0.095, 0.09, 0.07, 0.092, 0.208, 0.084, 0.022, 0.031, 0.028, 0.031, 0.008, 0.0001, 0.272, 0.381, 0.139, 0.08, 0.273, 0.07, 0.073, 0.036, 0.085, 0.047, 0.397, 0.244, 0.485, 0.279, 0.076, 0.136, 0.004, 0.069, 0.216, 0.116, 0.035, 0.048, 0.05, 0.005, 0.052, 0.034, 0.014, 0.018, 0.014, 0.0001, 0.004, 0.0001, 10.636, 2.915, 0.49, 0.988, 4.562, 0.3, 1.532, 0.289, 5.022, 0.059, 3.253, 3.932, 3.872, 5.27, 5.607, 1.218, 0.071, 1.319, 2.651, 2.571, 1.582, 0.27, 0.506, 0.061, 2.255, 1.262, 0.0001, 0.013, 0.0001, 0.0001, 0.0001, 0.045, 0.425, 0.034, 0.001, 0.004, 0.002, 0.016, 0.0001, 0.002, 0.009, 0.003, 0.0001, 0.047, 0.002, 0.03, 0.0001, 0.012, 0.0001, 0.122, 0.011, 0.585, 0.0001, 0.0001, 0.0001, 0.001, 0.025, 0.001, 0.584, 0.003, 0.002, 0.001, 0.0001, 0.21, 1.199, 0.019, 0.009, 0.001, 0.001, 0.002, 0.013, 0.036, 1.009, 0.021, 0.019, 0.002, 0.983, 0.006, 0.015, 0.003, 0.003, 0.005, 0.692, 0.016, 0.003, 0.001, 0.001, 0.001, 0.002, 0.332, 0.02, 0.002, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.228, 4.372, 0.034, 0.004, 0.008, 0.141, 0.0001, 1.137, 0.001, 0.001, 0.5, 0.0001, 0.009, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.014, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.057, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.442, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.079, 0.001, 0.049, 0.0001, 0.0001, 0.006, 0.0001, 0.004, 0.071, 0.071, 0.001, 0.001, 0.152, 0.034, 0.234, 0.012, 0.09, 0.111, 0.08, 0.044, 0.039, 0.045, 0.029, 0.03, 0.029, 0.051, 0.034, 0.006, 0.003, 0.002, 0.003, 0.001, 0.0001, 0.013, 0.008, 0.01, 0.008, 0.006, 0.005, 0.004, 0.005, 0.01, 0.003, 0.004, 0.008, 0.008, 0.007, 0.003, 0.012, 0.0001, 0.005, 0.013, 0.013, 0.003, 0.004, 0.004, 0.001, 0.001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.157, 0.027, 0.063, 0.059, 0.202, 0.033, 0.037, 0.067, 0.14, 0.003, 0.015, 0.078, 0.05, 0.13, 0.139, 0.039, 0.002, 0.117, 0.112, 0.142, 0.055, 0.018, 0.023, 0.006, 0.028, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.694, 1.8, 0.562, 0.336, 0.72, 0.0001, 0.034, 1.352, 1.675, 1.044, 0.455, 0.525, 0.031, 0.727, 0.002, 0.001, 0.005, 0.004, 0.004, 0.01, 1.254, 0.484, 0.162, 0.734, 0.009, 2.294, 0.653, 0.586, 0.209, 0.092, 0.491, 0.038, 0.022, 1.013, 0.148, 0.224, 0.003, 0.796, 0.001, 0.85, 0.016, 0.008, 0.816, 0.509, 0.001, 1.145, 0.216, 0.004, 1.227, 1.202, 2.293, 0.24, 0.573, 0.78, 0.113, 0.39, 1.673, 0.52, 24.114, 5.723, 0.116, 0.133, 0.001, 0.0001, 0.0001, 0.0001, 0.085, 0.006, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.829, 0.002, 0.538, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "lrc": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.503, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.494, 0.003, 0.04, 0.0001, 0.0001, 0.008, 0.0001, 0.002, 0.084, 0.084, 0.002, 0.001, 0.009, 0.028, 0.484, 0.015, 0.026, 0.035, 0.019, 0.017, 0.01, 0.011, 0.009, 0.009, 0.015, 0.017, 0.04, 0.001, 0.009, 0.002, 0.009, 0.0001, 0.0001, 0.006, 0.003, 0.003, 0.006, 0.003, 0.002, 0.001, 0.002, 0.005, 0.001, 0.002, 0.002, 0.003, 0.002, 0.001, 0.003, 0.0001, 0.002, 0.003, 0.003, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.003, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.043, 0.006, 0.016, 0.023, 0.041, 0.004, 0.008, 0.011, 0.044, 0.001, 0.008, 0.022, 0.01, 0.037, 0.028, 0.011, 0.001, 0.026, 0.016, 0.024, 0.015, 0.007, 0.006, 0.002, 0.007, 0.003, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.141, 0.397, 0.205, 0.02, 1.605, 1.614, 3.115, 2.266, 3.47, 0.018, 0.099, 0.005, 5.078, 0.005, 0.021, 0.008, 0.02, 0.003, 0.001, 0.004, 0.007, 0.374, 0.0001, 0.001, 0.02, 0.38, 0.001, 0.042, 0.001, 0.001, 0.001, 0.003, 0.001, 0.003, 0.35, 0.86, 0.5, 0.014, 1.898, 5.042, 0.917, 1.365, 1.804, 0.048, 0.445, 0.131, 0.29, 3.021, 0.146, 3.091, 0.704, 1.565, 1.062, 0.145, 0.062, 0.1, 0.06, 0.212, 0.059, 0.025, 0.002, 0.001, 0.485, 0.001, 0.0001, 0.0001, 0.044, 0.006, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.001, 0.017, 0.007, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.007, 20.669, 13.379, 3.076, 5.814, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.002, 0.138, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ltg": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.505, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.915, 0.002, 0.48, 0.0001, 0.0001, 0.03, 0.001, 0.011, 0.307, 0.306, 0.001, 0.001, 1.036, 0.186, 1.128, 0.03, 0.246, 0.429, 0.198, 0.13, 0.123, 0.148, 0.107, 0.108, 0.116, 0.317, 0.161, 0.072, 0.029, 0.005, 0.029, 0.004, 0.0001, 0.199, 0.123, 0.06, 0.183, 0.07, 0.028, 0.064, 0.021, 0.12, 0.092, 0.204, 0.323, 0.133, 0.101, 0.065, 0.288, 0.001, 0.16, 0.239, 0.103, 0.036, 0.21, 0.007, 0.022, 0.001, 0.052, 0.002, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 7.469, 0.962, 0.824, 2.269, 4.253, 0.17, 1.648, 0.088, 6.306, 1.422, 2.423, 2.524, 1.996, 2.559, 4.514, 1.853, 0.001, 3.554, 6.061, 4.103, 4.999, 1.941, 0.013, 0.006, 1.629, 1.118, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.287, 1.24, 0.018, 0.009, 0.002, 0.008, 0.371, 0.004, 0.003, 0.001, 0.001, 0.004, 0.014, 0.08, 0.001, 0.011, 0.003, 0.002, 0.002, 0.271, 0.142, 0.001, 0.001, 0.003, 0.003, 0.007, 0.001, 0.001, 0.003, 0.032, 0.034, 0.002, 0.043, 0.709, 0.001, 0.005, 0.004, 0.002, 0.0001, 0.0001, 0.001, 0.001, 0.015, 2.24, 0.001, 0.001, 0.0001, 0.001, 0.04, 0.01, 0.024, 0.01, 0.013, 0.028, 0.003, 0.011, 0.033, 0.006, 0.014, 0.026, 0.687, 0.026, 0.226, 0.009, 0.0001, 0.0001, 0.023, 0.015, 3.578, 2.215, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.002, 0.0001, 0.003, 0.001, 0.252, 0.098, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.265, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mai": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.888, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.023, 0.001, 0.03, 0.0001, 0.0001, 0.003, 0.001, 0.013, 0.071, 0.074, 0.0001, 0.001, 0.267, 0.061, 0.074, 0.006, 0.01, 0.016, 0.009, 0.005, 0.004, 0.005, 0.004, 0.004, 0.005, 0.012, 0.021, 0.006, 0.004, 0.001, 0.004, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.018, 0.005, 0.005, 0.005, 0.017, 0.003, 0.004, 0.008, 0.015, 0.001, 0.002, 0.009, 0.005, 0.013, 0.013, 0.004, 0.001, 0.014, 0.017, 0.012, 0.006, 0.002, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.792, 0.705, 0.351, 0.05, 0.0001, 0.548, 0.202, 1.331, 0.277, 0.165, 0.004, 0.356, 0.051, 2.185, 0.0001, 0.286, 0.005, 0.001, 0.0001, 0.066, 0.006, 1.874, 0.183, 0.514, 0.043, 0.102, 0.293, 0.463, 0.567, 0.024, 0.087, 0.255, 0.05, 0.178, 0.022, 0.166, 25.43, 6.866, 0.581, 0.373, 1.476, 0.06, 0.857, 0.137, 0.417, 0.41, 1.258, 0.71, 1.883, 0.001, 1.344, 0.001, 0.001, 0.686, 0.286, 0.227, 1.223, 0.469, 0.0001, 0.0001, 0.026, 0.025, 2.747, 1.736, 0.0001, 0.0001, 0.009, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.668, 0.0001, 0.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mdf": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.974, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.901, 0.002, 0.147, 0.0001, 0.0001, 0.003, 0.0001, 0.003, 0.239, 0.241, 0.0001, 0.001, 0.661, 0.233, 0.828, 0.004, 0.16, 0.227, 0.113, 0.065, 0.054, 0.071, 0.072, 0.058, 0.067, 0.13, 0.047, 0.019, 0.002, 0.0001, 0.002, 0.001, 0.0001, 0.006, 0.002, 0.008, 0.002, 0.002, 0.003, 0.002, 0.002, 0.025, 0.001, 0.002, 0.002, 0.005, 0.002, 0.002, 0.006, 0.001, 0.003, 0.005, 0.003, 0.001, 0.008, 0.001, 0.014, 0.0001, 0.0001, 0.004, 0.0001, 0.005, 0.0001, 0.002, 0.0001, 0.07, 0.006, 0.018, 0.016, 0.05, 0.004, 0.011, 0.014, 0.042, 0.003, 0.009, 0.03, 0.013, 0.041, 0.036, 0.013, 0.001, 0.037, 0.035, 0.028, 0.024, 0.005, 0.003, 0.003, 0.006, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.013, 2.98, 2.587, 0.748, 0.583, 0.414, 0.428, 0.203, 0.631, 0.045, 0.095, 0.17, 2.818, 0.257, 0.113, 1.375, 0.157, 0.181, 0.113, 0.066, 0.125, 0.013, 0.006, 0.022, 0.063, 0.005, 0.16, 0.068, 0.186, 0.053, 0.097, 0.114, 0.073, 0.188, 0.099, 0.03, 0.023, 0.016, 0.014, 0.014, 0.049, 0.003, 0.001, 0.054, 0.002, 0.05, 0.007, 0.022, 4.7, 0.292, 1.108, 0.449, 1.264, 2.755, 0.106, 0.711, 2.236, 0.41, 2.142, 1.743, 1.474, 3.418, 3.1, 0.637, 0.0001, 0.0001, 0.118, 0.006, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.003, 0.002, 0.0001, 0.004, 0.002, 28.205, 15.445, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.006, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.0001, 0.122, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mg":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.132, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.344, 0.0001, 0.051, 0.0001, 0.0001, 0.003, 0.0001, 1.722, 0.134, 0.134, 0.0001, 0.062, 0.6, 1.054, 1.426, 0.011, 0.88, 0.969, 0.776, 0.547, 0.574, 0.473, 0.464, 0.436, 0.531, 0.535, 0.029, 0.033, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.281, 0.132, 0.16, 0.072, 0.212, 0.148, 0.178, 0.056, 0.346, 0.102, 0.053, 0.101, 0.354, 0.788, 0.05, 0.139, 0.008, 0.098, 0.209, 0.172, 0.049, 0.057, 0.038, 0.005, 0.021, 0.009, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 15.071, 0.568, 0.216, 2.816, 2.902, 0.81, 0.249, 1.395, 7.562, 0.225, 1.469, 1.52, 3.108, 9.36, 4.666, 0.931, 0.023, 4.686, 1.843, 3.288, 0.414, 0.748, 0.044, 0.043, 4.297, 0.559, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.076, 0.002, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.002, 0.001, 0.008, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.001, 0.0001, 0.001, 0.052, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.017, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.15, 0.01, 0.007, 0.008, 0.001, 0.0001, 0.001, 0.006, 0.026, 0.088, 0.003, 0.004, 0.001, 0.005, 0.002, 0.002, 0.137, 0.002, 0.001, 0.004, 0.086, 0.001, 0.002, 0.001, 0.003, 0.001, 0.002, 0.01, 0.003, 0.001, 0.001, 0.008, 0.0001, 0.0001, 0.143, 0.408, 0.006, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.069, 0.004, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mh":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.376, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.504, 0.0001, 0.156, 0.0001, 0.0001, 0.0001, 0.039, 0.039, 0.039, 0.039, 0.0001, 0.0001, 1.325, 0.078, 1.247, 0.039, 0.156, 0.039, 0.078, 0.0001, 0.0001, 0.039, 0.0001, 0.0001, 0.039, 0.0001, 0.039, 0.078, 0.078, 0.039, 0.078, 0.0001, 0.0001, 0.701, 0.273, 0.156, 0.078, 0.312, 0.039, 0.156, 0.078, 0.351, 0.779, 0.779, 0.234, 0.779, 0.0001, 0.039, 0.156, 0.0001, 0.312, 0.195, 0.156, 0.195, 0.039, 0.078, 0.0001, 0.195, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.103, 1.558, 0.312, 0.818, 6.584, 0.078, 0.351, 1.013, 7.402, 4.675, 3.584, 3.039, 2.766, 5.804, 6.389, 0.779, 0.078, 4.753, 1.48, 2.337, 1.441, 0.117, 1.597, 0.0001, 1.558, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.351, 0.0001, 0.0001, 0.156, 0.0001, 0.039, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.039, 0.545, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.467, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.078, 0.0001, 0.117, 0.0001, 0.0001, 0.0001, 1.013, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.078, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.208, 0.429, 0.584, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.662, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mhr": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.247, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.433, 0.01, 0.105, 0.0001, 0.0001, 0.003, 0.0001, 0.003, 0.242, 0.243, 0.0001, 0.004, 0.563, 0.341, 0.763, 0.006, 0.23, 0.307, 0.193, 0.103, 0.088, 0.092, 0.076, 0.077, 0.081, 0.164, 0.099, 0.012, 0.003, 0.005, 0.003, 0.006, 0.0001, 0.002, 0.002, 0.003, 0.001, 0.001, 0.002, 0.001, 0.001, 0.045, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.016, 0.001, 0.019, 0.0001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.02, 0.004, 0.007, 0.008, 0.02, 0.002, 0.003, 0.004, 0.014, 0.0001, 0.002, 0.01, 0.005, 0.01, 0.012, 0.004, 0.0001, 0.013, 0.009, 0.01, 0.005, 0.002, 0.002, 0.0001, 0.002, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.161, 0.998, 2.463, 1.262, 0.079, 0.06, 0.073, 0.732, 2.145, 0.012, 0.024, 3.429, 0.167, 0.157, 0.039, 0.3, 0.114, 0.051, 0.084, 0.076, 0.173, 0.021, 0.005, 0.012, 0.07, 0.035, 0.245, 0.039, 0.204, 0.055, 0.073, 0.108, 0.142, 0.124, 0.167, 0.046, 0.023, 0.257, 0.01, 0.146, 0.069, 0.001, 0.001, 0.099, 0.002, 0.093, 0.02, 0.031, 3.766, 0.43, 0.916, 0.689, 1.067, 3.621, 0.573, 0.276, 1.798, 1.177, 2.133, 2.766, 1.884, 2.711, 2.445, 0.765, 0.0001, 0.0001, 0.222, 0.109, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.008, 0.004, 28.363, 13.911, 0.249, 0.424, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.203, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.242, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 18.048, 0.002, 0.114, 0.0001, 0.0001, 0.007, 0.0001, 0.316, 0.24, 0.24, 0.0001, 0.0001, 0.815, 0.729, 1.027, 0.003, 0.15, 0.245, 0.11, 0.069, 0.067, 0.071, 0.069, 0.066, 0.083, 0.097, 0.029, 0.194, 0.002, 0.0001, 0.002, 0.002, 0.0001, 0.243, 0.042, 0.09, 0.013, 0.207, 0.019, 0.023, 0.227, 0.154, 0.011, 0.858, 0.022, 0.414, 0.264, 0.035, 0.344, 0.001, 0.143, 0.039, 1.088, 0.016, 0.015, 0.518, 0.001, 0.002, 0.003, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 10.57, 0.047, 0.232, 0.102, 7.727, 0.029, 1.763, 3.618, 6.701, 0.008, 3.514, 0.582, 0.854, 4.652, 6.133, 0.788, 0.003, 3.052, 0.255, 6.464, 3.231, 0.037, 1.326, 0.008, 0.217, 0.009, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.025, 2.749, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.072, 0.357, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.284, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.004, 0.003, 0.001, 0.001, 0.004, 0.001, 0.003, 0.004, 0.003, 0.003, 0.013, 0.525, 0.001, 0.002, 0.001, 0.002, 0.003, 0.004, 0.018, 0.005, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.019, 0.015, 3.257, 0.759, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.006, 0.008, 0.006, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.002, 0.004, 0.001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "min": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.172, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.612, 0.0001, 0.04, 0.005, 0.0001, 0.002, 0.004, 0.018, 0.155, 0.155, 0.0001, 0.0001, 1.063, 0.022, 1.041, 0.001, 0.404, 0.298, 0.265, 0.112, 0.103, 0.128, 0.132, 0.113, 0.114, 0.233, 0.009, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.635, 0.069, 0.223, 0.216, 0.107, 0.023, 0.035, 0.059, 0.25, 0.026, 0.062, 0.356, 0.142, 0.089, 0.046, 0.143, 0.014, 0.06, 0.402, 0.123, 0.018, 0.017, 0.016, 0.015, 0.037, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.754, 1.953, 0.961, 4.093, 4.246, 0.532, 1.865, 1.575, 6.705, 0.46, 3.68, 3.421, 3.054, 5.905, 5.613, 2.448, 0.009, 4.152, 3.536, 3.358, 3.758, 0.175, 0.156, 0.045, 0.909, 0.044, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.018, 0.016, 0.004, 0.004, 0.011, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.017, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.005, 0.0001, 0.0001, 0.014, 0.007, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.016, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.029, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mk":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.442, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.507, 0.001, 0.094, 0.0001, 0.0001, 0.006, 0.001, 0.012, 0.086, 0.086, 0.001, 0.004, 0.588, 0.074, 0.535, 0.01, 0.197, 0.23, 0.143, 0.089, 0.082, 0.088, 0.076, 0.074, 0.08, 0.116, 0.032, 0.012, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.015, 0.008, 0.047, 0.006, 0.006, 0.005, 0.034, 0.005, 0.026, 0.002, 0.003, 0.006, 0.012, 0.023, 0.006, 0.014, 0.001, 0.007, 0.019, 0.01, 0.006, 0.006, 0.004, 0.004, 0.001, 0.001, 0.008, 0.0001, 0.008, 0.0001, 0.002, 0.0001, 0.08, 0.013, 0.03, 0.022, 0.08, 0.011, 0.022, 0.023, 0.061, 0.003, 0.011, 0.035, 0.039, 0.054, 0.06, 0.012, 0.001, 0.056, 0.049, 0.047, 0.027, 0.008, 0.006, 0.003, 0.012, 0.004, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.279, 1.922, 3.072, 0.896, 0.157, 0.085, 0.296, 0.344, 0.32, 0.001, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.012, 0.067, 0.066, 0.1, 0.11, 0.062, 0.046, 0.008, 0.029, 0.825, 0.009, 0.229, 0.032, 0.208, 0.077, 0.103, 0.118, 0.054, 0.125, 0.063, 0.016, 0.028, 0.03, 0.013, 0.01, 0.018, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 5.692, 0.585, 1.752, 0.746, 1.619, 3.647, 0.195, 0.665, 3.964, 0.001, 1.64, 1.494, 0.888, 3.068, 4.767, 1.117, 0.0001, 0.0001, 0.015, 0.006, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.003, 33.101, 10.345, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.096, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ml":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.283, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.554, 0.001, 0.034, 0.0001, 0.0001, 0.002, 0.0001, 0.013, 0.046, 0.046, 0.0001, 0.001, 0.155, 0.051, 0.434, 0.004, 0.069, 0.096, 0.051, 0.026, 0.025, 0.029, 0.025, 0.024, 0.03, 0.054, 0.011, 0.004, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.005, 0.003, 0.005, 0.002, 0.002, 0.002, 0.002, 0.002, 0.004, 0.001, 0.001, 0.002, 0.003, 0.002, 0.002, 0.004, 0.0001, 0.002, 0.005, 0.004, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.003, 0.0001, 0.001, 0.0001, 0.044, 0.007, 0.016, 0.014, 0.045, 0.007, 0.009, 0.015, 0.036, 0.001, 0.004, 0.022, 0.013, 0.031, 0.031, 0.01, 0.001, 0.031, 0.025, 0.029, 0.015, 0.004, 0.005, 0.002, 0.008, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.284, 1.637, 0.889, 0.045, 0.0001, 0.237, 0.843, 0.478, 0.108, 0.077, 0.086, 0.336, 0.062, 4.599, 0.152, 0.029, 0.008, 0.0001, 0.075, 0.022, 0.003, 1.759, 0.042, 0.219, 0.023, 0.382, 0.512, 0.004, 0.161, 0.001, 0.086, 0.887, 0.025, 0.094, 0.002, 0.484, 1.618, 0.083, 0.303, 0.146, 1.873, 0.0001, 0.931, 0.058, 0.143, 0.126, 0.78, 1.209, 1.122, 0.589, 0.667, 0.458, 22.229, 10.029, 0.199, 0.193, 0.652, 0.135, 0.025, 0.171, 0.328, 0.323, 1.631, 2.28, 0.0001, 0.0001, 0.014, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 31.391, 0.001, 0.071, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.502, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.684, 0.002, 0.094, 0.001, 0.001, 0.006, 0.001, 0.003, 0.078, 0.078, 0.001, 0.002, 0.423, 0.192, 0.522, 0.019, 0.207, 0.249, 0.16, 0.075, 0.065, 0.07, 0.06, 0.055, 0.066, 0.128, 0.025, 0.008, 0.003, 0.005, 0.004, 0.002, 0.0001, 0.018, 0.012, 0.019, 0.013, 0.012, 0.008, 0.007, 0.009, 0.026, 0.003, 0.004, 0.011, 0.017, 0.01, 0.009, 0.02, 0.002, 0.012, 0.024, 0.016, 0.006, 0.007, 0.006, 0.007, 0.003, 0.001, 0.006, 0.001, 0.006, 0.0001, 0.005, 0.0001, 0.097, 0.016, 0.039, 0.037, 0.119, 0.017, 0.023, 0.03, 0.088, 0.002, 0.012, 0.052, 0.031, 0.08, 0.086, 0.026, 0.002, 0.079, 0.064, 0.078, 0.038, 0.025, 0.012, 0.008, 0.018, 0.003, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 2.438, 1.425, 1.576, 1.589, 0.047, 1.639, 0.295, 0.416, 0.311, 0.001, 0.008, 0.672, 0.369, 2.886, 0.106, 0.163, 0.114, 0.151, 0.023, 0.067, 0.081, 0.017, 0.027, 0.033, 0.044, 0.004, 0.046, 0.028, 0.128, 0.083, 0.044, 0.031, 0.048, 0.074, 0.102, 0.063, 0.021, 0.125, 0.02, 0.022, 0.053, 1.026, 0.001, 0.019, 0.001, 0.067, 0.028, 1.192, 4.733, 1.04, 0.537, 2.615, 2.04, 0.399, 0.621, 0.396, 2.01, 1.723, 0.207, 2.589, 0.943, 3.889, 2.383, 0.107, 0.0001, 0.0001, 0.065, 0.012, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.001, 27.532, 13.908, 1.199, 1.049, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.072, 0.002, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.77, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.47, 0.002, 0.214, 0.0001, 0.0001, 0.017, 0.001, 0.035, 0.128, 0.128, 0.002, 0.001, 0.656, 0.155, 0.49, 0.006, 0.172, 0.19, 0.096, 0.052, 0.062, 0.054, 0.034, 0.043, 0.06, 0.129, 0.06, 0.015, 0.017, 0.012, 0.017, 0.0001, 0.0001, 0.018, 0.009, 0.023, 0.009, 0.011, 0.002, 0.006, 0.004, 0.035, 0.002, 0.005, 0.007, 0.014, 0.008, 0.008, 0.009, 0.001, 0.009, 0.019, 0.008, 0.005, 0.004, 0.007, 0.007, 0.001, 0.002, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.381, 0.035, 0.167, 0.122, 0.44, 0.045, 0.036, 0.034, 0.432, 0.005, 0.016, 0.206, 0.12, 0.248, 0.177, 0.096, 0.003, 0.253, 0.183, 0.236, 0.214, 0.038, 0.01, 0.011, 0.011, 0.03, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.01, 1.642, 2.712, 2.46, 0.4, 0.066, 0.487, 0.515, 0.507, 0.001, 0.001, 0.622, 0.372, 0.933, 0.029, 0.581, 0.134, 0.087, 0.042, 0.032, 0.081, 0.073, 0.022, 0.008, 0.061, 0.004, 0.139, 0.063, 0.145, 0.05, 0.043, 0.149, 0.144, 0.143, 0.069, 0.113, 0.038, 0.031, 0.007, 0.03, 0.013, 0.002, 0.001, 0.064, 0.002, 0.001, 0.029, 0.007, 3.78, 0.37, 0.558, 0.274, 1.316, 4.346, 0.072, 0.319, 3.558, 0.657, 1.356, 2.204, 1.073, 2.802, 2.13, 1.099, 0.0001, 0.0001, 0.025, 0.051, 0.091, 0.068, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.005, 0.0001, 0.008, 0.004, 27.537, 14.047, 0.001, 0.161, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.005, 0.022, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mr":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.525, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.348, 0.002, 0.043, 0.0001, 0.0001, 0.004, 0.0001, 0.024, 0.061, 0.064, 0.0001, 0.001, 0.221, 0.063, 0.539, 0.009, 0.009, 0.009, 0.006, 0.003, 0.003, 0.003, 0.003, 0.003, 0.003, 0.005, 0.03, 0.01, 0.003, 0.004, 0.003, 0.003, 0.0001, 0.008, 0.004, 0.006, 0.004, 0.003, 0.003, 0.003, 0.003, 0.007, 0.002, 0.002, 0.003, 0.006, 0.003, 0.002, 0.005, 0.0001, 0.004, 0.008, 0.009, 0.001, 0.002, 0.002, 0.0001, 0.001, 0.0001, 0.007, 0.0001, 0.007, 0.0001, 0.001, 0.0001, 0.138, 0.021, 0.046, 0.053, 0.162, 0.029, 0.028, 0.063, 0.114, 0.003, 0.011, 0.062, 0.038, 0.106, 0.103, 0.03, 0.002, 0.096, 0.09, 0.116, 0.04, 0.015, 0.019, 0.003, 0.023, 0.002, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 1.224, 0.397, 1.061, 0.056, 0.001, 0.297, 0.351, 1.664, 0.084, 0.127, 0.02, 0.461, 0.026, 2.286, 0.0001, 0.096, 0.005, 0.018, 0.001, 0.019, 0.005, 1.098, 0.145, 0.403, 0.083, 0.015, 0.659, 0.012, 0.404, 0.067, 0.014, 0.287, 0.125, 0.236, 0.039, 0.415, 24.995, 7.065, 0.585, 0.404, 1.081, 0.036, 0.727, 0.118, 0.317, 0.211, 0.844, 1.342, 1.809, 0.018, 1.056, 0.198, 0.001, 0.975, 0.327, 0.194, 1.035, 0.79, 0.001, 0.001, 0.003, 0.001, 3.71, 0.926, 0.0001, 0.0001, 0.015, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.418, 0.001, 0.048, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mrj": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.556, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.792, 0.004, 0.111, 0.0001, 0.0001, 0.008, 0.001, 0.036, 0.371, 0.372, 0.0001, 0.001, 0.508, 0.256, 0.9, 0.015, 0.334, 0.401, 0.27, 0.169, 0.152, 0.17, 0.137, 0.141, 0.168, 0.185, 0.1, 0.046, 0.009, 0.005, 0.008, 0.012, 0.0001, 0.017, 0.012, 0.012, 0.011, 0.006, 0.006, 0.008, 0.006, 0.083, 0.004, 0.011, 0.006, 0.014, 0.007, 0.024, 0.016, 0.001, 0.008, 0.014, 0.009, 0.003, 0.03, 0.002, 0.042, 0.002, 0.001, 0.008, 0.0001, 0.009, 0.0001, 0.003, 0.0001, 0.281, 0.025, 0.082, 0.065, 0.202, 0.013, 0.027, 0.052, 0.157, 0.003, 0.032, 0.08, 0.041, 0.09, 0.092, 0.03, 0.005, 0.117, 0.072, 0.076, 0.073, 0.015, 0.012, 0.004, 0.024, 0.01, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.356, 0.846, 2.179, 0.887, 0.116, 0.285, 0.312, 0.236, 2.316, 0.007, 0.004, 2.565, 0.266, 0.252, 0.05, 0.215, 0.187, 0.062, 0.078, 1.679, 0.285, 0.024, 0.005, 0.016, 0.067, 0.046, 0.237, 0.053, 0.116, 0.054, 0.059, 0.117, 0.058, 0.115, 0.145, 0.033, 0.102, 0.049, 0.064, 0.062, 0.066, 0.006, 0.001, 0.056, 0.003, 0.041, 0.007, 0.023, 2.651, 0.259, 1.194, 0.797, 1.113, 1.956, 0.572, 0.253, 2.277, 2.969, 1.78, 2.755, 1.532, 2.591, 1.704, 0.818, 0.0001, 0.0001, 0.138, 0.095, 0.012, 0.006, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.011, 0.0001, 0.008, 0.006, 24.363, 12.5, 0.002, 4.142, 0.0001, 0.0001, 0.0001, 0.001, 0.015, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.004, 0.341, 0.005, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ms":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.423, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.116, 0.004, 0.276, 0.001, 0.003, 0.028, 0.005, 0.04, 0.153, 0.154, 0.011, 0.002, 0.825, 0.313, 0.841, 0.02, 0.335, 0.324, 0.225, 0.11, 0.099, 0.112, 0.094, 0.087, 0.096, 0.171, 0.041, 0.019, 0.01, 0.005, 0.01, 0.002, 0.001, 0.327, 0.313, 0.169, 0.197, 0.08, 0.09, 0.097, 0.122, 0.22, 0.145, 0.326, 0.158, 0.369, 0.143, 0.065, 0.427, 0.013, 0.147, 0.487, 0.268, 0.071, 0.05, 0.063, 0.007, 0.038, 0.022, 0.015, 0.0001, 0.015, 0.0001, 0.002, 0.0001, 15.253, 2.008, 0.502, 3.234, 6.807, 0.209, 2.704, 2.141, 5.701, 0.605, 3.195, 3.049, 3.025, 7.562, 1.688, 2.054, 0.019, 4.172, 2.861, 3.513, 3.855, 0.159, 0.407, 0.024, 1.19, 0.123, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.025, 0.005, 0.003, 0.004, 0.003, 0.002, 0.002, 0.004, 0.003, 0.002, 0.002, 0.001, 0.002, 0.007, 0.004, 0.002, 0.001, 0.002, 0.001, 0.009, 0.003, 0.001, 0.001, 0.001, 0.002, 0.01, 0.001, 0.003, 0.004, 0.004, 0.001, 0.007, 0.031, 0.013, 0.003, 0.003, 0.003, 0.002, 0.001, 0.006, 0.007, 0.017, 0.002, 0.003, 0.001, 0.007, 0.002, 0.002, 0.004, 0.011, 0.003, 0.006, 0.005, 0.001, 0.006, 0.001, 0.004, 0.002, 0.003, 0.002, 0.008, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.034, 0.074, 0.022, 0.02, 0.0001, 0.0001, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.012, 0.015, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.024, 0.004, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mt":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.717, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.569, 0.003, 0.319, 0.001, 0.001, 0.009, 0.001, 0.699, 0.116, 0.117, 0.001, 0.002, 0.868, 2.789, 0.736, 0.014, 0.299, 0.341, 0.218, 0.093, 0.081, 0.087, 0.085, 0.082, 0.1, 0.201, 0.053, 0.022, 0.013, 0.012, 0.013, 0.002, 0.0001, 0.223, 0.171, 0.118, 0.162, 0.107, 0.236, 0.127, 0.076, 0.3, 0.048, 0.158, 0.199, 0.315, 0.08, 0.056, 0.187, 0.018, 0.103, 0.221, 0.127, 0.065, 0.054, 0.053, 0.02, 0.007, 0.009, 0.022, 0.0001, 0.023, 0.0001, 0.008, 0.002, 9.087, 1.533, 0.244, 1.812, 5.201, 1.498, 1.212, 0.809, 8.439, 2.13, 1.92, 5.784, 2.557, 4.221, 2.69, 1.16, 0.488, 3.837, 2.631, 5.521, 3.106, 0.451, 1.062, 0.484, 0.085, 0.753, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.211, 0.004, 0.004, 0.002, 0.003, 0.001, 0.001, 0.004, 0.002, 0.001, 0.016, 0.407, 0.001, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.042, 0.003, 0.001, 0.001, 0.001, 0.005, 0.141, 0.001, 0.001, 0.01, 0.01, 0.001, 0.002, 0.13, 0.527, 0.002, 0.004, 0.002, 0.001, 0.025, 1.521, 0.007, 0.014, 0.001, 0.004, 0.005, 0.008, 0.001, 0.001, 0.004, 0.005, 0.009, 0.004, 0.002, 0.002, 0.003, 0.001, 0.003, 0.01, 0.003, 0.015, 0.566, 0.003, 0.002, 0.002, 0.0001, 0.0001, 0.015, 0.129, 2.554, 0.578, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.011, 0.005, 0.011, 0.004, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.004, 0.006, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.004, 0.212, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mus": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.612, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 19.388, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.02, 0.0001, 1.02, 0.0001, 1.02, 1.02, 0.0001, 2.041, 1.02, 0.0001, 1.02, 1.02, 4.082, 1.02, 1.02, 2.041, 0.0001, 1.02, 1.02, 1.02, 1.02, 1.02, 1.02, 0.0001, 1.02, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.061, 0.0001, 0.0001, 0.0001, 5.102, 0.0001, 1.02, 0.0001, 1.02, 0.0001, 5.102, 0.0001, 1.02, 1.02, 2.041, 0.0001, 0.0001, 0.0001, 2.041, 0.0001, 0.0001, 2.041, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.02, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.02, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "my":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.476, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.676, 0.0001, 0.018, 0.0001, 0.0001, 0.001, 0.0001, 0.009, 0.072, 0.072, 0.0001, 0.001, 0.013, 0.027, 0.014, 0.004, 0.007, 0.006, 0.005, 0.003, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.001, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.009, 0.007, 0.011, 0.006, 0.004, 0.004, 0.005, 0.004, 0.006, 0.002, 0.003, 0.004, 0.008, 0.005, 0.004, 0.007, 0.0001, 0.005, 0.011, 0.008, 0.003, 0.002, 0.003, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.087, 0.015, 0.033, 0.032, 0.11, 0.015, 0.02, 0.035, 0.072, 0.001, 0.01, 0.046, 0.027, 0.071, 0.073, 0.021, 0.001, 0.069, 0.054, 0.072, 0.03, 0.01, 0.011, 0.003, 0.016, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 32.171, 1.737, 0.141, 0.03, 1.382, 0.783, 0.273, 0.069, 0.03, 0.083, 0.874, 0.307, 0.061, 0.061, 0.009, 0.119, 1.037, 0.261, 0.115, 0.031, 0.966, 0.888, 0.304, 0.058, 0.131, 1.12, 0.266, 0.843, 0.619, 0.172, 1.057, 0.095, 0.006, 0.703, 0.001, 0.001, 0.009, 0.019, 0.041, 0.006, 0.0001, 0.005, 0.0001, 0.239, 1.811, 1.255, 0.357, 1.497, 0.246, 1.317, 0.249, 0.0001, 0.0001, 0.0001, 0.294, 0.751, 1.889, 0.152, 3.975, 0.6, 0.881, 0.616, 0.651, 0.004, 0.0001, 0.0001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 31.801, 0.03, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "myv": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.363, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.917, 0.015, 0.248, 0.0001, 0.0001, 0.022, 0.0001, 0.001, 0.283, 0.286, 0.002, 0.004, 0.691, 0.215, 0.812, 0.009, 0.174, 0.262, 0.16, 0.093, 0.073, 0.077, 0.073, 0.069, 0.078, 0.133, 0.142, 0.014, 0.011, 0.005, 0.01, 0.008, 0.0001, 0.003, 0.002, 0.005, 0.001, 0.001, 0.002, 0.001, 0.001, 0.012, 0.001, 0.001, 0.002, 0.003, 0.001, 0.001, 0.004, 0.0001, 0.002, 0.003, 0.002, 0.001, 0.004, 0.001, 0.007, 0.0001, 0.001, 0.004, 0.0001, 0.004, 0.0001, 0.0001, 0.003, 0.048, 0.012, 0.02, 0.007, 0.038, 0.002, 0.005, 0.006, 0.024, 0.002, 0.008, 0.023, 0.008, 0.017, 0.019, 0.008, 0.0001, 0.032, 0.018, 0.013, 0.014, 0.004, 0.001, 0.001, 0.004, 0.002, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 2.092, 2.863, 2.802, 0.895, 0.06, 0.084, 0.303, 0.361, 0.574, 0.012, 0.006, 0.456, 2.653, 0.734, 0.106, 1.014, 0.129, 0.284, 0.186, 0.058, 0.27, 0.019, 0.007, 0.024, 0.083, 0.006, 0.182, 0.079, 0.175, 0.059, 0.072, 0.148, 0.231, 0.176, 0.101, 0.047, 0.012, 0.013, 0.018, 0.046, 0.024, 0.001, 0.0001, 0.091, 0.002, 0.065, 0.014, 0.024, 3.393, 0.354, 1.588, 0.391, 1.0, 3.63, 0.2, 0.667, 2.033, 0.447, 2.062, 1.616, 1.324, 3.27, 3.572, 0.635, 0.0001, 0.0001, 0.332, 0.006, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.017, 0.0001, 0.001, 0.001, 27.959, 14.855, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.281, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.032, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "mzn": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.201, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.629, 0.002, 0.049, 0.0001, 0.0001, 0.001, 0.004, 0.002, 0.134, 0.134, 0.0001, 0.0001, 0.026, 0.054, 0.593, 0.02, 0.019, 0.017, 0.017, 0.008, 0.004, 0.006, 0.012, 0.005, 0.01, 0.015, 0.042, 0.0001, 0.001, 0.014, 0.001, 0.004, 0.0001, 0.004, 0.003, 0.005, 0.003, 0.006, 0.016, 0.002, 0.002, 0.003, 0.001, 0.001, 0.009, 0.006, 0.002, 0.001, 0.004, 0.0001, 0.003, 0.005, 0.007, 0.001, 0.001, 0.01, 0.0001, 0.001, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.009, 0.0001, 0.072, 0.016, 0.031, 0.045, 0.106, 0.011, 0.011, 0.023, 0.094, 0.004, 0.019, 0.044, 0.012, 0.044, 0.054, 0.042, 0.001, 0.056, 0.056, 0.055, 0.021, 0.007, 0.011, 0.005, 0.015, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.44, 0.427, 0.449, 0.013, 1.516, 2.042, 3.291, 3.912, 3.162, 0.001, 0.032, 0.014, 4.412, 0.002, 0.195, 0.007, 0.446, 0.17, 0.001, 0.002, 0.012, 0.004, 0.0001, 0.001, 0.045, 0.005, 0.0001, 0.006, 0.001, 0.0001, 0.0001, 0.003, 0.003, 0.013, 0.18, 0.011, 0.008, 0.001, 0.211, 5.124, 1.425, 1.013, 2.263, 0.078, 0.559, 0.214, 0.344, 2.205, 0.318, 3.605, 0.725, 1.866, 1.033, 0.295, 0.164, 0.271, 0.156, 0.676, 0.058, 0.031, 0.001, 0.001, 0.258, 0.0001, 0.0001, 0.0001, 0.056, 0.008, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.009, 0.003, 0.0001, 0.003, 0.0001, 0.001, 0.001, 0.002, 19.953, 15.923, 1.548, 5.327, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.427, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "na":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.998, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.709, 0.015, 0.646, 0.0001, 0.002, 0.0001, 0.004, 0.021, 0.659, 0.659, 0.017, 0.002, 0.883, 0.333, 1.719, 0.03, 1.451, 2.231, 1.063, 0.565, 0.61, 0.611, 0.586, 0.586, 0.597, 1.829, 0.199, 0.094, 0.009, 0.006, 0.009, 0.002, 0.0001, 0.49, 0.423, 0.263, 0.348, 0.486, 0.143, 0.225, 0.131, 0.617, 0.095, 0.263, 0.178, 0.552, 0.272, 0.136, 0.483, 0.013, 0.313, 0.36, 0.336, 0.074, 0.114, 0.249, 0.018, 0.046, 0.052, 0.006, 0.0001, 0.007, 0.0001, 0.006, 0.0001, 7.914, 1.267, 0.542, 1.393, 6.136, 0.161, 1.565, 0.525, 5.317, 0.298, 1.632, 1.173, 1.479, 6.133, 5.204, 0.602, 0.04, 3.812, 1.491, 2.75, 1.848, 0.267, 2.105, 0.037, 0.79, 0.308, 0.0001, 0.0001, 0.0001, 0.013, 0.0001, 0.299, 0.053, 0.064, 0.017, 0.038, 0.029, 0.014, 0.008, 0.026, 0.003, 0.007, 0.007, 0.016, 0.016, 0.006, 0.008, 0.013, 0.032, 0.004, 0.043, 0.108, 0.002, 0.004, 0.01, 0.017, 0.017, 0.017, 0.015, 0.011, 0.008, 0.009, 0.017, 0.085, 0.124, 0.117, 0.04, 0.025, 0.023, 0.025, 0.032, 0.017, 0.097, 0.007, 0.031, 0.013, 0.031, 0.009, 0.016, 0.079, 0.095, 0.056, 0.083, 0.021, 0.063, 0.052, 0.013, 0.046, 0.015, 0.047, 0.045, 0.034, 0.035, 0.045, 0.013, 0.0001, 0.0001, 0.04, 0.466, 0.079, 0.167, 0.001, 0.002, 0.0001, 0.023, 0.01, 0.013, 0.002, 0.001, 0.027, 0.006, 0.292, 0.12, 0.0001, 0.0001, 0.009, 0.199, 0.04, 0.007, 0.055, 0.037, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.021, 0.074, 0.244, 0.0001, 0.005, 0.01, 0.006, 0.001, 0.002, 0.006, 0.003, 0.003, 0.009, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nah": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.08, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.795, 0.004, 0.316, 0.0001, 0.0001, 0.001, 0.0001, 0.011, 0.502, 0.505, 0.006, 0.329, 1.148, 0.317, 0.599, 1.663, 0.178, 0.339, 0.18, 0.099, 0.094, 0.088, 0.135, 0.085, 0.092, 0.204, 0.686, 0.012, 0.001, 0.002, 0.002, 0.012, 0.0001, 0.583, 0.136, 0.486, 0.282, 0.369, 0.108, 0.135, 0.149, 0.382, 0.043, 0.01, 0.153, 0.41, 0.267, 0.154, 0.356, 0.041, 0.209, 0.348, 0.531, 0.077, 0.099, 0.006, 0.046, 0.078, 0.021, 0.306, 0.0001, 0.304, 0.0001, 0.018, 0.0001, 8.12, 0.52, 3.826, 1.716, 6.024, 0.239, 0.598, 2.703, 7.016, 0.169, 0.062, 5.071, 1.759, 4.856, 5.013, 1.61, 0.66, 3.183, 2.38, 4.798, 3.279, 0.368, 0.013, 0.578, 0.571, 0.892, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.023, 0.52, 0.003, 0.003, 0.002, 0.001, 0.001, 0.002, 0.002, 0.007, 0.001, 0.001, 0.005, 0.446, 0.001, 0.001, 0.001, 0.002, 0.001, 0.26, 0.002, 0.003, 0.001, 0.001, 0.001, 0.002, 0.003, 0.001, 0.002, 0.001, 0.001, 0.001, 0.003, 0.117, 0.005, 0.001, 0.004, 0.001, 0.001, 0.002, 0.003, 0.168, 0.013, 0.475, 0.001, 0.136, 0.018, 0.008, 0.004, 0.071, 0.003, 0.269, 0.002, 0.003, 0.001, 0.001, 0.003, 0.002, 0.068, 0.005, 0.005, 0.002, 0.003, 0.016, 0.0001, 0.0001, 0.033, 0.838, 1.259, 0.446, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.009, 0.004, 0.012, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.003, 0.005, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.027, 0.002, 0.008, 0.004, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nap": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.664, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.609, 0.012, 0.443, 0.0001, 0.0001, 0.005, 0.002, 3.603, 0.188, 0.187, 0.001, 0.001, 0.851, 0.111, 0.916, 0.025, 0.289, 0.464, 0.288, 0.212, 0.187, 0.195, 0.184, 0.177, 0.191, 0.229, 0.063, 0.027, 0.064, 0.004, 0.064, 0.004, 0.0001, 0.359, 0.17, 0.431, 0.088, 0.101, 0.128, 0.139, 0.019, 0.153, 0.024, 0.014, 0.18, 0.269, 0.172, 0.129, 0.252, 0.015, 0.136, 0.331, 0.141, 0.042, 0.154, 0.012, 0.021, 0.004, 0.014, 0.007, 0.0001, 0.007, 0.0001, 0.001, 0.0001, 8.472, 0.677, 3.575, 1.818, 8.836, 0.628, 1.161, 0.605, 5.326, 0.294, 0.072, 2.854, 1.959, 5.855, 5.118, 1.978, 0.107, 4.154, 2.774, 4.302, 3.256, 1.068, 0.047, 0.011, 0.049, 0.778, 0.0001, 0.014, 0.0001, 0.0001, 0.0001, 0.167, 0.005, 0.004, 0.005, 0.003, 0.002, 0.001, 0.001, 0.017, 0.002, 0.001, 0.001, 0.001, 0.003, 0.001, 0.001, 0.003, 0.001, 0.001, 0.008, 0.005, 0.001, 0.001, 0.001, 0.033, 0.118, 0.001, 0.001, 0.005, 0.005, 0.001, 0.002, 0.266, 0.085, 0.051, 0.002, 0.003, 0.001, 0.003, 0.003, 0.62, 0.161, 0.023, 0.139, 0.069, 0.03, 0.001, 0.002, 0.025, 0.004, 0.164, 0.026, 0.08, 0.002, 0.003, 0.002, 0.002, 0.107, 0.016, 0.007, 0.005, 0.003, 0.002, 0.002, 0.0001, 0.0001, 0.057, 1.779, 0.007, 0.055, 0.0001, 0.001, 0.0001, 0.003, 0.002, 0.003, 0.0001, 0.0001, 0.013, 0.006, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.002, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.165, 0.003, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nds": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.919, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.282, 0.001, 0.235, 0.0001, 0.0001, 0.09, 0.002, 0.046, 0.155, 0.155, 0.017, 0.001, 0.777, 0.214, 1.137, 0.014, 0.414, 0.571, 0.279, 0.157, 0.152, 0.165, 0.151, 0.162, 0.212, 0.299, 0.042, 0.019, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.383, 0.397, 0.141, 0.527, 0.184, 0.228, 0.238, 0.278, 0.228, 0.153, 0.317, 0.238, 0.331, 0.224, 0.164, 0.212, 0.007, 0.201, 0.654, 0.206, 0.119, 0.194, 0.231, 0.003, 0.008, 0.039, 0.011, 0.0001, 0.011, 0.0001, 0.001, 0.0001, 4.393, 1.051, 1.267, 3.394, 10.917, 0.767, 1.396, 2.581, 3.914, 0.085, 1.325, 2.618, 1.593, 8.321, 3.314, 0.889, 0.009, 5.125, 3.768, 5.421, 2.363, 1.177, 0.929, 0.056, 0.171, 0.239, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.454, 0.002, 0.002, 0.001, 0.006, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.023, 0.001, 0.0001, 0.022, 0.001, 0.009, 0.334, 0.001, 0.001, 0.057, 0.001, 0.038, 0.018, 0.048, 0.004, 0.001, 0.001, 0.208, 0.002, 0.001, 0.001, 0.002, 0.009, 0.001, 0.001, 0.0001, 0.002, 0.0001, 0.001, 0.005, 0.002, 0.014, 0.003, 0.002, 0.002, 0.82, 0.001, 0.004, 0.001, 0.001, 0.001, 0.763, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.055, 1.884, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.007, 0.003, 0.008, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.454, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ne":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.49, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.629, 0.002, 0.033, 0.0001, 0.0001, 0.006, 0.0001, 0.018, 0.053, 0.057, 0.0001, 0.001, 0.2, 0.042, 0.073, 0.008, 0.003, 0.003, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.015, 0.002, 0.005, 0.002, 0.006, 0.002, 0.0001, 0.004, 0.003, 0.004, 0.002, 0.002, 0.002, 0.002, 0.002, 0.003, 0.001, 0.001, 0.002, 0.002, 0.002, 0.002, 0.003, 0.0001, 0.002, 0.004, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 0.056, 0.012, 0.021, 0.02, 0.066, 0.011, 0.012, 0.023, 0.048, 0.001, 0.005, 0.027, 0.017, 0.042, 0.043, 0.015, 0.001, 0.044, 0.036, 0.045, 0.017, 0.005, 0.008, 0.001, 0.01, 0.001, 0.0001, 0.009, 0.0001, 0.0001, 0.0001, 0.608, 0.76, 0.405, 0.065, 0.0001, 0.231, 0.124, 1.05, 0.333, 0.225, 0.003, 1.089, 0.054, 2.553, 0.0001, 0.268, 0.005, 0.0001, 0.0001, 0.016, 0.009, 1.681, 0.188, 0.574, 0.05, 0.059, 0.235, 0.302, 0.411, 0.024, 0.038, 0.296, 0.063, 0.16, 0.029, 0.16, 24.986, 7.481, 0.637, 0.298, 1.766, 0.034, 0.895, 0.142, 0.406, 0.37, 1.169, 0.857, 2.172, 0.0001, 1.023, 0.0001, 0.0001, 0.652, 0.263, 0.209, 1.099, 0.646, 0.0001, 0.0001, 0.001, 0.001, 3.096, 1.51, 0.0001, 0.0001, 0.006, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.857, 0.0001, 0.028, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "new": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.658, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.736, 0.0001, 0.005, 0.0001, 0.0001, 0.016, 0.0001, 0.016, 0.053, 0.053, 0.0001, 0.0001, 0.168, 0.064, 0.05, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.014, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.003, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.002, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.045, 0.006, 0.015, 0.015, 0.048, 0.008, 0.009, 0.021, 0.034, 0.001, 0.003, 0.019, 0.011, 0.032, 0.03, 0.01, 0.0001, 0.03, 0.026, 0.034, 0.014, 0.004, 0.005, 0.001, 0.007, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.748, 1.473, 1.078, 0.313, 0.0001, 0.165, 0.087, 0.934, 0.049, 0.145, 0.004, 0.213, 0.295, 3.035, 0.001, 0.021, 0.01, 0.0001, 0.001, 0.003, 0.002, 0.891, 0.378, 1.026, 0.007, 0.007, 0.145, 0.227, 0.557, 0.002, 0.008, 0.138, 0.03, 0.275, 0.076, 0.203, 24.519, 8.651, 0.655, 0.317, 1.238, 0.066, 0.765, 0.114, 0.288, 0.474, 0.695, 2.038, 1.25, 0.006, 0.967, 0.005, 0.016, 1.209, 0.15, 0.223, 0.893, 0.295, 0.0001, 0.001, 0.016, 0.0001, 3.268, 1.125, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.648, 0.0001, 0.246, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ng":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.332, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.852, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.028, 0.028, 0.0001, 0.0001, 0.569, 0.014, 0.833, 0.0001, 0.0001, 0.028, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.042, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.291, 0.028, 0.014, 0.069, 0.125, 0.0001, 0.194, 0.153, 0.5, 0.042, 0.069, 0.0001, 0.056, 0.153, 0.402, 0.083, 0.0001, 0.0001, 0.014, 0.18, 0.222, 0.0001, 0.222, 0.0001, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.728, 1.221, 0.236, 1.443, 7.106, 0.347, 2.859, 3.65, 4.136, 0.125, 4.316, 3.539, 3.983, 7.412, 7.883, 1.596, 0.0001, 1.138, 1.901, 3.511, 6.62, 0.402, 2.776, 0.0001, 2.11, 0.194, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.028, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.056, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.056, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.028, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nov": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.223, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.739, 0.005, 0.371, 0.0001, 0.0001, 0.004, 0.0001, 0.014, 0.258, 0.258, 0.003, 0.008, 0.827, 0.256, 1.132, 0.016, 0.643, 0.521, 0.435, 0.122, 0.19, 0.117, 0.125, 0.112, 0.14, 0.247, 0.122, 0.023, 0.02, 0.012, 0.021, 0.004, 0.0001, 0.495, 0.126, 0.101, 0.11, 0.205, 0.084, 0.113, 0.079, 0.113, 0.072, 0.274, 0.517, 0.205, 0.19, 0.072, 0.148, 0.01, 0.107, 0.475, 0.124, 0.103, 0.078, 0.049, 0.006, 0.028, 0.032, 0.002, 0.0001, 0.003, 0.0001, 0.002, 0.0001, 6.407, 1.088, 0.275, 3.233, 10.596, 0.875, 0.958, 0.605, 7.974, 0.212, 2.738, 4.237, 2.737, 5.182, 4.585, 1.557, 0.08, 4.568, 4.834, 4.299, 2.875, 0.823, 0.156, 0.296, 0.238, 0.085, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.026, 0.009, 0.01, 0.005, 0.006, 0.003, 0.001, 0.003, 0.003, 0.003, 0.001, 0.001, 0.003, 0.004, 0.0001, 0.0001, 0.003, 0.001, 0.003, 0.016, 0.003, 0.003, 0.001, 0.001, 0.001, 0.007, 0.002, 0.003, 0.003, 0.007, 0.001, 0.0001, 0.012, 0.008, 0.002, 0.005, 0.005, 0.002, 0.003, 0.003, 0.008, 0.015, 0.002, 0.001, 0.004, 0.007, 0.004, 0.004, 0.003, 0.005, 0.01, 0.013, 0.003, 0.002, 0.006, 0.004, 0.012, 0.004, 0.008, 0.008, 0.012, 0.008, 0.004, 0.006, 0.0001, 0.0001, 0.013, 0.071, 0.012, 0.013, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.04, 0.019, 0.02, 0.007, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.008, 0.025, 0.004, 0.0001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nrm": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.521, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.376, 0.004, 0.467, 0.0001, 0.0001, 0.003, 0.0001, 1.947, 0.163, 0.162, 0.0001, 0.0001, 0.761, 0.157, 0.914, 0.012, 1.102, 1.934, 0.564, 0.483, 0.476, 0.487, 0.557, 0.61, 0.644, 0.66, 0.069, 0.027, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.78, 0.094, 0.435, 0.253, 0.108, 0.06, 0.094, 0.042, 0.231, 0.101, 0.009, 0.332, 0.216, 0.104, 0.043, 0.114, 0.015, 0.096, 0.165, 0.06, 0.048, 0.107, 0.012, 0.147, 0.029, 0.002, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 4.848, 0.485, 1.639, 2.302, 7.56, 0.544, 0.696, 1.469, 3.879, 0.15, 0.04, 3.065, 1.372, 5.618, 3.164, 1.345, 0.499, 2.846, 4.958, 4.48, 3.809, 0.712, 0.015, 0.135, 0.467, 0.062, 0.0001, 1.427, 0.0001, 0.0001, 0.0001, 0.085, 0.003, 0.016, 0.002, 0.004, 0.002, 0.001, 0.0001, 0.002, 0.014, 0.026, 0.0001, 0.001, 0.001, 0.016, 0.0001, 0.001, 0.001, 0.001, 0.016, 0.002, 0.001, 0.0001, 0.0001, 0.002, 0.06, 0.0001, 0.023, 0.002, 0.001, 0.0001, 0.001, 0.219, 0.004, 0.233, 0.004, 0.005, 0.011, 0.0001, 0.016, 0.454, 2.065, 0.259, 0.013, 0.002, 0.004, 0.368, 0.008, 0.001, 0.008, 0.002, 0.004, 0.081, 0.004, 0.002, 0.012, 0.002, 0.151, 0.004, 0.112, 0.003, 0.005, 0.001, 0.003, 0.0001, 0.0001, 0.015, 4.079, 0.017, 0.014, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.029, 0.013, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.077, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nso": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.78, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.755, 0.002, 0.038, 0.001, 0.0001, 0.004, 0.0001, 0.009, 0.457, 0.457, 0.0001, 0.0001, 0.676, 0.052, 0.694, 0.005, 0.466, 0.863, 0.377, 0.269, 0.237, 0.244, 0.243, 0.245, 0.241, 0.264, 0.019, 0.004, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.278, 0.349, 0.291, 0.135, 0.079, 0.05, 0.105, 0.055, 0.212, 0.029, 0.164, 0.314, 0.912, 0.402, 0.025, 0.107, 0.005, 0.046, 0.407, 0.145, 0.02, 0.095, 0.034, 0.21, 0.005, 0.034, 0.003, 0.0001, 0.003, 0.0001, 0.001, 0.0001, 11.556, 1.304, 0.158, 0.76, 9.745, 0.742, 5.815, 1.511, 2.097, 0.062, 3.221, 3.795, 3.273, 4.341, 8.239, 1.391, 0.015, 2.291, 2.998, 2.898, 0.946, 0.091, 3.286, 0.014, 0.713, 0.058, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.019, 0.718, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.045, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.083, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.134, 0.0001, 0.733, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "nv":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.509, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.471, 0.0001, 0.553, 0.0001, 0.0001, 0.002, 0.0001, 0.001, 0.132, 0.131, 0.0001, 0.0001, 0.333, 0.05, 0.853, 0.008, 0.262, 0.198, 0.155, 0.093, 0.082, 0.107, 0.089, 0.066, 0.073, 0.069, 0.012, 0.267, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.105, 0.358, 0.12, 0.313, 0.022, 0.005, 0.023, 0.24, 0.014, 0.023, 0.045, 0.016, 0.036, 0.298, 0.014, 0.019, 0.001, 0.01, 0.1, 0.28, 0.004, 0.005, 0.04, 0.001, 0.048, 0.004, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 5.817, 1.241, 0.605, 3.905, 1.9, 0.016, 1.415, 4.266, 6.018, 0.364, 1.038, 1.394, 0.149, 2.559, 2.764, 0.063, 0.003, 0.149, 2.248, 2.042, 0.092, 0.019, 0.145, 0.037, 1.033, 1.245, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.08, 1.403, 1.567, 0.013, 0.082, 1.105, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.001, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.067, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.294, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 2.758, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 2.445, 0.0001, 0.311, 0.0001, 4.591, 0.0001, 0.504, 0.001, 0.001, 0.001, 2.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.013, 4.172, 0.001, 0.0001, 0.013, 0.0001, 0.0001, 0.002, 11.893, 1.899, 1.744, 0.0001, 0.311, 0.0001, 0.0001, 4.171, 0.0001, 1.234, 0.0001, 0.002, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.08, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ny":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.625, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.431, 0.001, 0.161, 0.0001, 0.001, 0.013, 0.004, 0.199, 0.151, 0.149, 0.001, 0.0001, 0.726, 0.052, 1.005, 0.011, 0.425, 0.321, 0.242, 0.114, 0.121, 0.146, 0.109, 0.097, 0.1, 0.188, 0.119, 0.008, 0.003, 0.006, 0.003, 0.001, 0.0001, 0.324, 0.194, 0.362, 0.113, 0.083, 0.06, 0.055, 0.057, 0.099, 0.059, 0.134, 0.101, 0.607, 0.172, 0.041, 0.204, 0.005, 0.064, 0.151, 0.1, 0.088, 0.025, 0.048, 0.003, 0.047, 0.083, 0.018, 0.0001, 0.019, 0.0001, 0.0001, 0.003, 13.746, 1.132, 1.623, 2.856, 4.347, 0.427, 1.15, 2.997, 7.993, 0.223, 3.837, 3.15, 3.985, 6.204, 4.4, 1.606, 0.018, 2.007, 1.898, 3.156, 4.108, 0.173, 2.53, 0.014, 1.184, 1.868, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 0.066, 0.003, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.004, 0.05, 0.0001, 0.0001, 0.006, 0.005, 0.001, 0.001, 0.013, 0.003, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.013, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.002, 0.004, 0.001, 0.006, 0.001, 0.06, 0.0001, 0.011, 0.002, 0.003, 0.003, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.023, 0.029, 0.004, 0.064, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.009, 0.002, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.066, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "oc":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.196, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.065, 0.002, 0.316, 0.046, 0.0001, 0.007, 0.001, 0.708, 0.193, 0.193, 0.002, 0.001, 0.913, 0.337, 0.785, 0.01, 0.177, 0.339, 0.143, 0.091, 0.092, 0.097, 0.087, 0.093, 0.11, 0.155, 0.065, 0.02, 0.037, 0.032, 0.038, 0.002, 0.0001, 0.316, 0.163, 0.28, 0.118, 0.18, 0.101, 0.108, 0.046, 0.126, 0.059, 0.019, 0.366, 0.206, 0.087, 0.061, 0.2, 0.02, 0.116, 0.247, 0.092, 0.045, 0.108, 0.016, 0.03, 0.009, 0.007, 0.013, 0.0001, 0.013, 0.0001, 0.002, 0.0001, 9.22, 0.793, 2.634, 3.53, 8.653, 0.714, 1.084, 0.594, 5.176, 0.154, 0.069, 4.196, 2.017, 5.599, 4.023, 1.809, 0.517, 4.973, 5.482, 4.438, 3.287, 0.768, 0.022, 0.148, 0.146, 0.139, 0.0001, 0.011, 0.002, 0.002, 0.0001, 0.071, 0.004, 0.003, 0.002, 0.001, 0.001, 0.002, 0.002, 0.007, 0.009, 0.001, 0.001, 0.001, 0.004, 0.001, 0.001, 0.001, 0.001, 0.006, 0.007, 0.005, 0.001, 0.0001, 0.0001, 0.001, 0.051, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.145, 0.074, 0.024, 0.002, 0.002, 0.001, 0.002, 0.134, 0.929, 0.452, 0.016, 0.026, 0.001, 0.096, 0.005, 0.03, 0.005, 0.004, 0.448, 0.027, 0.01, 0.002, 0.002, 0.002, 0.002, 0.004, 0.011, 0.027, 0.011, 0.002, 0.002, 0.002, 0.0001, 0.0001, 0.068, 2.417, 0.003, 0.007, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.008, 0.004, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.067, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "olo": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.555, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.165, 0.001, 0.091, 0.0001, 0.0001, 0.009, 0.001, 0.061, 0.221, 0.221, 0.0001, 0.001, 0.891, 0.306, 1.442, 0.031, 0.305, 0.513, 0.256, 0.174, 0.154, 0.153, 0.124, 0.134, 0.157, 0.296, 0.102, 0.011, 0.01, 0.003, 0.01, 0.002, 0.001, 0.151, 0.077, 0.024, 0.041, 0.069, 0.036, 0.06, 0.101, 0.085, 0.112, 0.389, 0.128, 0.177, 0.115, 0.068, 0.258, 0.001, 0.092, 0.352, 0.141, 0.032, 0.27, 0.012, 0.021, 0.024, 0.01, 0.006, 0.0001, 0.006, 0.0001, 0.001, 0.005, 7.441, 0.351, 0.076, 1.841, 5.414, 0.122, 0.914, 2.187, 8.145, 1.143, 3.3, 4.3, 1.89, 6.075, 4.589, 1.29, 0.001, 2.71, 3.519, 3.872, 5.598, 2.436, 0.016, 0.008, 1.054, 0.985, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.389, 0.086, 0.073, 0.032, 0.01, 0.011, 0.006, 0.01, 0.008, 0.001, 0.002, 0.006, 0.036, 0.349, 0.005, 0.017, 0.009, 0.004, 0.01, 0.105, 0.013, 0.002, 0.002, 0.002, 0.005, 0.148, 0.017, 0.006, 0.011, 0.056, 0.001, 0.018, 0.051, 0.118, 0.003, 0.002, 1.86, 0.003, 0.001, 0.007, 0.003, 0.003, 0.001, 0.01, 0.001, 0.003, 0.001, 0.002, 0.143, 0.012, 0.077, 0.019, 0.054, 0.096, 0.298, 0.031, 0.077, 0.038, 0.064, 0.07, 0.021, 0.096, 0.259, 0.018, 0.0001, 0.0001, 0.075, 2.173, 0.359, 0.275, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.002, 0.001, 0.0001, 0.01, 0.006, 0.977, 0.34, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.008, 0.007, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.004, 0.001, 0.32, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "om":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.856, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.47, 0.007, 0.135, 0.001, 0.001, 0.02, 0.001, 0.415, 0.149, 0.148, 0.003, 0.001, 0.598, 0.088, 0.774, 0.024, 0.177, 0.196, 0.107, 0.055, 0.058, 0.069, 0.055, 0.055, 0.062, 0.111, 0.03, 0.031, 0.014, 0.003, 0.015, 0.004, 0.0001, 0.377, 0.227, 0.059, 0.14, 0.066, 0.078, 0.179, 0.124, 0.131, 0.063, 0.149, 0.064, 0.172, 0.076, 0.188, 0.061, 0.049, 0.057, 0.159, 0.099, 0.042, 0.016, 0.108, 0.015, 0.076, 0.01, 0.011, 0.0001, 0.011, 0.0001, 0.003, 0.13, 18.959, 2.318, 0.672, 2.536, 5.221, 1.607, 1.449, 2.234, 8.071, 0.935, 2.586, 2.201, 2.811, 5.266, 4.391, 0.29, 0.68, 3.658, 3.072, 4.065, 4.017, 0.087, 0.574, 0.143, 1.425, 0.099, 0.01, 0.001, 0.01, 0.0001, 0.0001, 0.186, 0.003, 0.002, 0.003, 0.006, 0.002, 0.002, 0.001, 0.004, 0.002, 0.003, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.006, 0.004, 0.0001, 0.0001, 0.0001, 0.007, 0.12, 0.0001, 0.0001, 0.011, 0.011, 0.005, 0.0001, 0.099, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.03, 0.008, 0.002, 0.002, 0.003, 0.001, 0.001, 0.002, 0.001, 0.002, 0.003, 0.004, 0.003, 0.002, 0.001, 0.001, 0.002, 0.001, 0.001, 0.003, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.105, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.002, 0.0001, 0.0001, 0.007, 0.004, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.024, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.191, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "or":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.414, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.789, 0.001, 0.049, 0.0001, 0.0001, 0.027, 0.0001, 0.016, 0.066, 0.066, 0.001, 0.0001, 0.194, 0.029, 0.065, 0.008, 0.012, 0.014, 0.009, 0.005, 0.004, 0.005, 0.004, 0.004, 0.004, 0.006, 0.014, 0.004, 0.003, 0.001, 0.003, 0.0001, 0.0001, 0.006, 0.004, 0.006, 0.004, 0.003, 0.002, 0.002, 0.003, 0.007, 0.001, 0.002, 0.003, 0.005, 0.003, 0.003, 0.005, 0.0001, 0.003, 0.007, 0.006, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.081, 0.011, 0.027, 0.026, 0.08, 0.013, 0.015, 0.029, 0.067, 0.002, 0.006, 0.037, 0.021, 0.058, 0.062, 0.019, 0.001, 0.059, 0.056, 0.058, 0.033, 0.007, 0.008, 0.003, 0.014, 0.002, 0.0001, 0.009, 0.0001, 0.0001, 0.0001, 0.461, 0.87, 0.209, 0.086, 0.0001, 0.314, 0.231, 1.688, 0.023, 0.138, 0.001, 0.379, 0.073, 2.56, 0.0001, 0.427, 0.002, 0.0001, 0.0001, 0.16, 0.011, 1.385, 0.13, 0.389, 0.041, 0.233, 0.239, 0.105, 0.371, 0.014, 0.058, 0.871, 0.144, 0.225, 0.017, 0.346, 1.495, 0.883, 0.609, 0.353, 1.21, 0.043, 0.827, 0.113, 24.372, 7.183, 0.945, 0.296, 2.678, 0.059, 0.571, 0.283, 0.0001, 0.05, 0.328, 0.264, 0.929, 0.701, 0.0001, 0.0001, 0.094, 0.0001, 2.794, 2.229, 0.0001, 0.0001, 0.045, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.66, 0.0001, 0.069, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "os":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.314, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.198, 0.001, 0.075, 0.0001, 0.0001, 0.009, 0.0001, 0.003, 0.221, 0.221, 0.0001, 0.001, 0.656, 0.268, 0.647, 0.003, 0.149, 0.239, 0.119, 0.07, 0.068, 0.074, 0.065, 0.065, 0.08, 0.138, 0.043, 0.023, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.003, 0.003, 0.004, 0.002, 0.002, 0.002, 0.002, 0.001, 0.024, 0.001, 0.001, 0.002, 0.003, 0.002, 0.002, 0.003, 0.0001, 0.002, 0.005, 0.002, 0.001, 0.007, 0.001, 0.014, 0.001, 0.0001, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 0.032, 0.005, 0.009, 0.008, 0.024, 0.004, 0.005, 0.006, 0.02, 0.001, 0.005, 0.014, 0.008, 0.019, 0.019, 0.004, 0.0001, 0.021, 0.013, 0.014, 0.012, 0.003, 0.002, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.723, 1.959, 2.433, 1.742, 0.485, 1.045, 0.733, 0.104, 0.039, 0.003, 0.515, 3.703, 0.088, 0.047, 0.033, 0.054, 0.152, 0.11, 0.038, 0.09, 0.183, 0.021, 0.005, 0.03, 0.137, 0.06, 0.132, 0.043, 0.092, 0.063, 0.026, 0.055, 0.062, 0.15, 0.074, 0.109, 0.063, 0.115, 4.882, 0.027, 0.022, 0.001, 0.001, 0.076, 0.0001, 0.018, 0.005, 0.007, 3.663, 0.542, 0.469, 1.346, 2.223, 0.835, 0.243, 0.949, 1.778, 1.262, 0.952, 1.134, 1.447, 2.532, 1.739, 0.347, 0.0001, 0.0001, 0.16, 4.829, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.01, 0.0001, 0.002, 0.001, 23.204, 15.519, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.01, 0.127, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pa":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.424, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.354, 0.003, 0.052, 0.0001, 0.0001, 0.005, 0.0001, 0.026, 0.07, 0.07, 0.0001, 0.001, 0.252, 0.083, 0.057, 0.008, 0.09, 0.134, 0.073, 0.034, 0.034, 0.037, 0.032, 0.033, 0.039, 0.073, 0.023, 0.007, 0.005, 0.003, 0.005, 0.001, 0.0001, 0.004, 0.002, 0.004, 0.002, 0.003, 0.002, 0.002, 0.002, 0.006, 0.001, 0.001, 0.002, 0.003, 0.002, 0.002, 0.003, 0.0001, 0.002, 0.005, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.007, 0.0001, 0.007, 0.0001, 0.002, 0.003, 0.035, 0.006, 0.011, 0.011, 0.037, 0.006, 0.008, 0.011, 0.028, 0.001, 0.004, 0.018, 0.013, 0.026, 0.027, 0.009, 0.001, 0.025, 0.019, 0.024, 0.012, 0.004, 0.004, 0.001, 0.005, 0.001, 0.0001, 0.008, 0.0001, 0.0001, 0.0001, 1.534, 0.423, 1.17, 0.001, 0.001, 0.437, 0.548, 1.695, 0.606, 0.237, 0.024, 0.573, 0.09, 0.23, 0.001, 0.065, 0.035, 0.0001, 0.001, 0.028, 0.011, 1.172, 0.229, 0.428, 0.077, 0.013, 0.442, 0.041, 0.755, 0.032, 0.001, 0.286, 0.066, 0.192, 0.017, 0.267, 1.499, 0.487, 1.308, 0.154, 24.532, 6.371, 0.647, 0.143, 0.447, 0.172, 0.696, 0.068, 2.373, 0.727, 0.943, 0.005, 0.001, 0.891, 0.001, 0.001, 1.461, 1.12, 0.001, 0.001, 0.465, 0.001, 2.611, 1.487, 0.0001, 0.0001, 0.039, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.006, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 30.092, 0.001, 0.038, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pag": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.03, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.592, 0.019, 0.241, 0.0001, 0.0001, 0.002, 0.0001, 0.022, 0.579, 0.579, 0.001, 0.002, 1.012, 0.179, 1.28, 0.004, 0.607, 0.517, 0.439, 0.246, 0.233, 0.22, 0.206, 0.189, 0.188, 0.203, 0.287, 0.029, 0.2, 0.003, 0.2, 0.007, 0.0001, 0.514, 0.233, 0.299, 0.377, 0.293, 0.091, 0.061, 0.039, 0.141, 0.067, 0.298, 0.143, 0.228, 0.153, 0.107, 0.323, 0.017, 0.066, 0.835, 0.144, 0.158, 0.036, 0.141, 0.001, 0.016, 0.024, 0.044, 0.0001, 0.045, 0.0001, 0.001, 0.0001, 14.553, 2.155, 0.612, 2.213, 4.092, 0.158, 2.144, 0.652, 5.448, 0.026, 2.943, 4.371, 1.62, 6.468, 4.169, 1.489, 0.134, 1.92, 4.171, 4.111, 1.822, 0.133, 0.774, 0.019, 2.966, 0.181, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.082, 0.003, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.005, 0.002, 0.0001, 0.001, 0.001, 0.001, 0.048, 0.0001, 0.001, 0.01, 0.009, 0.0001, 0.0001, 0.004, 0.017, 0.0001, 0.001, 0.002, 0.001, 0.004, 0.002, 0.001, 0.008, 0.001, 0.001, 0.001, 0.005, 0.0001, 0.001, 0.003, 0.012, 0.002, 0.005, 0.002, 0.001, 0.001, 0.001, 0.004, 0.001, 0.005, 0.001, 0.001, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.004, 0.049, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.009, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.003, 0.075, 0.004, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pam": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.69, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.089, 0.008, 0.379, 0.001, 0.002, 0.006, 0.002, 0.032, 0.278, 0.278, 0.002, 0.002, 1.032, 0.272, 0.824, 0.024, 0.328, 0.294, 0.247, 0.119, 0.115, 0.114, 0.097, 0.092, 0.106, 0.173, 0.084, 0.034, 0.037, 0.005, 0.037, 0.004, 0.0001, 0.446, 0.258, 0.283, 0.225, 0.147, 0.167, 0.119, 0.095, 0.427, 0.078, 0.159, 0.17, 0.339, 0.137, 0.085, 0.26, 0.014, 0.119, 0.316, 0.173, 0.073, 0.129, 0.078, 0.01, 0.038, 0.027, 0.056, 0.0001, 0.058, 0.0001, 0.003, 0.0001, 11.717, 1.42, 1.079, 1.934, 5.65, 0.412, 5.535, 0.984, 6.498, 0.051, 2.215, 3.499, 2.546, 9.826, 2.443, 1.882, 0.041, 3.326, 2.846, 3.809, 3.421, 0.311, 0.524, 0.064, 1.429, 0.196, 0.0001, 0.012, 0.001, 0.0001, 0.0001, 0.064, 0.007, 0.004, 0.005, 0.004, 0.002, 0.003, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.006, 0.001, 0.001, 0.002, 0.001, 0.002, 0.019, 0.004, 0.002, 0.002, 0.002, 0.003, 0.009, 0.002, 0.002, 0.017, 0.008, 0.009, 0.013, 0.012, 0.022, 0.007, 0.002, 0.067, 0.005, 0.005, 0.005, 0.009, 0.033, 0.004, 0.003, 0.003, 0.006, 0.006, 0.004, 0.01, 0.012, 0.01, 0.01, 0.006, 0.003, 0.032, 0.002, 0.004, 0.003, 0.008, 0.007, 0.067, 0.001, 0.005, 0.003, 0.0001, 0.0001, 0.014, 0.263, 0.005, 0.006, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.008, 0.006, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.04, 0.011, 0.063, 0.0001, 0.001, 0.004, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pap": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.577, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 17.041, 0.006, 0.256, 0.002, 0.001, 0.008, 0.005, 0.088, 0.145, 0.144, 0.001, 0.001, 0.829, 0.082, 0.948, 0.014, 0.296, 0.379, 0.247, 0.133, 0.132, 0.125, 0.108, 0.098, 0.111, 0.184, 0.198, 0.046, 0.009, 0.003, 0.009, 0.012, 0.0001, 0.325, 0.157, 0.17, 0.173, 0.346, 0.09, 0.081, 0.124, 0.155, 0.09, 0.129, 0.105, 0.233, 0.162, 0.14, 0.176, 0.005, 0.132, 0.285, 0.134, 0.07, 0.056, 0.051, 0.003, 0.069, 0.012, 0.014, 0.0001, 0.014, 0.0001, 0.003, 0.0001, 10.584, 1.755, 0.96, 3.481, 6.611, 0.57, 0.793, 1.086, 6.922, 0.094, 2.168, 2.197, 2.209, 6.462, 5.124, 1.889, 0.017, 4.387, 3.838, 4.459, 3.468, 0.348, 0.249, 0.043, 0.415, 0.112, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.051, 0.006, 0.003, 0.006, 0.004, 0.001, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.012, 0.002, 0.005, 0.003, 0.001, 0.001, 0.001, 0.006, 0.016, 0.001, 0.001, 0.01, 0.01, 0.001, 0.001, 0.023, 0.288, 0.003, 0.003, 0.003, 0.001, 0.002, 0.007, 0.143, 0.171, 0.002, 0.003, 0.001, 0.133, 0.0001, 0.002, 0.003, 0.162, 0.171, 0.076, 0.001, 0.002, 0.002, 0.001, 0.004, 0.053, 0.027, 0.003, 0.086, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.023, 1.326, 0.004, 0.005, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.007, 0.004, 0.023, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.007, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.007, 0.048, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pcd": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.27, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.657, 0.008, 0.673, 0.0001, 0.0001, 0.003, 0.004, 1.579, 0.485, 0.482, 0.004, 0.002, 0.764, 0.743, 0.844, 0.029, 0.205, 0.439, 0.185, 0.104, 0.105, 0.099, 0.112, 0.109, 0.159, 0.228, 0.141, 0.017, 0.014, 0.103, 0.015, 0.005, 0.0001, 0.322, 0.315, 0.578, 0.163, 0.158, 0.188, 0.128, 0.139, 0.259, 0.095, 0.035, 0.293, 0.23, 0.148, 0.082, 0.416, 0.024, 0.141, 0.362, 0.132, 0.056, 0.121, 0.049, 0.026, 0.011, 0.008, 0.008, 0.0001, 0.009, 0.0001, 0.004, 0.004, 4.164, 0.548, 3.109, 2.876, 7.669, 0.633, 0.742, 2.062, 6.133, 0.168, 0.261, 3.265, 1.584, 5.6, 3.825, 1.718, 0.299, 3.984, 4.345, 3.994, 3.516, 0.729, 0.062, 0.099, 0.331, 0.121, 0.0001, 0.008, 0.001, 0.001, 0.0001, 0.34, 0.006, 0.009, 0.004, 0.003, 0.002, 0.003, 0.003, 0.01, 0.086, 0.002, 0.001, 0.004, 0.005, 0.003, 0.002, 0.003, 0.002, 0.002, 0.018, 0.009, 0.001, 0.001, 0.002, 0.003, 0.283, 0.001, 0.003, 0.003, 0.002, 0.001, 0.001, 0.283, 0.006, 0.038, 0.003, 0.005, 0.006, 0.003, 0.025, 0.548, 2.644, 0.014, 0.04, 0.002, 0.005, 0.009, 0.049, 0.019, 0.006, 0.015, 0.007, 0.051, 0.004, 0.002, 0.022, 0.007, 0.018, 0.005, 0.043, 0.008, 0.005, 0.004, 0.007, 0.0001, 0.0001, 0.109, 3.762, 0.007, 0.017, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.003, 0.0001, 0.0001, 0.018, 0.008, 0.022, 0.007, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.009, 0.005, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.006, 0.329, 0.001, 0.002, 0.009, 0.006, 0.002, 0.003, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pdc": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.347, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.345, 0.014, 0.296, 0.0001, 0.002, 0.001, 0.004, 0.083, 0.252, 0.251, 0.004, 0.0001, 0.899, 0.34, 1.232, 0.016, 0.318, 0.569, 0.233, 0.139, 0.122, 0.133, 0.151, 0.187, 0.197, 0.319, 0.11, 0.028, 0.009, 0.001, 0.009, 0.014, 0.001, 0.435, 0.396, 0.269, 0.612, 0.446, 0.261, 0.34, 0.266, 0.158, 0.126, 0.345, 0.316, 0.408, 0.189, 0.112, 0.446, 0.018, 0.164, 0.922, 0.173, 0.1, 0.127, 0.257, 0.003, 0.091, 0.087, 0.001, 0.0001, 0.002, 0.0001, 0.002, 0.001, 5.822, 0.784, 2.856, 3.127, 10.434, 0.913, 1.544, 3.717, 6.407, 0.033, 0.633, 2.751, 1.821, 6.435, 2.483, 0.545, 0.012, 4.834, 4.912, 3.94, 2.397, 0.665, 1.309, 0.053, 0.442, 0.521, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.13, 0.003, 0.001, 0.002, 0.004, 0.002, 0.002, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.012, 0.001, 0.0001, 0.001, 0.0001, 0.012, 0.052, 0.0001, 0.0001, 0.021, 0.02, 0.013, 0.012, 0.005, 0.005, 0.001, 0.002, 0.087, 0.001, 0.002, 0.004, 0.002, 0.014, 0.002, 0.001, 0.002, 0.005, 0.0001, 0.002, 0.002, 0.007, 0.005, 0.01, 0.013, 0.001, 0.014, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.03, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.019, 0.191, 0.004, 0.004, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.003, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.011, 0.007, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.129, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pfl": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.263, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.706, 0.009, 0.364, 0.0001, 0.0001, 0.013, 0.001, 0.091, 0.195, 0.195, 0.003, 0.001, 0.686, 0.342, 1.046, 0.015, 0.201, 0.285, 0.126, 0.084, 0.075, 0.083, 0.073, 0.08, 0.09, 0.133, 0.057, 0.007, 0.004, 0.002, 0.005, 0.003, 0.0001, 0.282, 0.468, 0.129, 0.696, 0.183, 0.211, 0.353, 0.23, 0.145, 0.127, 0.364, 0.254, 0.349, 0.172, 0.165, 0.214, 0.007, 0.306, 0.587, 0.085, 0.112, 0.133, 0.253, 0.004, 0.004, 0.083, 0.004, 0.0001, 0.004, 0.0001, 0.001, 0.001, 5.458, 1.123, 3.147, 5.166, 9.364, 1.012, 2.021, 4.491, 5.715, 0.138, 0.564, 2.856, 2.578, 5.721, 2.982, 0.339, 0.016, 4.316, 5.155, 2.124, 2.997, 0.789, 1.349, 0.04, 0.119, 0.948, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.073, 0.001, 0.001, 0.001, 0.025, 0.007, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.014, 0.001, 0.005, 0.002, 0.0001, 0.001, 0.004, 0.001, 0.001, 0.027, 0.001, 0.026, 0.15, 0.041, 0.008, 0.01, 0.001, 1.309, 0.09, 0.0001, 0.002, 0.017, 0.105, 0.002, 0.002, 0.07, 0.023, 0.0001, 0.002, 0.004, 0.001, 0.016, 0.015, 0.003, 0.033, 0.029, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.044, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.024, 1.987, 0.001, 0.015, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.001, 0.001, 0.0001, 0.002, 0.001, 0.006, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.07, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.055, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.983, 0.003, 0.019, 0.0001, 0.0001, 0.001, 0.0001, 0.005, 0.015, 0.015, 0.0001, 0.0001, 0.196, 0.05, 0.162, 0.001, 0.012, 0.032, 0.016, 0.014, 0.013, 0.015, 0.012, 0.01, 0.008, 0.011, 0.012, 0.002, 0.001, 0.001, 0.001, 0.003, 0.0001, 0.014, 0.008, 0.005, 0.012, 0.009, 0.003, 0.003, 0.008, 0.003, 0.003, 0.005, 0.005, 0.006, 0.016, 0.004, 0.013, 0.0001, 0.005, 0.023, 0.015, 0.011, 0.002, 0.003, 0.0001, 0.036, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.761, 0.196, 0.276, 0.305, 0.454, 0.017, 0.165, 0.648, 0.799, 0.068, 0.545, 0.11, 0.265, 0.512, 0.167, 0.383, 0.0001, 0.263, 0.542, 0.555, 0.3, 0.254, 0.017, 0.002, 0.396, 0.01, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.585, 1.349, 0.631, 0.309, 0.001, 0.704, 0.099, 0.738, 0.496, 0.135, 0.002, 0.598, 0.015, 3.249, 0.0001, 0.471, 0.004, 0.001, 0.0001, 0.007, 0.008, 1.087, 0.017, 0.945, 0.067, 0.032, 0.055, 0.004, 0.076, 0.002, 0.011, 0.133, 0.012, 0.507, 0.001, 0.315, 18.309, 9.394, 0.355, 1.002, 0.59, 0.225, 0.335, 0.42, 0.344, 0.341, 0.537, 1.408, 2.487, 0.078, 0.724, 0.001, 0.0001, 0.527, 0.043, 0.432, 2.004, 0.558, 0.0001, 0.0001, 0.014, 0.0001, 1.176, 0.438, 0.0001, 0.0001, 0.0001, 0.095, 0.876, 0.019, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 26.149, 0.497, 0.061, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pih": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.022, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.521, 0.009, 0.204, 0.0001, 0.0001, 0.004, 0.001, 2.454, 0.291, 0.293, 0.002, 0.001, 0.888, 0.295, 1.488, 0.018, 0.431, 0.479, 0.302, 0.13, 0.163, 0.176, 0.152, 0.154, 0.17, 0.228, 0.107, 0.024, 0.002, 0.004, 0.002, 0.0001, 0.0001, 0.58, 0.327, 0.222, 0.2, 0.438, 0.166, 0.153, 0.179, 0.225, 0.154, 0.281, 0.167, 0.407, 0.329, 0.234, 0.386, 0.004, 0.196, 0.528, 0.379, 0.134, 0.065, 0.116, 0.0001, 0.083, 0.034, 0.002, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 8.21, 0.769, 1.021, 1.638, 6.843, 0.747, 0.93, 1.82, 7.969, 0.261, 1.7, 3.13, 1.378, 5.431, 3.567, 1.341, 0.022, 3.668, 4.749, 4.715, 2.394, 0.289, 0.906, 0.045, 1.135, 0.094, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.092, 0.039, 0.046, 0.01, 0.019, 0.008, 0.006, 0.003, 0.009, 0.001, 0.001, 0.017, 0.011, 0.007, 0.006, 0.02, 0.015, 0.009, 0.001, 0.019, 0.012, 0.0001, 0.001, 0.006, 0.003, 0.028, 0.004, 0.009, 0.002, 0.007, 0.002, 0.006, 0.077, 0.019, 0.006, 0.004, 0.002, 0.001, 0.001, 0.004, 0.006, 0.035, 0.017, 0.004, 0.004, 0.012, 0.004, 0.0001, 0.082, 0.011, 0.044, 0.037, 0.018, 0.034, 0.007, 0.017, 0.057, 0.008, 0.047, 0.04, 0.021, 0.031, 0.077, 0.012, 0.0001, 0.0001, 0.098, 0.125, 0.02, 0.026, 0.002, 0.0001, 0.0001, 0.016, 0.007, 0.004, 0.0001, 0.001, 0.0001, 0.0001, 0.45, 0.193, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.026, 0.048, 0.0001, 0.0001, 0.003, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.002, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pms": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.299, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.832, 0.001, 0.162, 0.0001, 0.0001, 0.004, 0.001, 1.011, 0.156, 0.156, 0.0001, 0.0001, 1.253, 0.787, 0.992, 0.342, 0.318, 0.594, 0.327, 0.281, 0.214, 0.205, 0.231, 0.17, 0.187, 0.641, 0.121, 0.014, 0.149, 0.035, 0.149, 0.034, 0.0001, 0.513, 0.293, 0.3, 0.101, 0.054, 0.096, 0.128, 0.035, 0.079, 0.03, 0.029, 0.305, 0.216, 0.126, 0.053, 0.205, 0.008, 0.226, 0.291, 0.098, 0.028, 0.104, 0.023, 0.014, 0.012, 0.01, 0.002, 0.0001, 0.002, 0.0001, 0.011, 0.0001, 9.348, 0.753, 2.346, 3.335, 4.488, 0.731, 0.919, 0.76, 4.936, 0.31, 0.385, 3.85, 2.042, 6.393, 3.543, 1.375, 0.084, 3.728, 4.566, 4.041, 1.559, 0.688, 0.145, 0.036, 0.118, 0.134, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.165, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.007, 0.001, 0.059, 0.002, 0.001, 0.004, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.02, 0.115, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.661, 0.007, 0.005, 0.001, 0.004, 0.0001, 0.001, 0.004, 0.295, 0.64, 0.002, 2.121, 0.312, 0.006, 0.001, 0.001, 0.002, 0.006, 0.658, 0.012, 0.019, 0.001, 0.003, 0.0001, 0.001, 0.07, 0.002, 0.002, 0.005, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.286, 4.639, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.002, 0.007, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.137, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pnb": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.056, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.3, 0.001, 0.061, 0.0001, 0.0001, 0.004, 0.0001, 0.019, 0.084, 0.084, 0.0001, 0.001, 0.02, 0.088, 0.041, 0.008, 0.132, 0.201, 0.102, 0.067, 0.062, 0.067, 0.058, 0.058, 0.067, 0.099, 0.044, 0.011, 0.014, 0.019, 0.014, 0.0001, 0.0001, 0.005, 0.015, 0.004, 0.007, 0.004, 0.002, 0.006, 0.002, 0.006, 0.004, 0.001, 0.002, 0.002, 0.003, 0.002, 0.003, 0.0001, 0.004, 0.004, 0.005, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.0001, 0.016, 0.0001, 0.016, 0.0001, 0.001, 0.001, 0.078, 0.007, 0.025, 0.032, 0.067, 0.025, 0.012, 0.031, 0.075, 0.001, 0.005, 0.047, 0.03, 0.057, 0.061, 0.011, 0.001, 0.056, 0.035, 0.078, 0.015, 0.016, 0.005, 0.001, 0.026, 0.001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.037, 1.78, 0.208, 0.002, 1.668, 1.155, 3.673, 0.01, 3.168, 0.001, 0.005, 0.003, 4.417, 0.001, 0.019, 0.031, 0.026, 0.339, 2.06, 0.022, 0.634, 0.001, 0.0001, 0.0001, 0.015, 0.009, 0.0001, 0.004, 0.002, 0.002, 0.0001, 0.002, 0.01, 0.033, 0.198, 0.001, 0.068, 0.002, 0.419, 5.965, 1.027, 1.695, 1.567, 0.022, 0.752, 0.178, 0.132, 2.73, 0.025, 2.399, 0.214, 1.507, 0.366, 0.211, 0.103, 0.103, 0.037, 0.788, 1.21, 0.001, 0.001, 0.002, 1.61, 0.001, 0.0001, 0.0001, 0.01, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.007, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 17.991, 10.598, 5.545, 8.408, 0.0001, 0.004, 0.0001, 0.0001, 0.008, 0.001, 0.037, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "pnt": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.111, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 8.571, 0.001, 0.271, 0.0001, 0.0001, 0.001, 0.001, 0.535, 0.166, 0.167, 0.003, 0.001, 0.374, 0.066, 0.728, 0.006, 0.22, 0.256, 0.213, 0.123, 0.09, 0.093, 0.091, 0.105, 0.105, 0.136, 0.082, 0.003, 0.006, 0.005, 0.006, 0.0001, 0.0001, 0.027, 0.009, 0.018, 0.007, 0.008, 0.002, 0.007, 0.007, 0.011, 0.005, 0.01, 0.006, 0.014, 0.004, 0.003, 0.01, 0.001, 0.009, 0.009, 0.01, 0.001, 0.002, 0.009, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.102, 0.019, 0.03, 0.031, 0.116, 0.01, 0.025, 0.028, 0.096, 0.002, 0.026, 0.047, 0.026, 0.078, 0.081, 0.022, 0.002, 0.09, 0.05, 0.057, 0.045, 0.011, 0.005, 0.004, 0.012, 0.009, 0.0001, 0.182, 0.0001, 0.0001, 0.0001, 1.086, 2.28, 1.131, 1.649, 3.333, 0.911, 0.24, 0.496, 0.069, 0.319, 0.04, 0.001, 0.823, 0.357, 0.222, 0.002, 0.015, 0.24, 0.073, 0.124, 0.043, 0.159, 0.008, 0.102, 0.032, 0.07, 0.198, 0.037, 0.134, 0.044, 0.005, 0.136, 0.134, 0.042, 0.001, 0.198, 0.245, 0.005, 0.018, 0.079, 0.003, 0.008, 0.001, 0.042, 1.012, 0.786, 0.269, 1.272, 0.017, 4.369, 0.221, 0.746, 0.384, 2.578, 0.144, 1.385, 0.301, 1.937, 1.463, 1.533, 1.22, 4.421, 0.103, 3.268, 0.0001, 0.0001, 0.091, 0.022, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 29.379, 12.776, 0.049, 0.015, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.003, 0.011, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.059, 0.041, 0.0001, 0.001, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ps":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.579, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.932, 0.004, 0.044, 0.0001, 0.0001, 0.003, 0.0001, 0.002, 0.118, 0.118, 0.001, 0.001, 0.026, 0.037, 0.443, 0.009, 0.022, 0.03, 0.021, 0.014, 0.011, 0.012, 0.01, 0.009, 0.01, 0.013, 0.062, 0.001, 0.002, 0.005, 0.002, 0.0001, 0.0001, 0.015, 0.007, 0.011, 0.007, 0.006, 0.005, 0.004, 0.007, 0.009, 0.002, 0.003, 0.005, 0.01, 0.006, 0.004, 0.009, 0.001, 0.006, 0.013, 0.009, 0.003, 0.002, 0.003, 0.001, 0.001, 0.001, 0.004, 0.0001, 0.004, 0.0001, 0.003, 0.0001, 0.147, 0.023, 0.055, 0.054, 0.165, 0.027, 0.031, 0.061, 0.131, 0.002, 0.012, 0.073, 0.048, 0.109, 0.113, 0.034, 0.002, 0.103, 0.097, 0.116, 0.047, 0.015, 0.017, 0.005, 0.027, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.103, 0.528, 0.202, 0.231, 2.393, 1.822, 2.655, 3.163, 4.608, 0.307, 2.451, 0.006, 1.513, 0.136, 0.015, 0.009, 1.675, 0.004, 0.009, 0.507, 0.005, 0.0001, 0.154, 0.001, 0.093, 0.002, 0.229, 0.007, 0.005, 0.003, 0.0001, 0.006, 0.024, 0.025, 0.048, 0.014, 0.025, 0.008, 0.038, 4.145, 0.839, 1.375, 1.43, 0.077, 0.25, 0.229, 0.647, 2.983, 0.085, 2.528, 0.449, 1.14, 0.525, 0.146, 0.073, 0.106, 0.064, 0.333, 0.407, 0.02, 0.265, 0.005, 1.278, 0.002, 0.0001, 0.0001, 0.016, 0.003, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.028, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 16.081, 19.012, 3.763, 3.368, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.026, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.038, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "qu":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.204, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.108, 0.002, 0.638, 0.0001, 0.0001, 0.004, 0.002, 0.39, 0.494, 0.494, 0.05, 0.002, 1.04, 0.188, 0.977, 0.036, 0.261, 0.409, 0.214, 0.137, 0.12, 0.137, 0.117, 0.112, 0.136, 0.208, 0.401, 0.061, 0.041, 0.006, 0.041, 0.004, 0.0001, 0.371, 0.173, 0.36, 0.119, 0.076, 0.064, 0.097, 0.189, 0.192, 0.096, 0.255, 0.187, 0.305, 0.078, 0.042, 0.428, 0.148, 0.146, 0.31, 0.198, 0.198, 0.061, 0.174, 0.014, 0.102, 0.014, 0.014, 0.0001, 0.015, 0.0001, 0.002, 0.0001, 14.813, 0.229, 1.579, 0.795, 1.296, 0.084, 0.286, 2.331, 7.773, 0.067, 3.004, 4.09, 3.086, 5.006, 1.165, 2.96, 3.746, 3.293, 3.447, 3.494, 5.678, 0.135, 1.682, 0.024, 2.244, 0.111, 0.0001, 0.003, 0.001, 0.0001, 0.0001, 0.055, 0.016, 0.009, 0.013, 0.008, 0.006, 0.004, 0.006, 0.004, 0.004, 0.003, 0.002, 0.006, 0.007, 0.003, 0.002, 0.012, 0.023, 0.002, 0.011, 0.004, 0.004, 0.003, 0.002, 0.004, 0.015, 0.003, 0.003, 0.005, 0.003, 0.002, 0.003, 0.038, 0.068, 0.004, 0.005, 0.014, 0.005, 0.006, 0.009, 0.007, 0.056, 0.005, 0.005, 0.005, 0.075, 0.004, 0.006, 0.014, 0.34, 0.013, 0.069, 0.007, 0.008, 0.006, 0.005, 0.016, 0.009, 0.022, 0.008, 0.011, 0.009, 0.01, 0.01, 0.0001, 0.0001, 0.026, 0.649, 0.01, 0.009, 0.001, 0.001, 0.0001, 0.002, 0.001, 0.008, 0.002, 0.0001, 0.042, 0.02, 0.051, 0.016, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.006, 0.016, 0.012, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.031, 0.016, 0.047, 0.001, 0.001, 0.006, 0.005, 0.002, 0.002, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "rm":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.612, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.511, 0.003, 0.167, 0.0001, 0.0001, 0.015, 0.0001, 0.104, 0.151, 0.151, 0.003, 0.0001, 0.439, 0.065, 0.823, 0.012, 0.199, 0.275, 0.114, 0.07, 0.072, 0.078, 0.067, 0.073, 0.099, 0.138, 0.045, 0.054, 0.005, 0.002, 0.005, 0.002, 0.0001, 0.139, 0.111, 0.151, 0.1, 0.17, 0.069, 0.113, 0.045, 0.217, 0.028, 0.029, 0.208, 0.121, 0.055, 0.036, 0.132, 0.042, 0.086, 0.226, 0.106, 0.051, 0.07, 0.023, 0.003, 0.003, 0.008, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.001, 11.404, 0.601, 3.031, 3.677, 6.353, 0.757, 1.578, 1.412, 6.549, 0.076, 0.068, 4.778, 1.866, 6.046, 2.159, 1.856, 0.265, 4.97, 5.963, 4.375, 3.712, 1.407, 0.03, 0.131, 0.049, 0.802, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.828, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.043, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.675, 0.0001, 0.001, 0.003, 0.001, 0.001, 0.0001, 0.294, 0.003, 0.002, 0.0001, 0.006, 0.001, 0.001, 0.002, 0.333, 0.017, 0.002, 0.02, 0.128, 0.003, 0.0001, 0.001, 0.004, 0.002, 0.027, 0.004, 0.001, 0.001, 0.012, 0.0001, 0.001, 0.054, 0.054, 0.02, 0.042, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.079, 0.841, 0.004, 0.004, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.828, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "rmy": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.439, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.022, 0.009, 0.624, 0.001, 0.0001, 0.014, 0.003, 0.015, 0.443, 0.441, 0.003, 0.003, 1.038, 0.232, 0.983, 0.037, 0.243, 0.24, 0.172, 0.071, 0.075, 0.081, 0.064, 0.056, 0.058, 0.153, 0.179, 0.028, 0.0001, 0.007, 0.0001, 0.007, 0.0001, 0.429, 0.231, 0.183, 0.167, 0.219, 0.086, 0.091, 0.078, 0.193, 0.094, 0.424, 0.255, 0.23, 0.124, 0.22, 0.316, 0.009, 0.404, 0.577, 0.193, 0.066, 0.142, 0.014, 0.008, 0.107, 0.022, 0.003, 0.0001, 0.004, 0.0001, 0.001, 0.001, 10.986, 1.012, 0.769, 2.129, 6.975, 0.334, 0.958, 2.547, 6.287, 0.364, 2.952, 3.16, 1.944, 5.241, 5.031, 1.512, 0.108, 4.343, 3.649, 3.301, 2.127, 1.805, 0.051, 0.119, 2.702, 0.234, 0.0001, 0.003, 0.001, 0.001, 0.0001, 0.065, 0.024, 0.024, 0.136, 0.011, 0.006, 0.01, 0.023, 0.009, 0.001, 0.005, 0.016, 0.01, 0.054, 0.02, 0.004, 0.003, 0.007, 0.001, 0.025, 0.008, 0.009, 0.004, 0.007, 0.004, 0.054, 0.005, 0.044, 0.006, 0.002, 0.003, 0.008, 0.038, 0.056, 0.061, 0.004, 0.069, 0.023, 0.011, 0.016, 0.012, 0.016, 0.02, 0.006, 0.003, 0.014, 0.084, 0.01, 0.051, 0.028, 0.039, 0.03, 0.013, 0.014, 0.009, 0.007, 0.027, 0.007, 0.021, 0.023, 0.024, 0.034, 0.062, 0.014, 0.0001, 0.0001, 0.036, 0.28, 0.165, 0.061, 0.0001, 0.011, 0.089, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.007, 0.003, 0.242, 0.129, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.026, 0.02, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.128, 0.005, 0.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.004, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "rn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.466, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.984, 0.029, 0.062, 0.0001, 0.0001, 0.003, 0.0001, 0.574, 0.079, 0.089, 0.011, 0.0001, 1.075, 0.049, 1.054, 0.052, 0.102, 0.296, 0.219, 0.136, 0.097, 0.085, 0.071, 0.069, 0.063, 0.086, 0.222, 0.076, 0.048, 0.0001, 0.048, 0.055, 0.0001, 0.35, 0.153, 0.043, 0.033, 0.087, 0.042, 0.05, 0.106, 0.336, 0.016, 0.136, 0.035, 0.208, 0.24, 0.02, 0.055, 0.001, 0.138, 0.132, 0.082, 0.29, 0.029, 0.019, 0.0001, 0.195, 0.032, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 10.973, 3.104, 0.817, 1.265, 5.117, 0.295, 2.193, 1.806, 7.318, 0.542, 2.677, 0.668, 3.393, 5.179, 4.081, 0.677, 0.004, 4.518, 2.135, 2.161, 5.935, 1.185, 2.07, 0.007, 2.3, 1.432, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.174, 0.023, 0.002, 0.005, 0.007, 0.004, 0.003, 0.003, 0.006, 0.0001, 0.004, 0.0001, 0.015, 0.271, 0.012, 0.002, 0.008, 0.004, 0.004, 0.008, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.284, 0.0001, 0.196, 0.005, 0.002, 0.004, 0.014, 0.082, 0.437, 0.009, 0.0001, 0.0001, 0.006, 0.001, 0.018, 0.002, 0.13, 0.002, 0.17, 0.0001, 0.586, 0.007, 0.108, 0.004, 0.019, 0.001, 0.014, 0.007, 0.001, 0.01, 0.001, 0.001, 0.002, 0.049, 0.153, 0.015, 0.123, 0.121, 0.0001, 0.0001, 0.0001, 0.386, 1.335, 0.539, 0.45, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.015, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.02, 0.053, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.167, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "rue": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.059, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.931, 0.003, 0.134, 0.004, 0.0001, 0.006, 0.0001, 0.004, 0.175, 0.175, 0.004, 0.001, 0.601, 0.077, 0.708, 0.007, 0.16, 0.303, 0.147, 0.097, 0.093, 0.095, 0.084, 0.09, 0.099, 0.137, 0.031, 0.011, 0.003, 0.006, 0.002, 0.001, 0.0001, 0.009, 0.006, 0.011, 0.005, 0.005, 0.004, 0.005, 0.007, 0.017, 0.002, 0.003, 0.006, 0.008, 0.01, 0.004, 0.008, 0.0001, 0.006, 0.012, 0.008, 0.004, 0.005, 0.003, 0.005, 0.001, 0.001, 0.002, 0.011, 0.003, 0.0001, 0.004, 0.0001, 0.091, 0.013, 0.033, 0.032, 0.099, 0.015, 0.014, 0.02, 0.081, 0.006, 0.012, 0.045, 0.028, 0.055, 0.071, 0.018, 0.002, 0.069, 0.058, 0.052, 0.039, 0.016, 0.006, 0.004, 0.014, 0.011, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 2.51, 1.935, 2.11, 1.119, 0.168, 0.507, 0.468, 0.486, 0.356, 0.046, 0.009, 1.422, 0.678, 0.003, 0.239, 0.784, 0.089, 0.301, 0.121, 0.074, 0.371, 0.035, 1.647, 0.529, 0.012, 0.009, 0.09, 0.04, 0.101, 0.07, 0.061, 0.137, 0.108, 0.152, 0.055, 0.234, 0.024, 0.016, 0.017, 0.032, 0.025, 0.005, 0.001, 0.015, 0.002, 0.004, 0.009, 0.015, 3.672, 0.621, 2.19, 0.526, 1.267, 2.143, 0.348, 0.764, 1.53, 0.62, 1.849, 1.595, 1.255, 2.617, 4.166, 1.0, 0.0001, 0.0001, 0.065, 0.026, 0.003, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.006, 0.0001, 0.02, 0.009, 27.547, 15.28, 0.159, 0.0001, 0.0001, 0.001, 0.001, 0.004, 0.006, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.004, 0.115, 0.004, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "rw":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.278, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.768, 0.007, 0.605, 0.0001, 0.0001, 0.01, 0.005, 0.156, 0.48, 0.479, 0.0001, 0.001, 0.554, 0.104, 0.846, 0.03, 0.199, 0.216, 0.155, 0.113, 0.085, 0.082, 0.092, 0.069, 0.073, 0.138, 0.279, 0.158, 0.014, 0.006, 0.014, 0.022, 0.0001, 0.463, 0.203, 0.115, 0.093, 0.067, 0.068, 0.134, 0.078, 0.521, 0.056, 0.247, 0.076, 0.283, 0.278, 0.063, 0.134, 0.006, 0.219, 0.18, 0.135, 0.469, 0.053, 0.038, 0.003, 0.047, 0.033, 0.002, 0.005, 0.003, 0.0001, 0.005, 0.003, 10.187, 2.795, 0.85, 1.072, 4.678, 0.399, 2.704, 1.553, 8.747, 0.332, 2.605, 0.938, 3.443, 4.833, 3.164, 0.42, 0.036, 4.477, 2.07, 2.396, 6.084, 0.329, 1.941, 0.023, 2.813, 1.358, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.976, 0.03, 0.025, 0.03, 0.043, 0.049, 0.025, 0.021, 0.028, 0.009, 0.036, 0.011, 0.018, 0.014, 0.026, 0.013, 0.02, 0.015, 0.01, 0.15, 0.008, 0.015, 0.009, 0.008, 0.011, 0.787, 0.007, 0.008, 0.021, 0.011, 0.011, 0.034, 0.016, 0.016, 0.009, 0.011, 0.025, 0.012, 0.025, 0.083, 0.028, 0.057, 0.013, 0.014, 0.038, 0.019, 0.024, 0.039, 0.055, 0.067, 0.03, 0.036, 0.018, 0.034, 0.017, 0.013, 0.045, 0.024, 0.03, 0.026, 0.034, 0.02, 0.028, 0.012, 0.0001, 0.0001, 0.03, 0.13, 0.052, 0.028, 0.001, 0.001, 0.001, 0.009, 0.002, 0.001, 0.003, 0.0001, 0.02, 0.007, 0.21, 0.107, 0.008, 0.005, 0.001, 0.006, 0.005, 0.025, 0.234, 0.162, 0.005, 0.009, 0.0001, 0.0001, 0.009, 0.0001, 0.108, 0.055, 0.961, 0.001, 0.002, 0.013, 0.008, 0.004, 0.003, 0.002, 0.001, 0.001, 0.003, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sa":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.358, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.441, 0.003, 0.019, 0.0001, 0.0001, 0.003, 0.0001, 0.034, 0.04, 0.042, 0.0001, 0.001, 0.189, 0.124, 0.061, 0.009, 0.003, 0.004, 0.003, 0.002, 0.002, 0.001, 0.001, 0.001, 0.001, 0.002, 0.022, 0.001, 0.005, 0.003, 0.005, 0.005, 0.0001, 0.002, 0.001, 0.007, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.003, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.036, 0.006, 0.009, 0.018, 0.033, 0.003, 0.005, 0.01, 0.026, 0.001, 0.003, 0.015, 0.009, 0.022, 0.022, 0.013, 0.0001, 0.02, 0.014, 0.02, 0.007, 0.002, 0.005, 0.001, 0.004, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.485, 0.531, 0.737, 0.892, 0.001, 0.437, 0.14, 1.014, 0.103, 0.083, 0.003, 0.31, 0.053, 4.186, 0.001, 0.123, 0.004, 0.001, 0.0001, 0.016, 0.005, 0.929, 0.077, 0.397, 0.036, 0.098, 0.308, 0.026, 0.311, 0.017, 0.076, 0.146, 0.037, 0.11, 0.008, 0.362, 26.378, 7.803, 0.723, 0.32, 1.5, 0.025, 0.84, 0.05, 0.176, 0.36, 1.148, 1.481, 1.847, 0.0001, 0.431, 0.017, 0.001, 1.168, 0.412, 0.409, 1.347, 0.264, 0.0001, 0.0001, 0.002, 0.029, 2.443, 1.602, 0.0001, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 31.344, 0.0001, 0.071, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sah": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.686, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.508, 0.002, 0.063, 0.0001, 0.0001, 0.006, 0.0001, 0.001, 0.091, 0.092, 0.0001, 0.001, 0.559, 0.22, 0.713, 0.008, 0.141, 0.211, 0.115, 0.061, 0.06, 0.065, 0.051, 0.05, 0.056, 0.124, 0.035, 0.013, 0.006, 0.002, 0.006, 0.002, 0.0001, 0.005, 0.004, 0.005, 0.003, 0.002, 0.002, 0.002, 0.002, 0.017, 0.001, 0.002, 0.002, 0.004, 0.003, 0.002, 0.003, 0.0001, 0.002, 0.006, 0.004, 0.001, 0.004, 0.001, 0.007, 0.002, 0.0001, 0.003, 0.0001, 0.003, 0.0001, 0.004, 0.0001, 0.029, 0.008, 0.012, 0.008, 0.028, 0.005, 0.006, 0.038, 0.023, 0.001, 0.004, 0.015, 0.01, 0.02, 0.025, 0.007, 0.0001, 0.024, 0.015, 0.017, 0.012, 0.003, 0.003, 0.001, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.31, 1.601, 3.034, 2.129, 0.054, 0.947, 0.056, 0.303, 0.028, 0.008, 0.002, 2.384, 0.325, 2.811, 0.018, 0.129, 0.129, 0.118, 0.035, 0.051, 0.111, 0.426, 0.011, 0.007, 0.063, 0.002, 0.123, 0.02, 0.075, 0.059, 0.083, 0.041, 0.09, 0.172, 0.066, 0.042, 0.018, 0.342, 0.005, 0.017, 0.028, 0.688, 0.0001, 0.065, 0.003, 0.027, 0.027, 0.911, 6.03, 1.253, 0.256, 0.681, 0.862, 0.464, 0.024, 0.065, 2.76, 0.764, 1.431, 3.082, 0.589, 3.175, 2.114, 0.327, 0.0001, 0.0001, 0.173, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.002, 0.0001, 0.002, 0.001, 24.486, 17.038, 2.234, 0.706, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.102, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sc":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.057, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.585, 0.005, 0.337, 0.0001, 0.0001, 0.01, 0.001, 0.519, 0.179, 0.179, 0.0001, 0.001, 0.997, 0.097, 0.746, 0.025, 0.237, 0.292, 0.15, 0.087, 0.082, 0.09, 0.079, 0.083, 0.093, 0.166, 0.067, 0.032, 0.007, 0.002, 0.008, 0.003, 0.0001, 0.224, 0.132, 0.261, 0.092, 0.103, 0.086, 0.108, 0.028, 0.255, 0.028, 0.024, 0.103, 0.181, 0.091, 0.054, 0.171, 0.006, 0.093, 0.455, 0.116, 0.064, 0.058, 0.024, 0.025, 0.008, 0.012, 0.012, 0.0001, 0.012, 0.0001, 0.0001, 0.0001, 9.363, 0.921, 2.394, 4.179, 7.513, 0.745, 1.075, 0.764, 6.94, 0.075, 0.096, 1.956, 1.851, 5.606, 4.069, 1.688, 0.018, 4.534, 7.393, 5.178, 5.316, 0.445, 0.037, 0.029, 0.067, 0.854, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.107, 0.01, 0.006, 0.008, 0.004, 0.002, 0.002, 0.003, 0.006, 0.002, 0.002, 0.002, 0.006, 0.004, 0.002, 0.001, 0.003, 0.002, 0.003, 0.008, 0.004, 0.001, 0.002, 0.001, 0.003, 0.058, 0.001, 0.002, 0.016, 0.017, 0.001, 0.001, 0.306, 0.009, 0.002, 0.003, 0.003, 0.002, 0.001, 0.004, 0.232, 0.016, 0.003, 0.007, 0.252, 0.007, 0.001, 0.002, 0.008, 0.007, 0.176, 0.008, 0.002, 0.004, 0.003, 0.006, 0.005, 0.096, 0.007, 0.007, 0.006, 0.004, 0.004, 0.004, 0.0001, 0.0001, 0.032, 1.097, 0.007, 0.006, 0.0001, 0.0001, 0.0001, 0.011, 0.003, 0.007, 0.001, 0.0001, 0.018, 0.009, 0.023, 0.009, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.003, 0.006, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.096, 0.007, 0.001, 0.002, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "scn": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.769, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.859, 0.005, 0.376, 0.001, 0.0001, 0.006, 0.001, 0.62, 0.187, 0.189, 0.001, 0.001, 0.862, 0.093, 0.813, 0.026, 0.237, 0.31, 0.158, 0.093, 0.09, 0.099, 0.092, 0.087, 0.104, 0.161, 0.085, 0.032, 0.027, 0.008, 0.034, 0.003, 0.0001, 0.211, 0.12, 0.299, 0.095, 0.077, 0.106, 0.122, 0.053, 0.143, 0.031, 0.023, 0.333, 0.207, 0.158, 0.044, 0.198, 0.02, 0.123, 0.32, 0.125, 0.061, 0.101, 0.022, 0.022, 0.008, 0.012, 0.019, 0.0001, 0.019, 0.0001, 0.007, 0.001, 8.698, 0.715, 3.649, 2.751, 2.764, 0.729, 1.223, 0.71, 11.435, 0.098, 0.087, 3.105, 2.009, 5.881, 1.955, 1.977, 0.125, 4.751, 3.262, 4.998, 7.156, 1.012, 0.04, 0.022, 0.077, 0.978, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.123, 0.005, 0.005, 0.005, 0.003, 0.001, 0.002, 0.001, 0.009, 0.002, 0.001, 0.001, 0.004, 0.003, 0.001, 0.001, 0.001, 0.002, 0.002, 0.01, 0.003, 0.001, 0.002, 0.001, 0.012, 0.059, 0.001, 0.002, 0.014, 0.013, 0.001, 0.001, 0.271, 0.006, 0.312, 0.002, 0.001, 0.001, 0.001, 0.004, 0.478, 0.013, 0.046, 0.017, 0.359, 0.007, 0.096, 0.002, 0.008, 0.004, 0.23, 0.006, 0.189, 0.003, 0.002, 0.003, 0.004, 0.145, 0.006, 0.184, 0.003, 0.004, 0.002, 0.003, 0.0001, 0.0001, 0.038, 2.342, 0.007, 0.007, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.018, 0.008, 0.018, 0.009, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.107, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sco": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.424, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.792, 0.003, 0.394, 0.001, 0.002, 0.018, 0.003, 0.122, 0.265, 0.265, 0.001, 0.001, 1.079, 0.194, 0.878, 0.014, 0.365, 0.437, 0.262, 0.125, 0.119, 0.131, 0.116, 0.116, 0.135, 0.238, 0.065, 0.052, 0.004, 0.002, 0.004, 0.001, 0.0001, 0.38, 0.213, 0.323, 0.162, 0.132, 0.149, 0.164, 0.147, 0.281, 0.103, 0.117, 0.162, 0.281, 0.134, 0.098, 0.228, 0.015, 0.18, 0.416, 0.389, 0.071, 0.084, 0.096, 0.009, 0.031, 0.022, 0.004, 0.0001, 0.004, 0.0001, 0.0001, 0.001, 7.525, 0.973, 2.32, 1.915, 9.145, 0.867, 0.966, 3.175, 6.858, 0.086, 0.556, 2.986, 1.72, 5.779, 4.854, 1.317, 0.066, 4.925, 4.607, 6.432, 2.109, 0.676, 1.003, 0.125, 1.018, 0.144, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.052, 0.009, 0.005, 0.004, 0.004, 0.003, 0.002, 0.004, 0.002, 0.003, 0.002, 0.001, 0.002, 0.005, 0.001, 0.002, 0.001, 0.002, 0.001, 0.033, 0.006, 0.001, 0.002, 0.002, 0.002, 0.009, 0.001, 0.002, 0.003, 0.003, 0.001, 0.005, 0.039, 0.026, 0.003, 0.006, 0.015, 0.004, 0.002, 0.007, 0.005, 0.025, 0.002, 0.006, 0.002, 0.019, 0.001, 0.002, 0.01, 0.012, 0.013, 0.016, 0.004, 0.005, 0.011, 0.002, 0.007, 0.003, 0.007, 0.003, 0.009, 0.004, 0.004, 0.002, 0.0001, 0.0001, 0.051, 0.152, 0.018, 0.014, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.011, 0.005, 0.019, 0.007, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.009, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.005, 0.05, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sd":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.527, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.452, 0.004, 0.026, 0.001, 0.0001, 0.002, 0.0001, 0.004, 0.108, 0.108, 0.004, 0.001, 0.009, 0.252, 0.565, 0.015, 0.09, 0.18, 0.083, 0.051, 0.047, 0.052, 0.047, 0.048, 0.055, 0.112, 0.038, 0.003, 0.004, 0.004, 0.004, 0.0001, 0.0001, 0.01, 0.007, 0.009, 0.005, 0.004, 0.003, 0.004, 0.004, 0.005, 0.002, 0.003, 0.004, 0.007, 0.004, 0.003, 0.007, 0.001, 0.004, 0.011, 0.007, 0.002, 0.002, 0.003, 0.0001, 0.001, 0.0001, 0.005, 0.0001, 0.005, 0.001, 0.003, 0.0001, 0.093, 0.018, 0.025, 0.03, 0.086, 0.014, 0.018, 0.031, 0.075, 0.002, 0.009, 0.041, 0.026, 0.061, 0.064, 0.018, 0.001, 0.061, 0.048, 0.062, 0.025, 0.01, 0.009, 0.005, 0.016, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.358, 0.355, 0.367, 0.044, 1.492, 1.56, 3.771, 2.24, 3.043, 0.003, 5.411, 0.006, 0.747, 0.045, 0.229, 0.346, 0.139, 0.006, 0.013, 0.002, 0.041, 0.001, 0.001, 0.003, 0.036, 0.237, 0.0001, 0.007, 0.063, 0.063, 0.001, 0.009, 0.066, 0.346, 0.511, 0.005, 0.013, 0.004, 0.6, 4.552, 0.858, 0.458, 2.417, 0.053, 1.486, 0.357, 0.248, 1.426, 0.07, 2.412, 0.238, 1.475, 0.403, 0.209, 0.078, 0.141, 0.068, 0.553, 0.222, 0.578, 0.001, 0.647, 1.291, 0.291, 0.0001, 0.0001, 0.065, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 17.302, 19.772, 4.118, 0.84, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.198, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.104, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "se":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.476, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.534, 0.002, 0.233, 0.001, 0.001, 0.008, 0.001, 0.011, 0.255, 0.258, 0.0001, 0.001, 1.05, 0.297, 1.247, 0.015, 0.373, 0.513, 0.317, 0.174, 0.148, 0.157, 0.145, 0.145, 0.159, 0.258, 0.106, 0.008, 0.018, 0.002, 0.018, 0.002, 0.0001, 0.188, 0.166, 0.071, 0.251, 0.09, 0.093, 0.235, 0.162, 0.082, 0.134, 0.245, 0.18, 0.186, 0.183, 0.112, 0.147, 0.004, 0.206, 0.487, 0.123, 0.053, 0.171, 0.019, 0.003, 0.015, 0.006, 0.005, 0.0001, 0.005, 0.0001, 0.001, 0.0001, 10.038, 0.915, 0.217, 3.327, 5.602, 0.262, 2.678, 1.589, 6.671, 1.512, 2.136, 5.043, 2.364, 3.492, 4.102, 0.745, 0.01, 2.956, 3.613, 3.601, 3.337, 2.349, 0.035, 0.018, 0.282, 0.057, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.058, 0.044, 0.003, 0.004, 0.008, 0.009, 0.002, 0.003, 0.002, 0.002, 0.001, 0.119, 0.035, 0.369, 0.001, 0.002, 0.002, 0.302, 0.004, 0.024, 0.002, 0.002, 0.006, 0.001, 0.007, 0.006, 0.003, 0.005, 0.009, 0.024, 0.001, 0.003, 0.073, 3.336, 0.013, 0.002, 0.299, 0.044, 0.019, 0.052, 0.004, 0.017, 0.001, 0.011, 0.001, 0.006, 0.001, 0.004, 0.019, 0.004, 0.042, 0.006, 0.01, 0.008, 0.063, 0.001, 0.053, 0.001, 0.006, 0.009, 0.008, 0.006, 0.098, 0.001, 0.0001, 0.0001, 0.118, 3.226, 0.711, 1.0, 0.0001, 0.004, 0.002, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.008, 0.003, 0.027, 0.008, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.004, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.008, 0.055, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sg":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.098, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.879, 0.044, 0.115, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.198, 0.211, 0.0001, 0.0001, 0.85, 0.464, 1.499, 0.007, 0.524, 0.5, 0.381, 0.203, 0.246, 0.244, 0.244, 0.191, 0.262, 0.229, 0.082, 0.038, 0.0001, 0.0001, 0.0001, 0.029, 0.0001, 0.282, 0.315, 0.126, 0.106, 0.101, 0.06, 0.092, 0.086, 0.092, 0.093, 0.251, 0.348, 0.227, 0.266, 0.057, 0.128, 0.0001, 0.081, 0.346, 0.343, 0.013, 0.053, 0.416, 0.005, 0.029, 0.062, 0.002, 0.0001, 0.002, 0.0001, 0.015, 0.0001, 5.706, 1.515, 0.218, 1.433, 4.537, 0.262, 2.208, 0.762, 2.069, 0.092, 2.86, 1.7, 1.242, 5.365, 3.328, 0.599, 0.022, 1.81, 1.981, 3.619, 0.984, 0.189, 0.434, 0.086, 1.565, 0.929, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.004, 0.029, 0.0001, 0.015, 0.0001, 0.0001, 0.007, 0.004, 0.002, 0.007, 0.016, 0.004, 0.004, 0.022, 0.015, 0.018, 0.0001, 0.004, 0.007, 0.022, 0.0001, 0.009, 0.004, 0.0001, 0.005, 0.0001, 0.016, 0.002, 0.0001, 0.002, 0.059, 0.007, 0.009, 1.785, 0.013, 1.12, 0.007, 0.002, 0.044, 0.027, 0.112, 1.609, 0.647, 0.002, 0.026, 3.804, 0.577, 0.007, 0.004, 0.0001, 0.022, 0.564, 0.0001, 1.689, 0.002, 0.005, 0.046, 0.002, 0.403, 0.176, 0.0001, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 12.759, 0.005, 0.07, 0.0001, 0.007, 0.0001, 0.007, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.024, 0.011, 0.007, 0.0001, 0.0001, 0.015, 0.004, 0.002, 0.0001, 0.002, 0.002, 0.004, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sh":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.387, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.061, 0.002, 0.275, 0.0001, 0.001, 0.015, 0.001, 0.014, 0.187, 0.187, 0.0001, 0.002, 0.95, 0.147, 1.223, 0.02, 0.376, 0.495, 0.321, 0.161, 0.177, 0.142, 0.127, 0.123, 0.133, 0.217, 0.047, 0.016, 0.003, 0.003, 0.003, 0.002, 0.0001, 0.169, 0.152, 0.152, 0.124, 0.074, 0.068, 0.115, 0.09, 0.132, 0.082, 0.135, 0.104, 0.207, 0.208, 0.116, 0.297, 0.005, 0.117, 0.252, 0.137, 0.082, 0.102, 0.023, 0.008, 0.009, 0.058, 0.012, 0.0001, 0.012, 0.0001, 0.003, 0.0001, 8.404, 0.811, 0.782, 2.282, 6.596, 0.226, 1.221, 0.511, 7.186, 3.593, 2.512, 2.698, 2.143, 5.119, 6.434, 1.798, 0.011, 3.759, 3.618, 2.964, 3.066, 2.297, 0.032, 0.02, 0.068, 1.245, 0.001, 0.003, 0.001, 0.0001, 0.0001, 0.106, 0.072, 0.05, 0.059, 0.003, 0.006, 0.013, 0.266, 0.014, 0.001, 0.001, 0.001, 0.012, 0.537, 0.001, 0.001, 0.005, 0.13, 0.005, 0.01, 0.006, 0.001, 0.001, 0.002, 0.039, 0.019, 0.011, 0.012, 0.016, 0.006, 0.015, 0.006, 0.037, 0.532, 0.005, 0.005, 0.003, 0.001, 0.001, 0.002, 0.003, 0.013, 0.001, 0.004, 0.001, 0.013, 0.001, 0.001, 0.155, 0.023, 0.063, 0.029, 0.039, 0.108, 0.009, 0.019, 0.119, 0.002, 0.045, 0.036, 0.04, 0.092, 0.525, 0.041, 0.0001, 0.0001, 0.038, 0.074, 0.943, 0.945, 0.0001, 0.0001, 0.009, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.013, 0.006, 0.916, 0.332, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.045, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "si":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.314, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.928, 0.001, 0.061, 0.001, 0.001, 0.005, 0.0001, 0.009, 0.059, 0.059, 0.0001, 0.001, 0.185, 0.034, 0.404, 0.009, 0.086, 0.094, 0.056, 0.028, 0.026, 0.03, 0.026, 0.025, 0.029, 0.049, 0.012, 0.004, 0.002, 0.002, 0.002, 0.002, 0.0001, 0.015, 0.008, 0.015, 0.008, 0.007, 0.006, 0.005, 0.007, 0.015, 0.002, 0.003, 0.006, 0.01, 0.006, 0.006, 0.01, 0.001, 0.006, 0.015, 0.013, 0.004, 0.004, 0.005, 0.001, 0.001, 0.001, 0.005, 0.0001, 0.005, 0.0001, 0.003, 0.0001, 0.173, 0.028, 0.067, 0.071, 0.23, 0.039, 0.038, 0.081, 0.157, 0.003, 0.016, 0.086, 0.051, 0.142, 0.142, 0.042, 0.002, 0.132, 0.124, 0.159, 0.054, 0.019, 0.027, 0.007, 0.032, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 2.252, 0.201, 0.275, 1.149, 0.707, 0.497, 0.106, 0.13, 0.003, 0.08, 2.719, 0.062, 0.013, 0.472, 0.0001, 1.466, 0.446, 0.166, 2.231, 0.489, 1.129, 0.007, 0.144, 0.0001, 0.039, 0.612, 2.039, 0.034, 0.759, 0.19, 0.015, 0.026, 0.094, 0.008, 0.192, 0.001, 0.005, 0.01, 0.001, 0.537, 0.012, 0.162, 0.001, 0.275, 0.003, 1.26, 0.067, 0.843, 0.165, 1.921, 0.001, 0.089, 0.809, 0.008, 15.577, 14.437, 1.305, 0.017, 1.681, 1.481, 0.0001, 0.796, 0.0001, 0.001, 0.0001, 0.0001, 0.014, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 29.588, 0.0001, 0.504, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sm":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.213, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 19.463, 0.008, 0.168, 0.0001, 0.003, 0.014, 0.002, 0.885, 0.148, 0.148, 0.0001, 0.001, 1.0, 0.173, 0.914, 0.009, 0.254, 0.312, 0.179, 0.14, 0.095, 0.115, 0.095, 0.086, 0.112, 0.168, 0.033, 0.027, 0.006, 0.002, 0.006, 0.005, 0.0001, 0.462, 0.087, 0.119, 0.039, 0.23, 0.233, 0.074, 0.06, 0.345, 0.031, 0.147, 0.149, 0.348, 0.135, 0.431, 0.236, 0.003, 0.115, 0.459, 0.28, 0.072, 0.088, 0.128, 0.02, 0.007, 0.009, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 15.436, 0.147, 0.251, 0.268, 7.552, 1.798, 1.939, 0.261, 7.65, 0.014, 0.507, 6.117, 2.84, 3.141, 6.14, 1.094, 0.011, 1.189, 2.656, 4.384, 4.707, 0.608, 0.084, 0.018, 0.145, 0.038, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.05, 0.151, 0.0001, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.003, 0.002, 0.017, 0.002, 0.001, 0.003, 0.0001, 0.0001, 0.02, 0.001, 0.001, 0.0001, 0.001, 0.039, 0.002, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.005, 0.004, 0.0001, 0.011, 0.001, 0.001, 0.001, 0.001, 0.001, 0.006, 0.001, 0.028, 0.001, 0.004, 0.001, 0.001, 0.003, 0.003, 0.001, 0.003, 0.002, 0.002, 0.0001, 0.001, 0.002, 0.001, 0.002, 0.086, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.005, 0.033, 0.176, 0.042, 0.0001, 0.0001, 0.0001, 0.001, 0.085, 0.001, 0.0001, 0.0001, 0.007, 0.0001, 0.006, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.046, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.427, 0.001, 0.194, 0.0001, 0.001, 0.007, 0.003, 0.024, 0.281, 0.281, 0.0001, 0.005, 0.597, 0.124, 0.956, 0.038, 0.114, 0.113, 0.073, 0.04, 0.036, 0.036, 0.026, 0.025, 0.034, 0.053, 0.08, 0.124, 0.002, 0.009, 0.002, 0.006, 0.0001, 0.169, 0.097, 0.234, 0.083, 0.107, 0.043, 0.1, 0.097, 0.095, 0.037, 0.196, 0.037, 0.454, 0.178, 0.024, 0.119, 0.003, 0.094, 0.231, 0.097, 0.036, 0.089, 0.031, 0.003, 0.009, 0.113, 0.039, 0.0001, 0.038, 0.0001, 0.002, 0.0001, 12.237, 1.335, 1.505, 2.374, 5.54, 0.412, 1.524, 3.199, 8.126, 0.115, 3.86, 0.667, 3.205, 6.578, 4.667, 1.202, 0.019, 4.537, 2.41, 2.721, 5.562, 2.325, 2.211, 0.043, 1.41, 2.325, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.012, 0.003, 0.001, 0.003, 0.001, 0.001, 0.001, 0.0001, 0.017, 0.001, 0.004, 0.001, 0.004, 0.0001, 0.001, 0.001, 0.01, 0.005, 0.003, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.016, 0.001, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.009, 0.004, 0.001, 0.001, 0.001, 0.001, 0.003, 0.003, 0.003, 0.004, 0.008, 0.001, 0.0001, 0.002, 0.0001, 0.001, 0.004, 0.002, 0.002, 0.002, 0.001, 0.001, 0.002, 0.002, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.011, 0.016, 0.003, 0.002, 0.0001, 0.0001, 0.0001, 0.037, 0.008, 0.027, 0.001, 0.001, 0.002, 0.001, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "so":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.235, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.454, 0.003, 0.106, 0.0001, 0.001, 0.01, 0.006, 0.044, 0.175, 0.179, 0.001, 0.006, 0.698, 0.181, 0.663, 0.023, 0.173, 0.237, 0.118, 0.074, 0.068, 0.076, 0.069, 0.062, 0.061, 0.116, 0.103, 0.039, 0.006, 0.095, 0.008, 0.006, 0.001, 0.277, 0.176, 0.197, 0.21, 0.058, 0.067, 0.135, 0.123, 0.156, 0.069, 0.122, 0.08, 0.279, 0.092, 0.046, 0.025, 0.078, 0.077, 0.341, 0.096, 0.053, 0.009, 0.145, 0.085, 0.037, 0.009, 0.058, 0.001, 0.058, 0.0001, 0.009, 0.001, 20.28, 1.752, 0.781, 4.408, 3.807, 0.467, 1.801, 2.804, 6.156, 0.344, 2.692, 2.981, 1.937, 3.517, 5.007, 0.065, 0.666, 2.59, 2.645, 1.488, 3.47, 0.033, 1.517, 1.277, 3.257, 0.024, 0.006, 0.007, 0.006, 0.0001, 0.0001, 0.044, 0.021, 0.016, 0.015, 0.092, 0.046, 0.041, 0.026, 0.037, 0.007, 0.048, 0.005, 0.002, 0.004, 0.027, 0.011, 0.01, 0.009, 0.012, 0.004, 0.002, 0.001, 0.001, 0.002, 0.003, 0.016, 0.0001, 0.0001, 0.009, 0.011, 0.002, 0.005, 0.026, 0.005, 0.004, 0.02, 0.008, 0.009, 0.004, 0.102, 0.029, 0.015, 0.023, 0.008, 0.009, 0.018, 0.009, 0.021, 0.011, 0.034, 0.006, 0.02, 0.009, 0.011, 0.006, 0.006, 0.005, 0.024, 0.019, 0.018, 0.004, 0.003, 0.001, 0.004, 0.0001, 0.0001, 0.03, 0.015, 0.007, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.36, 0.404, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.045, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.034, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sq":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.871, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.83, 0.005, 0.212, 0.0001, 0.001, 0.008, 0.002, 0.025, 0.142, 0.143, 0.001, 0.002, 0.876, 0.197, 0.817, 0.021, 0.247, 0.322, 0.187, 0.096, 0.094, 0.095, 0.084, 0.083, 0.096, 0.185, 0.067, 0.019, 0.003, 0.004, 0.003, 0.004, 0.0001, 0.232, 0.164, 0.084, 0.121, 0.088, 0.104, 0.113, 0.084, 0.118, 0.051, 0.274, 0.113, 0.216, 0.178, 0.042, 0.229, 0.027, 0.103, 0.291, 0.126, 0.044, 0.092, 0.017, 0.024, 0.009, 0.037, 0.024, 0.0001, 0.024, 0.0001, 0.005, 0.001, 5.42, 0.732, 0.432, 2.174, 7.144, 0.635, 1.01, 2.972, 6.09, 2.066, 2.05, 2.101, 2.386, 4.875, 2.895, 1.724, 0.557, 5.177, 3.826, 5.956, 2.462, 1.012, 0.037, 0.057, 0.423, 0.487, 0.0001, 0.007, 0.0001, 0.0001, 0.0001, 0.107, 0.006, 0.004, 0.005, 0.003, 0.002, 0.002, 0.017, 0.002, 0.002, 0.001, 0.01, 0.001, 0.002, 0.002, 0.001, 0.001, 0.008, 0.001, 0.019, 0.002, 0.001, 0.001, 0.001, 0.003, 0.015, 0.001, 0.001, 0.032, 0.031, 0.002, 0.003, 0.048, 0.005, 0.005, 0.002, 0.005, 0.002, 0.002, 0.098, 0.005, 0.011, 0.001, 5.762, 0.002, 0.004, 0.002, 0.002, 0.006, 0.006, 0.012, 0.004, 0.005, 0.003, 0.003, 0.002, 0.003, 0.003, 0.003, 0.004, 0.006, 0.003, 0.003, 0.003, 0.0001, 0.0001, 0.063, 5.926, 0.008, 0.006, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.023, 0.009, 0.015, 0.012, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.007, 0.008, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.106, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "srn": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.777, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 18.537, 0.004, 0.236, 0.0001, 0.0001, 0.009, 0.0001, 0.081, 0.222, 0.175, 0.0001, 0.0001, 0.673, 0.268, 1.397, 0.005, 0.412, 0.368, 0.15, 0.085, 0.102, 0.103, 0.102, 0.071, 0.07, 0.14, 0.041, 0.016, 0.015, 0.002, 0.015, 0.0001, 0.0001, 0.384, 0.184, 0.068, 0.478, 0.061, 0.057, 0.098, 0.039, 0.172, 0.08, 0.05, 0.052, 0.288, 0.1, 0.075, 0.116, 0.004, 0.117, 0.271, 0.146, 0.008, 0.023, 0.047, 0.004, 0.014, 0.007, 0.005, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 8.95, 2.176, 0.221, 2.431, 7.818, 1.651, 1.874, 0.226, 8.782, 0.064, 2.479, 1.698, 2.095, 8.318, 4.117, 1.376, 0.003, 4.52, 3.577, 2.919, 3.347, 0.156, 1.329, 0.018, 1.038, 0.054, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.042, 0.007, 0.007, 0.002, 0.003, 0.002, 0.003, 0.006, 0.003, 0.001, 0.002, 0.001, 0.006, 0.003, 0.002, 0.005, 0.004, 0.002, 0.001, 0.035, 0.002, 0.002, 0.002, 0.006, 0.002, 0.002, 0.002, 0.002, 0.002, 0.007, 0.002, 0.002, 0.024, 0.012, 0.002, 0.005, 0.004, 0.007, 0.002, 0.002, 0.012, 0.012, 0.006, 0.009, 0.002, 0.021, 0.005, 0.003, 0.003, 0.003, 0.034, 0.007, 0.002, 0.002, 0.002, 0.0001, 0.005, 0.007, 0.019, 0.009, 0.005, 0.003, 0.004, 0.012, 0.0001, 0.0001, 0.029, 0.098, 0.021, 0.025, 0.002, 0.002, 0.002, 0.005, 0.001, 0.003, 0.0001, 0.0001, 0.01, 0.004, 0.009, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.053, 0.0001, 0.0001, 0.016, 0.016, 0.0001, 0.01, 0.0001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ss":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.873, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.454, 0.015, 0.301, 0.001, 0.003, 0.01, 0.003, 0.035, 0.203, 0.202, 0.001, 0.0001, 0.685, 0.328, 0.962, 0.019, 0.22, 0.221, 0.137, 0.048, 0.066, 0.07, 0.054, 0.061, 0.082, 0.144, 0.105, 0.052, 0.007, 0.003, 0.008, 0.003, 0.0001, 0.231, 0.18, 0.097, 0.094, 0.111, 0.055, 0.072, 0.058, 0.259, 0.082, 0.196, 0.342, 0.348, 0.356, 0.028, 0.088, 0.003, 0.097, 0.319, 0.164, 0.113, 0.024, 0.061, 0.025, 0.043, 0.044, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 10.793, 2.656, 0.706, 1.31, 8.505, 1.004, 2.081, 2.919, 7.091, 0.258, 4.271, 5.701, 2.568, 6.606, 3.595, 0.825, 0.028, 0.782, 3.437, 3.569, 4.546, 0.696, 2.323, 0.017, 1.567, 0.734, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.037, 0.016, 0.007, 0.01, 0.014, 0.008, 0.007, 0.004, 0.009, 0.007, 0.013, 0.004, 0.003, 0.014, 0.015, 0.004, 0.003, 0.006, 0.003, 0.008, 0.006, 0.002, 0.007, 0.004, 0.002, 0.004, 0.007, 0.002, 0.01, 0.003, 0.007, 0.003, 0.09, 0.039, 0.013, 0.006, 0.01, 0.005, 0.005, 0.023, 0.007, 0.024, 0.007, 0.009, 0.005, 0.109, 0.006, 0.007, 0.018, 0.014, 0.009, 0.035, 0.024, 0.01, 0.007, 0.005, 0.015, 0.006, 0.031, 0.01, 0.005, 0.01, 0.008, 0.005, 0.0001, 0.0001, 0.085, 0.273, 0.013, 0.008, 0.0001, 0.0001, 0.0001, 0.005, 0.001, 0.002, 0.002, 0.0001, 0.003, 0.002, 0.061, 0.022, 0.001, 0.0001, 0.0001, 0.003, 0.002, 0.003, 0.059, 0.053, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.042, 0.021, 0.034, 0.0001, 0.001, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.003, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "st":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.411, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.42, 0.002, 0.079, 0.0001, 0.001, 0.016, 0.003, 0.083, 0.165, 0.167, 0.001, 0.001, 0.789, 0.143, 0.973, 0.021, 0.355, 0.325, 0.221, 0.104, 0.116, 0.113, 0.108, 0.098, 0.108, 0.15, 0.061, 0.016, 0.007, 0.005, 0.006, 0.001, 0.0001, 0.408, 0.587, 0.149, 0.148, 0.115, 0.088, 0.067, 0.172, 0.071, 0.055, 0.339, 0.212, 0.509, 0.175, 0.046, 0.141, 0.01, 0.115, 0.317, 0.165, 0.126, 0.071, 0.047, 0.0001, 0.019, 0.026, 0.011, 0.0001, 0.01, 0.0001, 0.005, 0.0001, 12.26, 2.144, 0.403, 1.165, 9.234, 0.827, 1.837, 3.801, 3.704, 0.349, 2.878, 4.66, 2.188, 4.177, 7.024, 1.54, 0.085, 2.344, 4.067, 4.22, 1.114, 0.282, 1.372, 0.049, 0.996, 0.173, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.013, 0.01, 0.009, 0.0001, 0.005, 0.007, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.013, 0.003, 0.001, 0.009, 0.001, 0.001, 0.004, 0.005, 0.0001, 0.0001, 0.002, 0.001, 0.01, 0.006, 0.004, 0.004, 0.003, 0.0001, 0.0001, 0.049, 0.052, 0.003, 0.003, 0.006, 0.002, 0.001, 0.006, 0.002, 0.022, 0.037, 0.001, 0.003, 0.01, 0.0001, 0.001, 0.004, 0.002, 0.001, 0.03, 0.056, 0.001, 0.001, 0.001, 0.004, 0.001, 0.002, 0.007, 0.001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.046, 0.167, 0.019, 0.086, 0.0001, 0.003, 0.001, 0.01, 0.001, 0.003, 0.0001, 0.001, 0.0001, 0.0001, 0.01, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.008, 0.013, 0.0001, 0.001, 0.004, 0.002, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "stq": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.516, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.229, 0.003, 0.416, 0.007, 0.001, 0.015, 0.001, 0.047, 0.208, 0.207, 0.008, 0.003, 0.814, 0.425, 1.065, 0.021, 0.376, 0.623, 0.234, 0.148, 0.183, 0.183, 0.241, 0.167, 0.231, 0.214, 0.089, 0.019, 0.072, 0.007, 0.069, 0.006, 0.0001, 0.293, 0.408, 0.089, 0.454, 0.155, 0.334, 0.214, 0.273, 0.205, 0.248, 0.241, 0.264, 0.372, 0.199, 0.14, 0.214, 0.005, 0.245, 0.798, 0.226, 0.158, 0.049, 0.246, 0.003, 0.006, 0.016, 0.02, 0.0001, 0.02, 0.0001, 0.001, 0.0001, 3.929, 0.935, 0.799, 3.858, 10.176, 1.298, 1.131, 1.308, 4.615, 0.883, 2.156, 2.674, 1.358, 6.685, 4.841, 0.816, 0.012, 4.246, 3.53, 4.621, 4.666, 0.159, 1.055, 0.042, 0.141, 0.124, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.049, 0.004, 0.003, 0.001, 0.07, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.001, 0.001, 0.004, 0.002, 0.001, 0.003, 0.001, 0.009, 0.014, 0.001, 0.002, 0.008, 0.004, 0.005, 0.004, 0.021, 0.009, 0.015, 0.001, 2.394, 0.001, 0.001, 0.002, 0.004, 0.014, 0.003, 0.002, 0.001, 0.007, 0.002, 0.002, 0.004, 0.002, 0.026, 0.006, 0.003, 0.001, 0.134, 0.001, 0.003, 0.002, 0.004, 0.004, 0.245, 0.003, 0.002, 0.003, 0.0001, 0.0001, 0.038, 2.918, 0.006, 0.011, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.002, 0.001, 0.0001, 0.013, 0.006, 0.008, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.048, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "su":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.293, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.195, 0.001, 0.272, 0.0001, 0.001, 0.006, 0.001, 0.02, 0.129, 0.129, 0.0001, 0.002, 1.05, 0.168, 1.046, 0.037, 0.48, 0.412, 0.411, 0.202, 0.173, 0.175, 0.161, 0.145, 0.144, 0.197, 0.036, 0.015, 0.003, 0.003, 0.003, 0.001, 0.0001, 0.394, 0.22, 0.151, 0.149, 0.042, 0.047, 0.094, 0.073, 0.227, 0.16, 0.402, 0.071, 0.278, 0.12, 0.097, 0.305, 0.014, 0.09, 0.368, 0.175, 0.05, 0.031, 0.057, 0.016, 0.027, 0.009, 0.008, 0.0001, 0.008, 0.0001, 0.005, 0.0001, 13.373, 1.612, 0.819, 2.725, 4.093, 0.314, 2.685, 1.583, 5.788, 0.997, 2.729, 2.341, 2.09, 7.706, 2.801, 1.889, 0.016, 3.889, 3.272, 4.14, 4.781, 0.134, 0.635, 0.029, 0.708, 0.032, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.025, 0.005, 0.003, 0.004, 0.006, 0.004, 0.004, 0.002, 0.004, 0.073, 0.003, 0.001, 0.001, 0.004, 0.007, 0.003, 0.003, 0.002, 0.003, 0.007, 0.008, 0.001, 0.001, 0.001, 0.001, 0.004, 0.001, 0.001, 0.004, 0.004, 0.001, 0.001, 0.047, 0.002, 0.001, 0.002, 0.004, 0.002, 0.002, 0.006, 0.003, 2.276, 0.003, 0.002, 0.001, 0.002, 0.007, 0.002, 0.004, 0.005, 0.002, 0.002, 0.001, 0.001, 0.002, 0.001, 0.002, 0.003, 0.002, 0.001, 0.002, 0.033, 0.001, 0.033, 0.0001, 0.0001, 0.051, 2.355, 0.003, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.02, 0.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.007, 0.025, 0.004, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.032, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "sw":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.454, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.93, 0.002, 0.217, 0.002, 0.001, 0.01, 0.003, 0.027, 0.171, 0.171, 0.001, 0.001, 0.703, 0.109, 0.942, 0.015, 0.41, 0.383, 0.266, 0.126, 0.108, 0.126, 0.108, 0.107, 0.119, 0.201, 0.062, 0.024, 0.003, 0.004, 0.003, 0.003, 0.0001, 0.226, 0.167, 0.122, 0.086, 0.058, 0.057, 0.065, 0.116, 0.13, 0.09, 0.638, 0.08, 0.504, 0.137, 0.044, 0.113, 0.006, 0.074, 0.173, 0.147, 0.165, 0.059, 0.218, 0.013, 0.04, 0.023, 0.04, 0.0001, 0.04, 0.001, 0.001, 0.001, 16.478, 1.326, 0.611, 1.343, 3.374, 0.678, 1.131, 2.383, 9.629, 0.827, 4.598, 2.609, 3.253, 5.284, 3.187, 0.805, 0.008, 1.616, 2.094, 2.468, 4.443, 0.427, 3.161, 0.026, 2.095, 1.273, 0.001, 0.006, 0.001, 0.0001, 0.0001, 0.04, 0.005, 0.004, 0.002, 0.004, 0.002, 0.002, 0.002, 0.002, 0.002, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.013, 0.002, 0.001, 0.001, 0.001, 0.002, 0.006, 0.001, 0.001, 0.009, 0.008, 0.001, 0.004, 0.009, 0.003, 0.002, 0.002, 0.003, 0.001, 0.001, 0.005, 0.003, 0.009, 0.001, 0.002, 0.001, 0.002, 0.001, 0.002, 0.005, 0.009, 0.009, 0.004, 0.002, 0.003, 0.004, 0.001, 0.004, 0.003, 0.003, 0.003, 0.006, 0.003, 0.002, 0.003, 0.0001, 0.0001, 0.018, 0.029, 0.009, 0.005, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.014, 0.007, 0.011, 0.004, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.005, 0.012, 0.01, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.038, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "szl": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.884, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.63, 0.002, 0.452, 0.0001, 0.0001, 0.012, 0.001, 0.026, 0.296, 0.296, 0.001, 0.001, 1.094, 0.318, 1.181, 0.015, 0.332, 0.469, 0.289, 0.138, 0.131, 0.151, 0.118, 0.131, 0.157, 0.273, 0.087, 0.014, 0.006, 0.003, 0.006, 0.0001, 0.0001, 0.207, 0.209, 0.155, 0.118, 0.048, 0.111, 0.139, 0.08, 0.122, 0.125, 0.213, 0.123, 0.287, 0.122, 0.062, 0.309, 0.005, 0.156, 0.329, 0.126, 0.154, 0.05, 0.233, 0.034, 0.017, 0.083, 0.004, 0.0001, 0.004, 0.0001, 0.006, 0.001, 5.741, 0.894, 2.016, 2.128, 5.35, 0.327, 1.279, 0.968, 3.438, 2.841, 2.633, 2.099, 2.293, 3.364, 5.857, 1.423, 0.012, 3.389, 2.85, 2.58, 2.277, 0.102, 3.144, 0.017, 3.623, 2.205, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.191, 0.035, 0.624, 0.044, 0.945, 0.014, 0.009, 0.333, 0.008, 0.003, 0.006, 0.005, 0.012, 0.221, 0.005, 0.196, 0.006, 0.005, 0.003, 0.168, 0.01, 0.003, 0.005, 0.005, 0.005, 0.109, 0.059, 0.562, 0.005, 0.005, 0.004, 0.006, 0.062, 0.111, 0.006, 0.016, 0.01, 0.004, 0.004, 0.012, 0.011, 0.03, 0.005, 0.012, 0.003, 0.012, 0.008, 1.67, 0.032, 0.015, 0.058, 0.035, 0.048, 0.018, 0.012, 0.004, 0.02, 0.013, 0.335, 0.026, 0.282, 0.022, 0.098, 0.006, 0.0001, 0.0001, 0.109, 0.208, 0.455, 5.073, 0.0001, 0.001, 0.0001, 0.008, 0.003, 0.003, 0.004, 0.0001, 0.015, 0.008, 0.161, 0.06, 0.003, 0.002, 0.0001, 0.003, 0.001, 0.009, 0.025, 0.019, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.011, 0.01, 0.176, 0.006, 0.001, 0.005, 0.003, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ta":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.357, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.862, 0.001, 0.077, 0.0001, 0.001, 0.006, 0.001, 0.007, 0.055, 0.056, 0.0001, 0.001, 0.234, 0.03, 0.384, 0.005, 0.084, 0.106, 0.063, 0.029, 0.028, 0.034, 0.027, 0.032, 0.031, 0.052, 0.017, 0.006, 0.002, 0.002, 0.002, 0.001, 0.0001, 0.008, 0.004, 0.008, 0.004, 0.004, 0.003, 0.005, 0.004, 0.006, 0.002, 0.003, 0.003, 0.005, 0.004, 0.003, 0.008, 0.0001, 0.004, 0.008, 0.005, 0.002, 0.002, 0.002, 0.001, 0.001, 0.0001, 0.006, 0.0001, 0.006, 0.0001, 0.002, 0.0001, 0.062, 0.006, 0.017, 0.014, 0.042, 0.007, 0.009, 0.018, 0.038, 0.001, 0.006, 0.024, 0.018, 0.035, 0.032, 0.011, 0.001, 0.036, 0.022, 0.032, 0.017, 0.005, 0.004, 0.002, 0.01, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.122, 2.149, 0.144, 0.01, 0.0001, 0.297, 0.436, 0.597, 0.764, 0.136, 0.24, 0.226, 0.005, 5.298, 0.158, 0.027, 0.013, 0.0001, 0.078, 0.014, 0.0001, 2.36, 0.0001, 0.0001, 0.001, 0.171, 0.627, 0.0001, 0.037, 0.002, 0.021, 1.319, 0.014, 0.0001, 0.001, 0.32, 2.012, 0.001, 0.001, 0.0001, 0.539, 0.989, 1.521, 0.0001, 0.0001, 0.001, 23.215, 10.185, 1.322, 0.801, 1.028, 0.757, 0.189, 0.942, 0.0001, 0.015, 0.06, 0.015, 0.0001, 0.0001, 0.0001, 0.0001, 1.18, 2.177, 0.0001, 0.0001, 0.016, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 31.245, 0.0001, 0.013, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tcy": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.391, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.751, 0.001, 0.026, 0.0001, 0.0001, 0.002, 0.0001, 0.028, 0.048, 0.047, 0.0001, 0.001, 0.244, 0.028, 0.533, 0.012, 0.014, 0.02, 0.01, 0.005, 0.005, 0.007, 0.006, 0.004, 0.008, 0.009, 0.009, 0.003, 0.002, 0.003, 0.002, 0.002, 0.0001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.003, 0.0001, 0.001, 0.001, 0.02, 0.002, 0.008, 0.006, 0.018, 0.002, 0.005, 0.006, 0.017, 0.0001, 0.003, 0.009, 0.008, 0.014, 0.015, 0.008, 0.0001, 0.013, 0.012, 0.015, 0.006, 0.002, 0.005, 0.0001, 0.003, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.354, 1.789, 1.221, 0.031, 0.0001, 0.268, 1.686, 0.484, 0.152, 0.21, 0.745, 0.196, 0.087, 4.125, 0.064, 0.014, 0.014, 0.0001, 0.109, 0.011, 0.001, 1.28, 0.033, 0.613, 0.012, 0.007, 0.23, 0.003, 0.404, 0.002, 0.011, 0.433, 0.058, 1.007, 0.002, 0.198, 1.312, 0.064, 1.397, 0.124, 1.439, 0.012, 1.248, 0.035, 0.624, 0.105, 0.769, 0.62, 1.755, 0.0001, 22.872, 9.408, 0.0001, 0.629, 0.164, 0.121, 0.665, 0.124, 0.0001, 0.0001, 0.003, 0.0001, 1.377, 1.63, 0.0001, 0.0001, 0.05, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.955, 0.0001, 0.194, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "te":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.34, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.746, 0.003, 0.051, 0.0001, 0.001, 0.003, 0.002, 0.007, 0.042, 0.043, 0.0001, 0.001, 0.336, 0.028, 0.611, 0.018, 0.129, 0.152, 0.069, 0.038, 0.034, 0.073, 0.03, 0.032, 0.034, 0.047, 0.02, 0.007, 0.002, 0.004, 0.002, 0.002, 0.0001, 0.008, 0.004, 0.006, 0.005, 0.003, 0.003, 0.002, 0.002, 0.006, 0.001, 0.002, 0.003, 0.005, 0.003, 0.002, 0.005, 0.0001, 0.003, 0.008, 0.005, 0.003, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.006, 0.0001, 0.007, 0.0001, 0.005, 0.0001, 0.053, 0.008, 0.019, 0.022, 0.056, 0.009, 0.01, 0.021, 0.046, 0.001, 0.004, 0.022, 0.015, 0.038, 0.038, 0.014, 0.0001, 0.036, 0.036, 0.045, 0.017, 0.006, 0.006, 0.002, 0.007, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.485, 1.801, 1.898, 0.051, 0.0001, 0.236, 0.427, 0.575, 0.238, 0.222, 0.152, 0.685, 0.105, 2.799, 0.055, 0.027, 0.006, 0.0001, 0.047, 0.007, 0.005, 1.329, 0.049, 0.668, 0.014, 0.002, 0.428, 0.004, 0.25, 0.001, 0.004, 0.537, 0.039, 0.598, 0.002, 0.137, 0.864, 0.099, 0.843, 0.149, 1.628, 0.0001, 0.909, 0.085, 0.267, 0.128, 0.942, 0.804, 25.531, 7.165, 1.487, 0.074, 0.0001, 0.877, 0.211, 0.153, 0.855, 0.145, 0.0001, 0.001, 0.0001, 0.0001, 2.169, 2.359, 0.0001, 0.0001, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 30.736, 0.0001, 0.069, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tet": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.506, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.056, 0.014, 0.345, 0.0001, 0.004, 0.018, 0.001, 0.455, 0.383, 0.382, 0.001, 0.004, 1.067, 0.53, 0.968, 0.029, 0.443, 0.39, 0.316, 0.132, 0.112, 0.137, 0.105, 0.106, 0.119, 0.181, 0.186, 0.018, 0.015, 0.005, 0.015, 0.003, 0.0001, 0.338, 0.226, 0.145, 0.169, 0.132, 0.156, 0.098, 0.111, 0.215, 0.061, 0.136, 0.43, 0.301, 0.181, 0.101, 0.266, 0.01, 0.137, 0.345, 0.37, 0.107, 0.065, 0.041, 0.021, 0.008, 0.014, 0.01, 0.0001, 0.01, 0.0001, 0.0001, 0.0001, 11.569, 1.502, 0.408, 2.068, 6.067, 0.587, 0.66, 2.225, 7.509, 0.16, 2.246, 2.814, 2.311, 6.307, 4.401, 1.282, 0.035, 4.022, 4.063, 3.545, 4.826, 0.518, 0.1, 0.064, 0.126, 0.341, 0.0001, 0.009, 0.0001, 0.0001, 0.0001, 0.318, 0.081, 0.003, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.015, 0.001, 0.0001, 0.0001, 0.001, 0.004, 0.275, 0.001, 0.0001, 0.014, 0.013, 0.001, 0.002, 0.021, 0.254, 0.002, 0.025, 0.0001, 0.0001, 0.003, 0.02, 0.002, 0.389, 0.006, 0.001, 0.001, 0.167, 0.001, 0.001, 0.002, 0.048, 0.071, 0.284, 0.01, 0.003, 0.001, 0.0001, 0.001, 0.001, 0.076, 0.003, 0.014, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.1, 1.362, 0.004, 0.006, 0.0001, 0.0001, 0.0001, 0.009, 0.011, 0.0001, 0.0001, 0.0001, 0.007, 0.003, 0.006, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.316, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tg":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.272, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.893, 0.001, 0.026, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.324, 0.326, 0.0001, 0.001, 0.765, 0.105, 0.581, 0.006, 0.139, 0.257, 0.13, 0.073, 0.063, 0.072, 0.065, 0.068, 0.082, 0.185, 0.026, 0.048, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.026, 0.01, 0.018, 0.007, 0.005, 0.01, 0.006, 0.007, 0.018, 0.002, 0.005, 0.008, 0.009, 0.006, 0.004, 0.009, 0.001, 0.007, 0.015, 0.007, 0.003, 0.006, 0.004, 0.006, 0.002, 0.002, 0.004, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.081, 0.01, 0.03, 0.023, 0.086, 0.012, 0.015, 0.021, 0.065, 0.002, 0.009, 0.037, 0.017, 0.055, 0.061, 0.017, 0.001, 0.07, 0.039, 0.054, 0.023, 0.01, 0.007, 0.003, 0.013, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.968, 1.483, 1.764, 1.455, 0.398, 0.384, 0.008, 0.116, 0.704, 0.002, 0.17, 0.01, 0.024, 0.035, 0.045, 0.663, 0.178, 0.263, 0.119, 0.126, 0.303, 0.007, 0.009, 0.022, 0.136, 0.003, 0.143, 0.343, 0.148, 0.063, 0.071, 0.071, 0.134, 0.159, 0.101, 0.347, 0.121, 0.05, 0.002, 0.026, 0.059, 0.003, 0.003, 0.057, 0.003, 0.035, 0.012, 0.164, 5.899, 1.075, 1.071, 1.816, 2.336, 1.339, 0.082, 0.882, 4.885, 0.258, 1.014, 1.438, 1.445, 2.22, 3.885, 0.208, 0.0001, 0.0001, 0.132, 0.006, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.002, 0.001, 30.166, 10.131, 1.965, 0.481, 0.0001, 0.0001, 0.0001, 0.0001, 0.024, 0.016, 0.001, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.209, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ti":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.164, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.862, 0.026, 0.05, 0.0001, 0.0001, 0.012, 0.0001, 0.044, 0.1, 0.1, 0.0001, 0.0001, 0.075, 0.114, 0.14, 0.02, 0.098, 0.121, 0.073, 0.033, 0.026, 0.04, 0.027, 0.03, 0.029, 0.042, 0.024, 0.004, 0.001, 0.013, 0.001, 0.007, 0.0001, 0.018, 0.013, 0.015, 0.007, 0.006, 0.007, 0.011, 0.013, 0.022, 0.004, 0.004, 0.024, 0.018, 0.012, 0.005, 0.015, 0.004, 0.01, 0.013, 0.022, 0.007, 0.009, 0.006, 0.002, 0.004, 0.002, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.329, 0.063, 0.099, 0.16, 0.451, 0.14, 0.111, 0.211, 0.297, 0.027, 0.053, 0.155, 0.097, 0.283, 0.275, 0.071, 0.007, 0.228, 0.261, 0.255, 0.122, 0.059, 0.08, 0.007, 0.069, 0.014, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.358, 0.069, 0.074, 0.236, 0.007, 0.331, 0.023, 0.001, 9.303, 5.576, 6.47, 5.805, 1.549, 3.066, 0.251, 0.003, 0.505, 0.172, 0.135, 1.034, 0.015, 2.293, 0.054, 0.001, 0.75, 0.233, 0.32, 0.51, 0.12, 1.725, 0.08, 0.002, 0.83, 0.546, 0.753, 1.425, 0.111, 2.053, 0.138, 0.011, 0.764, 0.373, 0.244, 0.731, 0.034, 1.854, 0.258, 0.004, 1.053, 0.166, 0.551, 0.69, 0.031, 2.007, 0.179, 0.005, 0.189, 0.048, 0.045, 0.156, 0.011, 0.447, 0.067, 0.002, 0.0001, 0.0001, 0.386, 0.04, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.027, 0.012, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.008, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 27.967, 0.209, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tk":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.842, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.847, 0.005, 0.052, 0.0001, 0.001, 0.008, 0.001, 0.006, 0.121, 0.125, 0.004, 0.002, 0.691, 0.455, 1.024, 0.011, 0.191, 0.306, 0.153, 0.096, 0.091, 0.095, 0.077, 0.079, 0.095, 0.155, 0.055, 0.012, 0.028, 0.003, 0.028, 0.005, 0.0001, 0.227, 0.204, 0.012, 0.086, 0.083, 0.04, 0.177, 0.112, 0.174, 0.027, 0.109, 0.037, 0.173, 0.054, 0.141, 0.071, 0.001, 0.074, 0.173, 0.153, 0.029, 0.028, 0.04, 0.045, 0.029, 0.016, 0.01, 0.0001, 0.01, 0.001, 0.003, 0.0001, 8.711, 1.574, 0.069, 3.499, 5.666, 0.119, 2.22, 0.895, 5.266, 0.476, 2.165, 5.087, 2.1, 4.83, 1.754, 1.161, 0.002, 5.326, 1.953, 2.216, 1.612, 0.014, 0.863, 0.003, 4.905, 0.889, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.208, 0.022, 0.019, 0.011, 0.017, 0.007, 0.003, 0.027, 1.247, 0.001, 0.0001, 0.008, 0.005, 0.003, 0.002, 0.006, 0.003, 0.005, 0.002, 0.04, 0.02, 0.001, 0.017, 0.002, 0.001, 0.002, 0.001, 0.001, 0.068, 0.139, 0.083, 1.114, 0.015, 0.004, 0.009, 0.002, 0.694, 0.003, 0.003, 0.67, 0.001, 0.002, 0.0001, 0.027, 0.0001, 0.192, 0.001, 0.002, 0.056, 0.114, 0.02, 0.061, 0.013, 0.043, 0.813, 0.006, 0.038, 0.007, 0.016, 0.096, 0.984, 2.385, 0.053, 0.019, 0.0001, 0.0001, 0.268, 5.753, 0.012, 2.464, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.324, 0.111, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.054, 0.182, 0.0001, 0.005, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tl":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.527, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.015, 0.006, 0.416, 0.001, 0.001, 0.006, 0.002, 0.043, 0.2, 0.202, 0.001, 0.002, 0.702, 0.264, 0.789, 0.017, 0.219, 0.272, 0.17, 0.08, 0.075, 0.082, 0.072, 0.075, 0.087, 0.155, 0.061, 0.022, 0.066, 0.004, 0.066, 0.002, 0.0001, 0.555, 0.199, 0.186, 0.134, 0.118, 0.059, 0.112, 0.181, 0.214, 0.066, 0.204, 0.127, 0.268, 0.176, 0.063, 0.292, 0.011, 0.11, 0.398, 0.188, 0.06, 0.045, 0.055, 0.008, 0.035, 0.014, 0.016, 0.0001, 0.015, 0.001, 0.003, 0.0001, 16.44, 1.457, 0.382, 1.246, 2.379, 0.123, 6.741, 1.192, 6.121, 0.033, 2.118, 3.173, 2.569, 9.845, 3.868, 2.142, 0.019, 2.313, 4.125, 3.402, 2.226, 0.121, 0.559, 0.032, 2.131, 0.078, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.038, 0.008, 0.005, 0.004, 0.004, 0.003, 0.002, 0.002, 0.004, 0.002, 0.002, 0.002, 0.003, 0.007, 0.003, 0.002, 0.004, 0.004, 0.002, 0.014, 0.006, 0.001, 0.002, 0.001, 0.002, 0.008, 0.001, 0.002, 0.013, 0.007, 0.002, 0.002, 0.028, 0.01, 0.003, 0.002, 0.004, 0.002, 0.002, 0.004, 0.003, 0.01, 0.002, 0.004, 0.002, 0.008, 0.002, 0.002, 0.005, 0.01, 0.003, 0.007, 0.003, 0.003, 0.002, 0.002, 0.006, 0.003, 0.004, 0.003, 0.005, 0.003, 0.003, 0.003, 0.0001, 0.0001, 0.029, 0.045, 0.007, 0.011, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.012, 0.006, 0.01, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.008, 0.007, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 0.012, 0.037, 0.005, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.004, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tn":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.716, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 17.981, 0.003, 0.08, 0.013, 0.001, 0.009, 0.002, 0.01, 0.075, 0.075, 0.0001, 0.0001, 0.66, 0.106, 0.757, 0.034, 0.2, 0.226, 0.113, 0.036, 0.039, 0.039, 0.04, 0.035, 0.043, 0.09, 0.021, 0.015, 0.01, 0.005, 0.011, 0.004, 0.0001, 0.148, 0.357, 0.071, 0.097, 0.07, 0.054, 0.125, 0.028, 0.051, 0.019, 0.166, 0.104, 0.374, 0.087, 0.085, 0.102, 0.001, 0.088, 0.173, 0.113, 0.019, 0.017, 0.023, 0.006, 0.007, 0.021, 0.023, 0.0001, 0.022, 0.0001, 0.004, 0.0001, 12.488, 2.445, 0.191, 1.643, 9.389, 0.795, 4.171, 1.899, 3.702, 0.312, 2.67, 5.097, 2.631, 4.499, 8.158, 1.075, 0.008, 1.917, 4.118, 4.684, 0.837, 0.048, 2.161, 0.014, 0.955, 0.029, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.0001, 0.0001, 0.003, 0.003, 0.0001, 0.0001, 0.034, 0.011, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.036, 0.008, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "to":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.293, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.821, 0.001, 0.44, 0.0001, 0.0001, 0.001, 0.005, 0.111, 0.238, 0.237, 0.002, 0.0001, 0.847, 0.076, 1.066, 0.045, 0.084, 0.141, 0.063, 0.039, 0.037, 0.032, 0.036, 0.05, 0.065, 0.067, 0.09, 0.023, 0.003, 0.011, 0.005, 0.027, 0.0001, 0.126, 0.034, 0.039, 0.011, 0.049, 0.193, 0.01, 0.178, 0.123, 0.01, 0.599, 0.145, 0.204, 0.188, 0.245, 0.136, 0.001, 0.012, 0.185, 0.547, 0.059, 0.124, 0.026, 0.001, 0.005, 0.001, 0.004, 0.0001, 0.005, 0.0001, 0.002, 0.001, 10.579, 0.223, 0.423, 0.627, 6.707, 1.724, 1.525, 3.199, 6.545, 0.014, 3.573, 2.547, 1.814, 3.859, 6.712, 1.277, 0.01, 0.909, 1.504, 3.555, 4.441, 0.529, 0.312, 0.02, 0.255, 0.009, 0.0001, 0.0001, 0.0001, 0.004, 0.0001, 0.028, 0.432, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.082, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.183, 0.003, 0.0001, 0.0001, 0.001, 0.011, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.057, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.002, 0.001, 0.078, 0.0001, 0.015, 0.0001, 0.0001, 0.013, 0.0001, 0.001, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 4.517, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.022, 0.094, 0.659, 0.119, 0.0001, 0.0001, 0.0001, 0.0001, 4.513, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.024, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tpi": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.506, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.422, 0.004, 0.225, 0.0001, 0.0001, 0.006, 0.0001, 0.033, 0.226, 0.227, 0.001, 0.0001, 0.976, 0.07, 1.357, 0.011, 0.339, 0.409, 0.202, 0.102, 0.113, 0.106, 0.09, 0.101, 0.134, 0.258, 0.112, 0.01, 0.016, 0.001, 0.016, 0.001, 0.0001, 0.28, 0.281, 0.358, 0.108, 0.184, 0.096, 0.132, 0.102, 0.251, 0.103, 0.247, 0.515, 0.27, 0.273, 0.17, 0.405, 0.016, 0.129, 0.696, 0.311, 0.02, 0.133, 0.076, 0.006, 0.097, 0.011, 0.006, 0.0001, 0.006, 0.0001, 0.003, 0.0001, 9.267, 1.534, 0.295, 1.028, 5.418, 0.186, 3.091, 0.44, 8.286, 0.1, 1.968, 5.697, 3.075, 7.815, 5.428, 2.623, 0.013, 2.618, 3.22, 3.51, 1.911, 0.537, 0.798, 0.013, 0.388, 0.104, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.026, 0.016, 0.007, 0.003, 0.007, 0.001, 0.002, 0.003, 0.001, 0.001, 0.001, 0.002, 0.006, 0.002, 0.001, 0.001, 0.004, 0.002, 0.001, 0.01, 0.002, 0.002, 0.002, 0.003, 0.001, 0.004, 0.001, 0.005, 0.009, 0.009, 0.003, 0.002, 0.021, 0.037, 0.001, 0.006, 0.0001, 0.001, 0.001, 0.002, 0.002, 0.013, 0.005, 0.003, 0.004, 0.024, 0.002, 0.002, 0.006, 0.026, 0.007, 0.298, 0.002, 0.005, 0.003, 0.003, 0.01, 0.004, 0.011, 0.015, 0.005, 0.005, 0.003, 0.004, 0.0001, 0.0001, 0.019, 0.408, 0.007, 0.009, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.02, 0.011, 0.021, 0.008, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.012, 0.021, 0.009, 0.003, 0.009, 0.003, 0.001, 0.001, 0.002, 0.004, 0.003, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ts":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.117, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.445, 0.004, 0.183, 0.0001, 0.0001, 0.006, 0.001, 0.136, 0.107, 0.107, 0.0001, 0.0001, 0.868, 0.158, 0.838, 0.021, 0.152, 0.161, 0.081, 0.037, 0.038, 0.052, 0.045, 0.043, 0.056, 0.092, 0.041, 0.025, 0.03, 0.001, 0.03, 0.006, 0.0001, 0.18, 0.088, 0.068, 0.084, 0.075, 0.029, 0.061, 0.137, 0.055, 0.032, 0.132, 0.116, 0.387, 0.232, 0.02, 0.062, 0.002, 0.075, 0.171, 0.121, 0.04, 0.219, 0.021, 0.119, 0.045, 0.021, 0.003, 0.0001, 0.003, 0.0001, 0.002, 0.005, 13.463, 1.384, 0.275, 1.092, 4.958, 0.572, 1.347, 3.614, 7.958, 0.047, 4.285, 4.291, 2.768, 5.921, 3.615, 0.489, 0.025, 2.056, 2.585, 2.874, 4.929, 1.994, 3.082, 0.68, 2.172, 0.64, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.055, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.0001, 0.005, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.031, 0.0001, 0.001, 0.008, 0.008, 0.0001, 0.0001, 0.05, 0.004, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.001, 0.003, 0.0001, 0.002, 0.001, 0.005, 0.002, 0.011, 0.002, 0.0001, 0.0001, 0.001, 0.002, 0.002, 0.001, 0.002, 0.002, 0.002, 0.0001, 0.002, 0.0001, 0.0001, 0.051, 0.023, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.018, 0.006, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.054, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tt":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.086, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.219, 0.001, 0.085, 0.0001, 0.0001, 0.04, 0.0001, 0.002, 0.22, 0.221, 0.0001, 0.008, 0.529, 0.164, 0.713, 0.007, 0.223, 0.276, 0.185, 0.093, 0.09, 0.084, 0.067, 0.069, 0.089, 0.159, 0.097, 0.008, 0.002, 0.001, 0.002, 0.003, 0.0001, 0.01, 0.009, 0.017, 0.009, 0.006, 0.003, 0.002, 0.003, 0.017, 0.001, 0.009, 0.003, 0.013, 0.003, 0.003, 0.004, 0.005, 0.005, 0.013, 0.017, 0.009, 0.006, 0.002, 0.01, 0.003, 0.001, 0.002, 0.0001, 0.002, 0.0001, 0.002, 0.0001, 0.245, 0.051, 0.015, 0.059, 0.152, 0.017, 0.027, 0.019, 0.108, 0.002, 0.051, 0.14, 0.059, 0.158, 0.057, 0.025, 0.035, 0.149, 0.073, 0.108, 0.056, 0.01, 0.015, 0.014, 0.048, 0.025, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.852, 1.726, 1.824, 1.398, 0.151, 0.194, 0.076, 0.605, 0.638, 0.004, 0.1, 2.623, 0.236, 0.061, 0.057, 0.479, 0.123, 0.129, 0.053, 0.062, 0.279, 0.075, 0.02, 0.174, 0.096, 1.916, 0.222, 0.025, 0.1, 0.049, 0.069, 0.128, 0.159, 0.146, 0.119, 0.43, 0.164, 0.055, 0.003, 0.065, 0.036, 0.325, 0.0001, 0.038, 0.001, 0.013, 0.042, 0.429, 4.958, 1.044, 0.394, 1.429, 0.959, 3.011, 0.048, 0.384, 1.557, 0.433, 1.901, 3.01, 1.056, 3.108, 1.043, 0.407, 0.0001, 0.0001, 0.106, 0.225, 0.139, 0.034, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.004, 0.0001, 0.003, 0.001, 26.093, 12.748, 1.127, 2.265, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.275, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tum": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.34, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.852, 0.004, 0.573, 0.003, 0.004, 0.004, 0.017, 0.083, 0.308, 0.306, 0.001, 0.0001, 1.303, 0.412, 1.2, 0.024, 0.557, 0.476, 0.366, 0.176, 0.172, 0.213, 0.206, 0.176, 0.165, 0.191, 0.118, 0.025, 0.012, 0.007, 0.012, 0.0001, 0.001, 0.268, 0.377, 0.217, 0.158, 0.11, 0.095, 0.125, 0.123, 0.134, 0.277, 0.29, 0.111, 0.727, 0.21, 0.076, 0.143, 0.01, 0.116, 0.269, 0.294, 0.069, 0.067, 0.069, 0.003, 0.068, 0.042, 0.008, 0.0001, 0.008, 0.0001, 0.0001, 0.0001, 10.116, 1.728, 1.817, 1.937, 5.125, 1.225, 1.488, 3.251, 6.548, 0.159, 2.454, 2.854, 2.514, 5.282, 4.292, 2.074, 0.028, 2.715, 2.7, 3.62, 4.127, 0.602, 1.862, 0.051, 1.299, 0.758, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.102, 0.017, 0.014, 0.014, 0.01, 0.006, 0.008, 0.005, 0.003, 0.001, 0.007, 0.006, 0.02, 0.058, 0.017, 0.003, 0.008, 0.005, 0.001, 0.016, 0.005, 0.005, 0.003, 0.004, 0.009, 0.043, 0.004, 0.001, 0.008, 0.005, 0.006, 0.002, 0.103, 0.006, 0.008, 0.007, 0.001, 0.005, 0.009, 0.025, 0.006, 0.01, 0.003, 0.011, 0.006, 0.004, 0.0001, 0.003, 0.016, 0.015, 0.003, 0.014, 0.008, 0.112, 0.003, 0.014, 0.012, 0.008, 0.012, 0.012, 0.008, 0.009, 0.01, 0.003, 0.0001, 0.0001, 0.101, 0.045, 0.006, 0.195, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.003, 0.063, 0.038, 0.001, 0.001, 0.001, 0.006, 0.003, 0.007, 0.053, 0.034, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.016, 0.022, 0.093, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.012, 0.008, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tw":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.984, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.303, 0.001, 0.389, 0.0001, 0.001, 0.0001, 0.004, 0.077, 0.488, 0.486, 0.0001, 0.0001, 0.756, 0.118, 1.791, 0.025, 0.73, 0.614, 0.579, 0.248, 0.221, 0.18, 0.206, 0.176, 0.192, 0.286, 0.065, 0.035, 0.004, 0.0001, 0.004, 0.01, 0.0001, 0.602, 0.283, 0.296, 0.116, 0.311, 0.173, 0.2, 0.1, 0.303, 0.048, 0.367, 0.187, 0.399, 0.306, 0.149, 0.189, 0.019, 0.18, 0.508, 0.305, 0.203, 0.099, 0.096, 0.049, 0.077, 0.01, 0.003, 0.0001, 0.019, 0.0001, 0.0001, 0.0001, 8.315, 0.995, 0.605, 1.602, 5.365, 0.628, 0.659, 0.955, 4.58, 0.091, 2.249, 1.426, 1.892, 5.378, 5.608, 0.884, 0.03, 3.156, 2.583, 1.888, 2.004, 0.328, 1.708, 0.075, 2.441, 0.168, 0.0001, 0.0001, 0.0001, 0.01, 0.0001, 0.083, 0.035, 0.035, 0.017, 0.032, 0.015, 0.093, 0.059, 0.023, 0.016, 0.025, 0.022, 0.019, 0.022, 0.029, 0.012, 0.046, 0.013, 0.009, 0.017, 0.855, 0.004, 0.017, 0.017, 0.006, 0.004, 0.012, 1.236, 0.017, 0.012, 0.01, 0.004, 0.081, 0.046, 0.012, 0.012, 0.086, 0.028, 0.017, 0.054, 0.03, 0.075, 0.019, 0.012, 0.016, 0.036, 0.009, 0.019, 0.074, 0.048, 0.057, 0.049, 0.013, 2.039, 0.016, 0.03, 0.109, 0.023, 0.064, 0.039, 0.051, 0.048, 0.068, 0.015, 0.0001, 0.0001, 0.075, 0.196, 0.058, 0.036, 0.106, 0.0001, 0.001, 1.812, 0.004, 0.0001, 0.001, 0.0001, 2.053, 0.006, 0.306, 0.086, 0.0001, 0.0001, 0.0001, 0.012, 0.003, 0.267, 0.158, 0.09, 0.007, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.209, 0.016, 0.044, 0.0001, 0.016, 0.052, 0.016, 0.023, 0.012, 0.003, 0.001, 0.0001, 0.003, 0.0001, 0.0001, 0.019, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ty":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 5.596, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.482, 0.002, 0.148, 0.0001, 0.0001, 0.0001, 0.001, 0.103, 0.185, 0.187, 0.0001, 0.0001, 0.459, 0.229, 1.457, 0.013, 0.217, 0.354, 0.181, 0.099, 0.109, 0.09, 0.093, 0.094, 0.097, 0.295, 0.032, 0.014, 0.002, 0.001, 0.023, 0.0001, 0.0001, 0.336, 0.259, 0.191, 0.056, 0.549, 0.206, 0.061, 0.142, 0.109, 0.062, 0.031, 0.131, 0.411, 0.099, 0.644, 0.477, 0.008, 0.194, 0.401, 0.951, 0.146, 0.18, 0.019, 0.004, 0.015, 0.007, 0.008, 0.0001, 0.01, 0.0001, 0.003, 0.0001, 9.536, 0.253, 0.42, 0.705, 6.452, 0.803, 0.335, 1.722, 7.016, 0.092, 0.277, 1.311, 1.613, 3.693, 4.012, 0.994, 0.04, 4.455, 1.038, 5.804, 2.543, 0.371, 0.019, 0.027, 0.146, 0.201, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.052, 0.908, 0.007, 0.002, 0.007, 0.001, 0.0001, 0.003, 0.006, 0.001, 0.003, 0.002, 0.002, 1.282, 0.0001, 0.001, 0.007, 0.0001, 0.043, 0.549, 0.01, 0.0001, 0.0001, 0.003, 0.114, 1.916, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.19, 0.144, 0.074, 0.002, 0.002, 0.003, 0.003, 0.022, 0.06, 0.039, 0.051, 0.598, 0.116, 0.035, 0.003, 0.018, 0.003, 0.029, 0.506, 0.059, 0.005, 0.003, 0.0001, 0.001, 0.002, 0.008, 0.013, 0.037, 0.005, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.033, 1.417, 1.711, 1.627, 0.0001, 0.0001, 0.0001, 0.008, 0.01, 0.005, 0.002, 0.001, 0.0001, 0.0001, 0.009, 0.006, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.012, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.01, 2.037, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "tyv": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.67, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.942, 0.005, 0.141, 0.0001, 0.0001, 0.004, 0.001, 0.003, 0.097, 0.1, 0.0001, 0.001, 0.649, 0.583, 0.64, 0.009, 0.087, 0.151, 0.08, 0.042, 0.04, 0.04, 0.033, 0.032, 0.035, 0.099, 0.046, 0.011, 0.008, 0.002, 0.008, 0.008, 0.0001, 0.007, 0.002, 0.003, 0.002, 0.002, 0.002, 0.001, 0.002, 0.022, 0.0001, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.001, 0.003, 0.003, 0.001, 0.006, 0.002, 0.011, 0.001, 0.0001, 0.005, 0.0001, 0.005, 0.0001, 0.005, 0.0001, 0.081, 0.005, 0.006, 0.008, 0.025, 0.002, 0.005, 0.006, 0.02, 0.002, 0.007, 0.012, 0.016, 0.015, 0.021, 0.013, 0.003, 0.017, 0.01, 0.014, 0.01, 0.002, 0.004, 0.007, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.263, 0.883, 1.755, 1.893, 0.056, 0.377, 0.045, 1.004, 0.604, 0.005, 0.051, 2.643, 0.086, 0.75, 0.036, 0.173, 0.125, 0.135, 0.03, 0.065, 0.108, 0.011, 0.018, 0.005, 0.038, 0.005, 0.129, 0.036, 0.079, 0.041, 0.11, 0.022, 0.066, 0.107, 0.147, 0.782, 0.015, 0.082, 0.008, 0.088, 0.054, 0.476, 0.001, 0.089, 0.001, 0.039, 0.018, 0.892, 5.51, 0.98, 0.415, 1.888, 1.904, 2.436, 0.478, 0.679, 2.249, 0.486, 1.593, 2.459, 0.684, 3.034, 1.582, 0.744, 0.0001, 0.0001, 0.143, 0.011, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.002, 0.004, 0.003, 0.01, 0.001, 0.011, 0.002, 28.453, 13.514, 1.663, 0.515, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.001, 0.094, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "udm": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.306, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.09, 0.004, 0.114, 0.0001, 0.0001, 0.008, 0.0001, 0.002, 0.237, 0.238, 0.002, 0.001, 0.557, 0.317, 0.775, 0.018, 0.183, 0.302, 0.16, 0.086, 0.075, 0.092, 0.071, 0.074, 0.085, 0.189, 0.048, 0.012, 0.017, 0.014, 0.016, 0.001, 0.0001, 0.018, 0.008, 0.012, 0.004, 0.003, 0.003, 0.003, 0.003, 0.016, 0.004, 0.004, 0.006, 0.014, 0.003, 0.019, 0.021, 0.0001, 0.006, 0.011, 0.006, 0.003, 0.006, 0.001, 0.009, 0.001, 0.001, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.242, 0.027, 0.103, 0.053, 0.195, 0.007, 0.026, 0.039, 0.148, 0.005, 0.015, 0.074, 0.03, 0.111, 0.083, 0.028, 0.002, 0.108, 0.083, 0.059, 0.078, 0.015, 0.004, 0.004, 0.02, 0.008, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 2.622, 2.823, 2.068, 1.727, 0.105, 0.092, 0.121, 0.28, 0.404, 0.054, 0.451, 2.424, 1.272, 0.932, 0.131, 0.626, 0.166, 0.634, 0.123, 0.164, 0.252, 0.027, 0.006, 0.023, 0.083, 0.009, 0.22, 0.069, 0.124, 0.088, 0.082, 0.223, 0.15, 0.209, 0.107, 0.132, 0.033, 0.405, 0.01, 0.179, 0.05, 0.004, 0.001, 0.088, 0.001, 0.03, 0.018, 0.022, 2.886, 0.44, 0.8, 0.564, 1.075, 2.236, 0.315, 1.165, 1.904, 0.34, 1.795, 2.214, 1.337, 2.854, 2.759, 0.664, 0.0001, 0.0001, 0.24, 0.028, 0.005, 0.005, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.023, 0.0001, 0.001, 0.0001, 25.262, 16.34, 0.005, 0.714, 0.0001, 0.005, 0.001, 0.002, 0.005, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.006, 0.277, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ug":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.4, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 6.843, 0.005, 0.045, 0.0001, 0.0001, 0.006, 0.0001, 0.05, 0.059, 0.061, 0.001, 0.001, 0.064, 0.182, 0.431, 0.006, 0.116, 0.137, 0.086, 0.058, 0.051, 0.055, 0.044, 0.042, 0.045, 0.072, 0.055, 0.007, 0.018, 0.009, 0.017, 0.0001, 0.0001, 0.014, 0.005, 0.004, 0.003, 0.002, 0.001, 0.002, 0.002, 0.011, 0.008, 0.009, 0.003, 0.013, 0.002, 0.002, 0.005, 0.001, 0.002, 0.015, 0.014, 0.019, 0.001, 0.002, 0.002, 0.003, 0.0001, 0.003, 0.001, 0.003, 0.0001, 0.008, 0.0001, 0.198, 0.04, 0.041, 0.081, 0.144, 0.022, 0.07, 0.096, 0.317, 0.009, 0.06, 0.138, 0.069, 0.164, 0.09, 0.038, 0.044, 0.138, 0.091, 0.118, 0.088, 0.011, 0.018, 0.015, 0.072, 0.022, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.146, 0.075, 1.421, 1.142, 2.553, 1.322, 3.07, 1.622, 1.224, 6.252, 1.181, 0.454, 0.501, 0.027, 0.124, 0.02, 0.545, 0.041, 0.008, 0.046, 0.025, 2.705, 0.02, 0.099, 0.121, 0.09, 0.015, 0.082, 0.041, 0.012, 0.015, 0.06, 0.068, 0.006, 0.005, 0.06, 0.019, 0.028, 1.456, 3.601, 1.011, 0.28, 1.856, 0.056, 0.228, 0.623, 0.346, 2.099, 0.163, 2.119, 0.524, 1.075, 0.873, 0.045, 0.014, 0.035, 0.226, 0.052, 1.208, 0.825, 0.077, 0.089, 1.1, 0.024, 0.0001, 0.0001, 0.118, 0.051, 0.009, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.765, 0.262, 0.112, 0.09, 0.0001, 0.0001, 0.0001, 0.001, 14.938, 17.649, 1.694, 5.905, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.067, 0.002, 0.002, 0.006, 0.003, 0.003, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.007, 1.731, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ur":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.979, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.161, 0.002, 0.04, 0.0001, 0.0001, 0.001, 0.0001, 0.006, 0.157, 0.157, 0.0001, 0.001, 0.081, 0.085, 0.055, 0.007, 0.121, 0.179, 0.119, 0.082, 0.072, 0.073, 0.068, 0.065, 0.07, 0.096, 0.098, 0.002, 0.004, 0.003, 0.004, 0.0001, 0.0001, 0.02, 0.016, 0.035, 0.016, 0.006, 0.007, 0.013, 0.009, 0.011, 0.009, 0.012, 0.015, 0.025, 0.011, 0.007, 0.016, 0.003, 0.012, 0.029, 0.016, 0.005, 0.006, 0.007, 0.001, 0.005, 0.003, 0.004, 0.0001, 0.004, 0.0001, 0.004, 0.0001, 0.265, 0.03, 0.059, 0.059, 0.181, 0.032, 0.039, 0.075, 0.194, 0.006, 0.027, 0.102, 0.048, 0.197, 0.175, 0.037, 0.004, 0.142, 0.109, 0.147, 0.083, 0.021, 0.026, 0.005, 0.049, 0.011, 0.0001, 0.014, 0.0001, 0.0001, 0.0001, 0.055, 2.387, 0.534, 0.013, 1.581, 2.193, 2.297, 0.009, 2.712, 0.004, 0.024, 0.012, 4.725, 0.004, 0.025, 0.025, 0.036, 0.091, 1.735, 0.008, 0.507, 0.001, 0.001, 0.002, 0.02, 0.012, 0.0001, 0.005, 0.005, 0.004, 0.001, 0.005, 0.009, 0.069, 0.224, 0.005, 0.08, 0.002, 0.401, 5.353, 1.186, 2.395, 1.412, 0.054, 0.699, 0.376, 0.232, 1.576, 0.068, 2.734, 0.325, 1.531, 0.466, 0.218, 0.1, 0.222, 0.073, 1.112, 0.88, 0.012, 0.002, 0.002, 1.074, 0.003, 0.0001, 0.0001, 0.008, 0.011, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.005, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 18.028, 10.547, 4.494, 8.618, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.001, 0.049, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.043, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "uz":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.321, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.468, 0.001, 0.189, 0.0001, 0.0001, 0.012, 0.0001, 0.019, 0.383, 0.392, 0.002, 0.002, 1.018, 0.346, 1.56, 0.012, 0.451, 0.539, 0.363, 0.217, 0.199, 0.207, 0.182, 0.168, 0.187, 0.31, 0.029, 0.042, 0.003, 0.005, 0.003, 0.002, 0.0001, 0.288, 0.177, 0.127, 0.096, 0.051, 0.092, 0.103, 0.072, 0.123, 0.042, 0.115, 0.075, 0.277, 0.092, 0.158, 0.088, 0.099, 0.095, 0.293, 0.135, 0.08, 0.063, 0.021, 0.043, 0.077, 0.019, 0.006, 0.0001, 0.006, 0.001, 0.001, 0.005, 11.395, 1.621, 0.663, 2.97, 1.946, 0.469, 2.488, 2.791, 9.732, 0.446, 2.32, 4.562, 2.354, 4.897, 4.652, 0.487, 1.34, 4.598, 3.575, 3.341, 2.208, 1.083, 0.027, 0.322, 2.128, 0.799, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 0.456, 0.006, 0.008, 0.004, 0.002, 0.001, 0.001, 0.001, 0.003, 0.002, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.001, 0.001, 0.165, 0.164, 0.0001, 0.001, 0.001, 0.064, 0.017, 0.001, 0.002, 0.019, 0.002, 0.019, 0.002, 0.169, 0.003, 0.003, 0.0001, 0.002, 0.0001, 0.0001, 0.002, 0.007, 0.014, 0.0001, 0.005, 0.001, 0.001, 0.0001, 0.0001, 0.04, 0.006, 0.006, 0.01, 0.015, 0.009, 0.006, 0.002, 0.016, 0.002, 0.006, 0.916, 0.127, 0.009, 0.012, 0.002, 0.0001, 0.0001, 0.192, 0.06, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 1.018, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.124, 0.036, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.449, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "ve":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.731, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.522, 0.012, 0.078, 0.0001, 0.0001, 0.001, 0.0001, 0.009, 0.159, 0.16, 0.0001, 0.001, 0.539, 0.225, 1.016, 0.019, 0.145, 0.2, 0.126, 0.043, 0.046, 0.05, 0.05, 0.043, 0.035, 0.051, 0.043, 0.011, 0.01, 0.003, 0.01, 0.007, 0.001, 0.246, 0.066, 0.041, 0.13, 0.054, 0.04, 0.046, 0.163, 0.081, 0.023, 0.129, 0.141, 0.422, 0.243, 0.021, 0.074, 0.002, 0.073, 0.154, 0.414, 0.061, 0.436, 0.032, 0.007, 0.055, 0.059, 0.001, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 13.088, 1.237, 0.128, 2.934, 4.075, 0.966, 1.256, 7.989, 6.478, 0.01, 1.611, 2.964, 2.428, 5.855, 4.328, 0.793, 0.003, 1.372, 2.898, 2.532, 4.835, 2.93, 2.215, 0.021, 0.876, 1.698, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.04, 0.003, 0.001, 0.0001, 0.002, 0.021, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.005, 0.137, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.006, 0.001, 0.001, 0.006, 0.005, 0.0001, 0.0001, 0.002, 0.001, 0.008, 0.001, 0.0001, 0.0001, 0.007, 0.001, 0.0001, 0.0001, 0.0001, 0.004, 0.002, 0.0001, 0.0001, 0.001, 0.008, 0.049, 0.003, 0.004, 0.0001, 0.0001, 0.001, 0.0001, 0.157, 0.074, 0.001, 0.002, 0.0001, 0.026, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.017, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.014, 0.002, 0.006, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.006, 0.231, 0.039, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "vec": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.253, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.683, 0.003, 0.435, 0.0001, 0.0001, 0.011, 0.001, 0.612, 0.188, 0.187, 0.0001, 0.002, 0.962, 0.099, 0.799, 0.015, 0.255, 0.324, 0.176, 0.103, 0.099, 0.11, 0.096, 0.095, 0.113, 0.179, 0.07, 0.031, 0.016, 0.004, 0.016, 0.001, 0.0001, 0.22, 0.135, 0.257, 0.11, 0.212, 0.092, 0.123, 0.029, 0.211, 0.028, 0.03, 0.164, 0.197, 0.115, 0.055, 0.192, 0.012, 0.112, 0.28, 0.113, 0.043, 0.127, 0.024, 0.034, 0.008, 0.022, 0.006, 0.0001, 0.006, 0.0001, 0.006, 0.0001, 9.014, 0.584, 2.527, 3.084, 9.08, 0.695, 1.267, 0.67, 6.478, 0.14, 0.121, 3.361, 1.486, 5.29, 5.96, 1.776, 0.156, 4.436, 3.403, 4.054, 1.601, 1.042, 0.044, 0.834, 0.071, 0.222, 0.0001, 0.006, 0.0001, 0.0001, 0.0001, 0.081, 0.084, 1.282, 0.004, 0.002, 0.002, 0.001, 0.002, 0.002, 0.001, 0.001, 0.002, 0.003, 0.004, 0.001, 0.001, 0.002, 0.013, 0.001, 0.01, 0.002, 0.001, 0.001, 0.008, 0.004, 0.058, 0.055, 0.001, 0.003, 0.003, 0.0001, 0.001, 0.74, 0.012, 0.002, 0.002, 0.005, 0.001, 0.002, 0.041, 0.204, 0.163, 0.002, 0.004, 0.188, 0.007, 0.001, 0.002, 0.019, 0.005, 0.113, 0.084, 0.004, 0.003, 0.003, 0.001, 0.003, 0.085, 0.013, 0.006, 0.006, 0.01, 0.027, 0.003, 0.0001, 0.0001, 0.074, 1.6, 0.013, 1.389, 0.061, 0.0001, 0.005, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.014, 0.007, 0.012, 0.005, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.013, 0.075, 0.002, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "vep": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.78, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.379, 0.003, 0.471, 0.0001, 0.001, 0.103, 0.0001, 0.568, 0.495, 0.495, 0.0001, 0.017, 1.052, 0.379, 1.489, 0.012, 0.568, 0.707, 0.478, 0.223, 0.214, 0.232, 0.198, 0.192, 0.203, 0.325, 0.211, 0.045, 0.002, 0.001, 0.002, 0.001, 0.0001, 0.203, 0.112, 0.053, 0.077, 0.109, 0.05, 0.072, 0.067, 0.085, 0.066, 0.318, 0.157, 0.187, 0.127, 0.087, 0.197, 0.001, 0.106, 0.305, 0.17, 0.046, 0.359, 0.008, 0.005, 0.004, 0.023, 0.011, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 7.907, 0.771, 0.299, 4.189, 5.699, 0.182, 1.123, 1.305, 7.031, 1.198, 2.907, 3.562, 2.965, 5.97, 3.852, 1.33, 0.003, 2.724, 3.29, 3.069, 2.779, 1.746, 0.01, 0.004, 0.024, 0.95, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.243, 0.042, 0.031, 0.026, 0.016, 0.009, 0.007, 0.007, 0.018, 0.003, 0.008, 0.014, 0.04, 0.228, 0.004, 0.014, 0.011, 0.008, 0.007, 0.006, 0.198, 0.004, 0.004, 0.004, 0.004, 0.01, 0.011, 0.006, 0.059, 0.006, 0.007, 0.007, 0.049, 0.512, 0.005, 0.004, 1.459, 0.005, 0.005, 0.012, 0.007, 0.009, 0.006, 0.076, 0.003, 0.005, 0.006, 0.008, 0.087, 0.02, 0.049, 0.021, 0.019, 0.048, 0.155, 0.011, 0.041, 0.019, 0.037, 0.102, 0.539, 0.049, 0.808, 0.016, 0.0001, 0.0001, 0.208, 2.197, 0.255, 1.283, 0.0001, 0.0001, 0.0001, 0.018, 0.007, 0.013, 0.002, 0.001, 0.025, 0.012, 0.469, 0.173, 0.003, 0.003, 0.001, 0.006, 0.006, 0.011, 0.026, 0.019, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.026, 0.012, 0.203, 0.002, 0.001, 0.003, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "vls": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.228, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.739, 0.003, 0.38, 0.0001, 0.0001, 0.005, 0.003, 0.744, 0.196, 0.195, 0.001, 0.001, 0.727, 0.325, 0.943, 0.009, 0.279, 0.491, 0.2, 0.128, 0.127, 0.144, 0.128, 0.137, 0.161, 0.217, 0.083, 0.01, 0.008, 0.003, 0.008, 0.002, 0.0001, 0.184, 0.236, 0.118, 0.332, 0.112, 0.115, 0.126, 0.103, 0.261, 0.107, 0.122, 0.141, 0.163, 0.108, 0.113, 0.118, 0.004, 0.127, 0.191, 0.088, 0.03, 0.223, 0.122, 0.006, 0.022, 0.104, 0.001, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 4.751, 0.962, 1.1, 3.988, 12.635, 0.533, 2.162, 1.118, 4.159, 0.386, 1.909, 2.864, 1.62, 7.645, 4.865, 1.022, 0.013, 4.762, 3.511, 4.63, 2.292, 1.812, 1.033, 0.041, 0.74, 0.83, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.13, 0.003, 0.003, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.008, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.008, 0.015, 0.0001, 0.0001, 0.0001, 0.025, 0.093, 0.0001, 0.0001, 0.002, 0.002, 0.001, 0.001, 0.016, 0.003, 0.001, 0.001, 0.002, 0.001, 0.001, 0.004, 0.09, 0.034, 0.493, 0.075, 0.001, 0.002, 0.001, 0.006, 0.006, 0.003, 0.004, 0.004, 0.299, 0.002, 0.003, 0.001, 0.002, 0.001, 0.002, 0.002, 0.005, 0.002, 0.001, 0.002, 0.0001, 0.0001, 0.02, 1.045, 0.002, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.007, 0.004, 0.008, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.002, 0.13, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "vo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.865, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 14.101, 0.0001, 0.089, 0.0001, 0.177, 0.768, 0.0001, 0.013, 0.471, 0.471, 0.0001, 0.0001, 1.958, 0.301, 1.263, 0.002, 1.009, 1.484, 1.145, 0.885, 0.977, 0.988, 0.827, 0.571, 0.867, 0.731, 0.368, 0.112, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.099, 0.202, 0.186, 0.179, 0.034, 0.122, 0.068, 0.069, 0.028, 0.029, 0.035, 0.486, 0.223, 0.193, 0.038, 0.198, 0.004, 0.074, 0.506, 0.089, 0.221, 0.126, 0.048, 0.001, 0.008, 0.039, 0.004, 0.001, 0.005, 0.0001, 0.0001, 0.0001, 5.558, 2.077, 0.284, 2.834, 4.622, 1.332, 0.379, 0.28, 4.679, 0.128, 1.147, 4.377, 2.51, 5.854, 4.077, 1.175, 0.015, 1.237, 3.788, 2.427, 1.276, 0.657, 0.06, 0.029, 0.621, 0.304, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.73, 0.001, 0.0001, 0.005, 0.073, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.0001, 0.033, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.623, 0.0001, 0.0001, 0.045, 0.0001, 0.038, 0.009, 0.001, 0.006, 0.006, 0.01, 2.184, 0.0001, 0.0001, 0.003, 0.022, 0.052, 0.002, 0.001, 0.0001, 0.004, 0.0001, 0.0001, 0.27, 0.001, 0.247, 0.003, 0.014, 0.006, 1.503, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 1.216, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.516, 5.121, 0.006, 0.01, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.73, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "wa":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.065, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 16.234, 0.018, 0.387, 0.0001, 0.0001, 0.001, 0.005, 1.403, 0.391, 0.397, 0.0001, 0.006, 1.323, 0.328, 1.748, 0.02, 0.212, 0.344, 0.172, 0.086, 0.07, 0.086, 0.076, 0.078, 0.098, 0.148, 0.393, 0.059, 0.003, 0.003, 0.006, 0.012, 0.0001, 0.126, 0.117, 0.176, 0.177, 0.152, 0.211, 0.071, 0.055, 0.154, 0.041, 0.016, 0.325, 0.219, 0.065, 0.097, 0.121, 0.003, 0.069, 0.125, 0.076, 0.016, 0.053, 0.079, 0.005, 0.007, 0.005, 0.103, 0.0001, 0.103, 0.0001, 0.0001, 0.0001, 4.343, 0.71, 2.121, 3.465, 9.326, 0.692, 0.491, 0.929, 5.047, 0.968, 0.844, 3.108, 1.647, 4.913, 4.614, 1.529, 0.028, 3.303, 5.504, 4.286, 1.947, 1.135, 0.682, 0.179, 1.059, 0.366, 0.0001, 0.075, 0.0001, 0.0001, 0.0001, 0.076, 0.002, 0.002, 0.001, 0.001, 0.022, 0.001, 0.008, 0.005, 0.003, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.002, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.065, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.001, 0.371, 0.002, 0.017, 0.001, 0.001, 0.706, 0.003, 0.089, 0.451, 0.662, 0.205, 0.03, 0.001, 0.001, 0.639, 0.002, 0.006, 0.002, 0.002, 0.001, 0.243, 0.004, 0.001, 0.001, 0.001, 0.001, 0.002, 0.257, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.478, 3.239, 0.002, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.006, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.08, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "war": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.118, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 12.933, 0.0001, 1.377, 0.0001, 0.0001, 0.0001, 0.003, 0.004, 0.008, 0.008, 0.0001, 0.0001, 0.432, 0.073, 1.214, 0.001, 0.079, 0.266, 0.062, 0.046, 0.041, 0.046, 0.05, 0.055, 0.111, 0.217, 0.037, 0.004, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 1.082, 0.154, 0.38, 0.175, 0.141, 0.098, 0.127, 0.173, 0.102, 0.057, 0.046, 0.208, 0.316, 0.091, 0.096, 0.293, 0.004, 0.105, 0.232, 0.146, 0.033, 0.038, 0.367, 0.012, 0.008, 0.019, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.129, 0.835, 2.123, 1.488, 5.092, 0.584, 3.71, 3.47, 8.491, 0.033, 1.376, 3.841, 1.504, 9.228, 3.14, 2.313, 0.025, 2.807, 5.239, 2.428, 2.957, 0.216, 0.413, 0.116, 1.506, 0.106, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.006, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.002, 0.004, 0.019, 0.0001, 0.001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.004, 0.003, 0.0001, 0.004, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.006, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.06, 0.002, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "wo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.906, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 18.371, 0.007, 0.083, 0.0001, 0.0001, 0.002, 0.0001, 0.048, 0.243, 0.244, 0.0001, 0.001, 1.526, 0.299, 0.6, 0.011, 0.077, 0.162, 0.075, 0.048, 0.048, 0.043, 0.038, 0.041, 0.05, 0.065, 0.149, 0.021, 0.001, 0.005, 0.001, 0.009, 0.0001, 0.248, 0.196, 0.082, 0.079, 0.037, 0.083, 0.06, 0.026, 0.08, 0.079, 0.082, 0.102, 0.179, 0.109, 0.049, 0.052, 0.005, 0.054, 0.208, 0.113, 0.015, 0.012, 0.059, 0.037, 0.106, 0.002, 0.002, 0.0001, 0.002, 0.0001, 0.001, 0.0001, 10.502, 2.142, 1.408, 2.296, 5.004, 0.815, 2.647, 0.171, 6.017, 1.265, 2.73, 3.516, 3.296, 5.064, 5.377, 0.616, 0.08, 2.151, 1.518, 2.39, 4.356, 0.021, 1.494, 1.066, 2.37, 0.019, 0.002, 0.0001, 0.004, 0.0001, 0.0001, 0.102, 0.006, 0.003, 0.003, 0.01, 0.005, 0.004, 0.003, 0.005, 0.001, 0.005, 0.02, 0.001, 0.001, 0.008, 0.002, 0.005, 0.039, 0.003, 0.026, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.041, 0.001, 0.0001, 0.016, 0.015, 0.0001, 0.0001, 0.641, 0.001, 0.002, 0.004, 0.002, 0.001, 0.001, 0.012, 0.011, 0.402, 0.004, 0.775, 0.001, 0.002, 0.001, 0.002, 0.004, 0.912, 0.013, 0.056, 0.002, 0.002, 0.001, 0.002, 0.003, 0.003, 0.002, 0.013, 0.001, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.028, 2.826, 0.002, 0.019, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.002, 0.0001, 0.004, 0.002, 0.016, 0.018, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.007, 0.033, 0.053, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.096, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "wuu": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.208, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.344, 0.001, 0.064, 0.0001, 0.0001, 0.012, 0.001, 0.005, 0.037, 0.032, 0.001, 0.002, 0.029, 0.05, 0.042, 0.019, 0.267, 0.364, 0.21, 0.108, 0.106, 0.12, 0.107, 0.1, 0.123, 0.181, 0.013, 0.001, 0.013, 0.002, 0.013, 0.001, 0.0001, 0.027, 0.021, 0.029, 0.015, 0.013, 0.01, 0.014, 0.013, 0.018, 0.007, 0.011, 0.017, 0.022, 0.016, 0.01, 0.023, 0.002, 0.017, 0.03, 0.019, 0.009, 0.006, 0.008, 0.002, 0.003, 0.002, 0.03, 0.0001, 0.03, 0.0001, 0.003, 0.0001, 0.184, 0.024, 0.041, 0.051, 0.161, 0.019, 0.037, 0.056, 0.143, 0.005, 0.024, 0.082, 0.047, 0.138, 0.118, 0.028, 0.006, 0.111, 0.081, 0.088, 0.07, 0.016, 0.015, 0.01, 0.024, 0.008, 0.001, 0.002, 0.001, 0.001, 0.0001, 2.843, 1.238, 1.324, 0.655, 0.418, 1.022, 0.586, 0.937, 1.267, 1.305, 0.731, 1.421, 2.335, 0.988, 0.859, 1.016, 1.143, 0.568, 0.436, 0.439, 0.836, 0.673, 0.873, 1.003, 0.932, 0.655, 0.691, 1.033, 1.591, 0.82, 0.469, 0.875, 0.536, 0.577, 0.431, 0.453, 0.911, 0.859, 0.578, 0.722, 0.777, 0.496, 1.371, 0.496, 0.553, 1.219, 0.891, 1.125, 1.185, 0.888, 0.563, 0.66, 0.876, 0.472, 0.61, 0.726, 3.021, 1.231, 1.855, 1.189, 2.708, 1.052, 0.869, 1.001, 0.0001, 0.0001, 0.059, 0.019, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.005, 0.002, 0.002, 0.002, 0.0001, 0.011, 0.004, 0.02, 0.007, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.003, 0.011, 0.009, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.0001, 0.068, 0.005, 0.208, 1.565, 4.388, 9.361, 5.679, 3.099, 2.882, 2.131, 0.002, 0.004, 0.008, 0.002, 0.0001, 1.953, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "xal": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 2.016, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 7.801, 0.002, 0.076, 0.0001, 0.0001, 0.005, 0.0001, 0.002, 0.134, 0.134, 0.001, 0.003, 0.529, 0.574, 0.918, 0.006, 0.214, 0.423, 0.268, 0.17, 0.177, 0.128, 0.129, 0.128, 0.121, 0.185, 0.028, 0.007, 0.006, 0.001, 0.006, 0.006, 0.0001, 0.005, 0.004, 0.004, 0.001, 0.002, 0.002, 0.002, 0.002, 0.006, 0.001, 0.002, 0.002, 0.003, 0.001, 0.001, 0.002, 0.0001, 0.002, 0.005, 0.003, 0.001, 0.002, 0.003, 0.004, 0.001, 0.001, 0.005, 0.0001, 0.006, 0.0001, 0.005, 0.0001, 0.064, 0.016, 0.035, 0.026, 0.079, 0.017, 0.024, 0.144, 0.059, 0.003, 0.012, 0.04, 0.028, 0.059, 0.05, 0.015, 0.002, 0.048, 0.045, 0.048, 0.035, 0.008, 0.009, 0.006, 0.012, 0.006, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 2.512, 1.678, 1.585, 1.178, 0.036, 0.859, 0.336, 0.487, 0.211, 0.008, 0.012, 0.272, 0.319, 0.492, 0.054, 0.135, 0.09, 0.152, 0.041, 0.073, 0.19, 0.017, 0.022, 0.69, 0.054, 1.446, 0.115, 0.043, 0.168, 0.153, 0.159, 0.053, 0.055, 0.105, 0.151, 0.242, 0.028, 0.118, 0.031, 0.02, 0.093, 0.554, 0.004, 0.02, 0.002, 0.072, 0.031, 0.849, 3.75, 1.252, 0.825, 1.816, 2.139, 1.256, 0.115, 0.387, 2.666, 0.446, 0.987, 3.364, 1.079, 4.101, 2.147, 0.166, 0.0001, 0.0001, 0.041, 0.006, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.038, 0.0001, 0.0001, 0.004, 0.0001, 0.007, 0.004, 27.749, 10.017, 2.264, 1.98, 0.0001, 0.003, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.024, 0.035, 0.127, 0.0001, 0.0001, 0.004, 0.002, 0.001, 0.001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "xh":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.827, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 11.133, 0.013, 0.259, 0.004, 0.001, 0.009, 0.002, 0.046, 0.125, 0.123, 0.001, 0.005, 0.846, 0.831, 0.912, 0.026, 0.163, 0.218, 0.112, 0.059, 0.052, 0.058, 0.048, 0.051, 0.067, 0.118, 0.048, 0.023, 0.018, 0.006, 0.018, 0.006, 0.0001, 0.218, 0.122, 0.114, 0.05, 0.111, 0.054, 0.063, 0.043, 0.32, 0.057, 0.15, 0.086, 0.186, 0.216, 0.074, 0.101, 0.011, 0.057, 0.136, 0.094, 0.198, 0.022, 0.071, 0.041, 0.042, 0.046, 0.076, 0.001, 0.076, 0.0001, 0.013, 0.0001, 10.703, 2.404, 0.805, 1.231, 8.068, 0.529, 2.029, 3.142, 7.484, 0.244, 4.325, 4.529, 2.518, 6.863, 5.226, 0.943, 0.434, 1.064, 2.867, 2.574, 4.687, 0.307, 2.513, 0.353, 2.341, 2.213, 0.002, 0.028, 0.002, 0.0001, 0.0001, 0.043, 0.003, 0.001, 0.001, 0.002, 0.0001, 0.004, 0.012, 0.003, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.004, 0.01, 0.003, 0.0001, 0.0001, 0.001, 0.003, 0.018, 0.0001, 0.0001, 0.005, 0.005, 0.0001, 0.001, 0.1, 0.005, 0.001, 0.004, 0.001, 0.0001, 0.0001, 0.003, 0.001, 0.007, 0.001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.001, 0.004, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.002, 0.001, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.101, 0.03, 0.014, 0.003, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.004, 0.003, 0.004, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.049, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "xmf": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.601, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 4.701, 0.001, 0.058, 0.0001, 0.0001, 0.01, 0.0001, 0.002, 0.121, 0.121, 0.0001, 0.001, 0.458, 0.166, 0.464, 0.005, 0.164, 0.192, 0.121, 0.06, 0.056, 0.064, 0.055, 0.055, 0.065, 0.102, 0.028, 0.018, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.008, 0.006, 0.008, 0.007, 0.003, 0.004, 0.003, 0.003, 0.027, 0.001, 0.002, 0.006, 0.007, 0.003, 0.003, 0.005, 0.0001, 0.004, 0.007, 0.009, 0.002, 0.008, 0.003, 0.01, 0.001, 0.0001, 0.006, 0.0001, 0.006, 0.0001, 0.001, 0.0001, 0.041, 0.006, 0.016, 0.012, 0.042, 0.004, 0.007, 0.012, 0.032, 0.001, 0.006, 0.021, 0.011, 0.029, 0.03, 0.007, 0.001, 0.029, 0.023, 0.023, 0.015, 0.005, 0.003, 0.002, 0.006, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.172, 0.003, 0.002, 30.333, 0.002, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 4.083, 0.506, 0.555, 0.957, 2.283, 0.421, 0.156, 0.803, 3.59, 0.653, 1.19, 1.236, 1.788, 2.02, 0.312, 0.098, 2.097, 1.217, 0.638, 1.469, 0.698, 0.389, 0.172, 0.093, 1.339, 0.152, 0.183, 0.083, 0.259, 0.102, 0.41, 0.184, 0.054, 0.009, 0.013, 0.002, 0.001, 0.003, 0.001, 0.323, 0.062, 0.002, 0.002, 0.002, 0.002, 0.003, 0.004, 0.002, 0.0001, 0.0001, 0.043, 0.004, 0.001, 0.001, 0.007, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.001, 0.0001, 0.011, 0.002, 0.023, 0.008, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.001, 0.004, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.009, 30.332, 0.17, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "yi":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.709, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.298, 0.002, 0.186, 0.0001, 0.001, 0.004, 0.006, 0.121, 0.075, 0.076, 0.0001, 0.0001, 0.466, 0.059, 0.46, 0.006, 0.099, 0.114, 0.062, 0.037, 0.035, 0.037, 0.03, 0.03, 0.038, 0.064, 0.034, 0.015, 0.001, 0.001, 0.001, 0.002, 0.0001, 0.003, 0.003, 0.004, 0.003, 0.002, 0.002, 0.002, 0.002, 0.002, 0.001, 0.001, 0.002, 0.003, 0.002, 0.002, 0.002, 0.0001, 0.002, 0.004, 0.003, 0.001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.003, 0.0001, 0.003, 0.0001, 0.001, 0.0001, 0.02, 0.003, 0.006, 0.007, 0.022, 0.003, 0.004, 0.006, 0.017, 0.0001, 0.003, 0.01, 0.006, 0.015, 0.021, 0.004, 0.006, 0.015, 0.011, 0.018, 0.013, 0.002, 0.003, 0.001, 0.003, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.013, 0.004, 0.003, 0.001, 0.001, 0.001, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 0.001, 0.0001, 0.0001, 0.001, 5.002, 1.068, 1.228, 1.611, 0.814, 3.904, 1.071, 0.178, 2.364, 5.673, 0.275, 0.347, 1.459, 0.389, 1.018, 2.472, 1.73, 1.057, 4.356, 0.098, 1.356, 0.06, 0.547, 0.832, 3.227, 0.975, 0.239, 0.001, 0.001, 0.001, 0.0001, 0.001, 0.02, 0.006, 0.026, 0.005, 0.016, 0.005, 0.002, 0.163, 0.104, 0.003, 0.002, 0.002, 0.041, 0.002, 0.022, 0.034, 0.0001, 0.0001, 0.015, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.003, 0.001, 0.029, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.372, 43.367, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.0001, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "yo":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 3.162, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 13.013, 0.002, 0.102, 0.0001, 0.001, 0.004, 0.001, 0.025, 0.251, 0.249, 0.0001, 0.001, 0.499, 0.259, 1.047, 0.013, 0.386, 0.744, 0.471, 0.336, 0.305, 0.301, 0.323, 0.299, 0.314, 0.417, 0.09, 0.08, 0.008, 0.009, 0.008, 0.006, 0.0001, 0.462, 0.171, 0.128, 0.102, 0.134, 0.101, 0.156, 0.11, 0.251, 0.116, 0.194, 0.108, 0.188, 0.187, 0.263, 0.133, 0.007, 0.102, 0.27, 0.148, 0.037, 0.042, 0.07, 0.006, 0.044, 0.016, 0.007, 0.0001, 0.008, 0.0001, 0.001, 0.001, 4.068, 1.959, 0.507, 1.515, 3.958, 0.547, 1.326, 0.747, 4.508, 1.331, 1.562, 2.445, 1.011, 4.469, 3.265, 1.008, 0.02, 3.063, 1.958, 2.732, 1.408, 0.219, 0.852, 0.039, 0.732, 0.092, 0.0001, 0.013, 0.0001, 0.0001, 0.0001, 0.678, 1.441, 0.002, 0.002, 0.064, 0.002, 0.001, 0.002, 0.025, 0.003, 0.001, 0.001, 0.172, 1.046, 0.0001, 0.0001, 0.001, 0.001, 0.032, 0.052, 0.002, 0.0001, 0.0001, 0.0001, 0.018, 0.066, 0.002, 0.001, 0.007, 0.006, 0.0001, 0.001, 1.085, 1.316, 0.01, 0.17, 0.004, 0.001, 0.001, 0.003, 0.307, 0.812, 0.001, 0.003, 1.559, 1.199, 0.001, 0.002, 0.003, 0.004, 0.287, 0.374, 0.003, 0.002, 0.006, 0.001, 0.038, 1.787, 1.887, 1.09, 0.005, 0.003, 0.003, 0.001, 0.0001, 0.0001, 0.021, 7.862, 0.009, 0.075, 0.0001, 0.008, 0.0001, 0.001, 0.001, 0.001, 1.898, 0.0001, 0.005, 0.002, 0.012, 0.004, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.005, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.004, 2.718, 0.12, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "za":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.779, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 9.452, 0.003, 0.07, 0.0001, 0.001, 0.016, 0.002, 0.054, 0.186, 0.179, 0.001, 0.0001, 0.82, 0.089, 0.74, 0.012, 0.236, 0.344, 0.171, 0.097, 0.104, 0.109, 0.078, 0.094, 0.113, 0.172, 0.091, 0.029, 0.001, 0.001, 0.002, 0.003, 0.0001, 0.117, 0.253, 0.245, 0.236, 0.047, 0.096, 0.232, 0.128, 0.101, 0.031, 0.049, 0.109, 0.142, 0.114, 0.031, 0.114, 0.005, 0.051, 0.316, 0.07, 0.028, 0.136, 0.041, 0.012, 0.157, 0.02, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 4.452, 1.127, 1.557, 2.16, 5.66, 0.54, 3.525, 2.807, 4.357, 1.245, 0.519, 1.215, 1.057, 5.149, 2.46, 0.332, 0.75, 1.554, 1.842, 1.639, 2.859, 0.55, 1.169, 0.375, 1.034, 2.115, 0.002, 0.0001, 0.002, 0.001, 0.0001, 1.059, 0.53, 0.446, 0.215, 0.472, 0.297, 0.257, 0.268, 0.372, 0.375, 0.213, 0.338, 0.751, 0.361, 0.284, 0.332, 0.27, 0.144, 0.117, 0.272, 0.266, 0.278, 0.305, 0.293, 0.26, 0.335, 0.49, 0.247, 0.537, 0.19, 0.142, 0.27, 0.209, 0.19, 0.122, 0.13, 0.301, 0.259, 0.231, 0.235, 0.283, 0.134, 0.154, 0.156, 0.162, 0.375, 0.302, 0.377, 0.293, 0.227, 0.124, 0.201, 0.231, 0.092, 0.229, 0.184, 0.748, 0.296, 0.646, 0.455, 0.756, 0.262, 0.268, 0.277, 0.0001, 0.0001, 0.072, 0.167, 0.018, 0.011, 0.0001, 0.002, 0.0001, 0.002, 0.001, 0.001, 0.001, 0.0001, 0.006, 0.002, 0.014, 0.004, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.012, 0.01, 0.002, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.022, 0.008, 0.114, 0.555, 1.309, 2.559, 1.698, 1.364, 0.916, 0.712, 0.001, 0.001, 0.001, 0.0001, 0.0001, 0.518, 0.001, 0.0001, 0.0001, 0.003, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "zea": [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.532, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 15.056, 0.008, 0.162, 0.0001, 0.0001, 0.007, 0.001, 1.415, 0.162, 0.162, 0.0001, 0.0001, 1.0, 0.532, 1.127, 0.004, 0.395, 0.563, 0.389, 0.394, 0.405, 0.319, 0.329, 0.24, 0.265, 0.382, 0.062, 0.076, 0.002, 0.003, 0.001, 0.008, 0.0001, 0.28, 0.228, 0.122, 0.346, 0.208, 0.185, 0.11, 0.117, 0.317, 0.071, 0.084, 0.154, 0.152, 0.245, 0.188, 0.145, 0.004, 0.099, 0.307, 0.104, 0.026, 0.15, 0.089, 0.002, 0.005, 0.114, 0.003, 0.0001, 0.003, 0.0001, 0.0001, 0.001, 4.665, 0.916, 0.779, 3.731, 13.123, 0.39, 1.695, 1.202, 4.867, 0.455, 1.861, 2.604, 1.621, 7.033, 3.935, 1.063, 0.011, 4.601, 3.105, 3.908, 1.82, 1.832, 0.958, 0.04, 0.135, 0.573, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.482, 0.005, 0.003, 0.002, 0.003, 0.002, 0.001, 0.001, 0.002, 0.005, 0.002, 0.001, 0.001, 0.002, 0.0001, 0.001, 0.001, 0.001, 0.001, 0.005, 0.003, 0.001, 0.0001, 0.001, 0.021, 0.432, 0.001, 0.001, 0.01, 0.009, 0.003, 0.005, 0.009, 0.008, 0.021, 0.001, 0.002, 0.001, 0.003, 0.005, 0.09, 0.052, 0.453, 0.056, 0.009, 0.006, 0.003, 0.006, 0.115, 0.002, 0.14, 0.027, 0.252, 0.001, 0.064, 0.0001, 0.002, 0.002, 0.002, 0.006, 0.004, 0.002, 0.002, 0.001, 0.0001, 0.0001, 0.239, 1.084, 0.009, 0.01, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.005, 0.002, 0.008, 0.003, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.002, 0.007, 0.005, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.003, 0.481, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
    "zu":  [0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 1.261, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 10.94, 0.004, 0.267, 0.001, 0.002, 0.016, 0.003, 0.041, 0.181, 0.181, 0.001, 0.001, 0.907, 0.49, 0.797, 0.099, 0.343, 0.379, 0.279, 0.134, 0.118, 0.116, 0.102, 0.097, 0.11, 0.223, 0.065, 0.035, 0.134, 0.003, 0.135, 0.005, 0.0001, 0.296, 0.147, 0.142, 0.093, 0.11, 0.08, 0.077, 0.065, 0.3, 0.114, 0.141, 0.151, 0.361, 0.387, 0.067, 0.128, 0.012, 0.082, 0.239, 0.152, 0.188, 0.039, 0.182, 0.012, 0.045, 0.07, 0.138, 0.0001, 0.139, 0.0001, 0.001, 0.0001, 10.325, 2.215, 0.829, 1.627, 7.521, 0.687, 2.042, 3.525, 7.719, 0.199, 3.874, 4.421, 2.406, 6.494, 4.881, 0.951, 0.342, 1.361, 3.011, 2.552, 4.691, 0.394, 2.227, 0.134, 1.688, 1.779, 0.0001, 0.002, 0.0001, 0.0001, 0.0001, 0.08, 0.007, 0.001, 0.001, 0.002, 0.0001, 0.014, 0.002, 0.002, 0.001, 0.0001, 0.001, 0.003, 0.005, 0.001, 0.002, 0.002, 0.014, 0.001, 0.01, 0.003, 0.0001, 0.001, 0.001, 0.002, 0.06, 0.0001, 0.001, 0.003, 0.003, 0.001, 0.0001, 0.084, 0.004, 0.0001, 0.001, 0.001, 0.0001, 0.002, 0.004, 0.002, 0.005, 0.003, 0.001, 0.001, 0.002, 0.001, 0.0001, 0.004, 0.003, 0.003, 0.005, 0.002, 0.002, 0.001, 0.006, 0.005, 0.002, 0.003, 0.005, 0.002, 0.003, 0.003, 0.0001, 0.0001, 0.0001, 0.089, 0.024, 0.004, 0.007, 0.0001, 0.0001, 0.0001, 0.002, 0.001, 0.001, 0.0001, 0.0001, 0.002, 0.001, 0.029, 0.011, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.005, 0.002, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.002, 0.002, 0.091, 0.001, 0.0001, 0.001, 0.002, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001, 0.0001],
});

/**
 * Determines whether two ArrayBuffers contain the same values.
 *
 * @param {ArrayBuffer} a
 * @param {ArrayBuffer} b
 * @returns {boolean}
 */
function _buffersEqual(a, b) {
    if (a === b) {
        return true;
    }

    if (a.byteLength !== b.byteLength) {
        return false;
    }

    const ai = new Uint8Array(a),
        bi = new Uint8Array(b);

    let i = a.byteLength;
    while (i--) {
        if (ai[i] !== bi[i]) {
            return false;
        }
    }

    return true;
}

export default Magic;
